<?php

namespace clues\model\payment;

use clues\system\Logger;
use clues\system\DI;
use clues\model\payment\exception\PaymentException;
use clues\model\payment\exception\PaymentStatus;
use clues\model\payment\PaymentConstants;
use clues\system\BasePlatform;
use clues\system\exception\Status;
use clues\system\util\StringOperations;
use clues\system\exception\CommonException;

/**
 * This class is used for all Payment related operations
 * @class Payment
 * @access public
 * @category Model
 * @package model/payment
 * @since 03-Mar-2017
 */
class Payment extends BasePlatform implements PaymentConstants {

    /**
     * Used to hold payment db object
     * @var object
     */
    private $_paymentDb;

    /**
     * Used to hold input params
     * @var object
     */
    protected $inputParams;

    /**
     * variable to hold reference for clues history object
     * @var object
     */
    private $_cluesHistoryObj;

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @param array $inputParams
     * @return void
     */
    public function __construct(array $inputParams = array()) {
        parent::__construct();
        $this->inputParams = $inputParams;
        DI::Map('PaymentDbData', 'clues\\model\\payment\\PaymentDbData');
        $this->_paymentDb = DI::Singleton('PaymentDbData', $inputParams);
        DI::Map('CluesHistory', 'clues\\system\\CluesHistory');
        $this->_cluesHistoryObj = DI::Singleton('CluesHistory');
    }

    /**
     * used to get paymentInfo
     * @method getPaymentInfo
     * @access public
     * @param integer $paymentId
     * @return array array of paymentinfo
     */
    public function getPaymentInfo($paymentId) {
        if (!empty($paymentId)) {
            return $this->_paymentDb->getPaymentInfoByPaymentId($paymentId);
        } else {
            return false;
        }
    }

    /**
     * used to get priority payment gateway
     * @method getPaymentId
     * @access public
     * @param integer $paymentOptionId
     * @param integer $emiId
     * @param string $clientID client id
     * @return integer 
     */
    public function getPaymentId($paymentOptionId, $emiId = 0, $clientID = null) {
        if (empty($clientID)) {
            $clientID = $this->getOAuthClientId(true);
        }
        if (empty($clientID)) {
            $clientID = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        if (!empty($paymentOptionId) && !empty($emiId)) {
            $response = $this->_paymentDb->getPriorityPGWEmi($paymentOptionId, $emiId, $clientID);
            if (empty($response) && $clientID != PaymentConstants::DEFAULT_CLIENT_ID) {
                $response = $this->_paymentDb->getPriorityPGWEmi($paymentOptionId, $emiId, PaymentConstants::DEFAULT_CLIENT_ID);
            }
        } else if (!empty($paymentOptionId)) {
            $response = $this->_paymentDb->getPriorityPGW($paymentOptionId, $clientID);
            if (empty($response) && $clientID != PaymentConstants::DEFAULT_CLIENT_ID) {
                $response = $this->_paymentDb->getPriorityPGW($paymentOptionId, PaymentConstants::DEFAULT_CLIENT_ID);
            }
        } else {
            $response = 0;
        }
        return $response;
    }

    /**
     * This method is used to get payment last success payment details by shopclues order ids
     * @method getLastSuccessPaymentDetailsByOrderIds
     * @access public
     * @param array $orderIds array of order id
     * @param array $extraInfoArray array that contains extra information to be resulted
     * example: {"order_ids":[167805330],"extra":{"pg_expense":"pg_expense","order_meta_data":{"167805330":{"immediate_verify":{"payment_option_id":310}}}}}
     * @return array
     */
    public function getLastSuccessPaymentDetailsByOrderIds(array $orderIds = array(), array $extraInfoArray = array()) {
        if (empty($orderIds)) {
            return array();
        }
        $orderIds = array_unique($orderIds);
        $retVal = array();
        foreach ($orderIds as $orderId) {
            if (!empty($orderId)) {
                $temp = array();
                $temp = $this->_paymentDb->getLastSuccessPaymentDetailsByOrderId($orderId, $extraInfoArray);
                if (!empty($temp)) {
                    $retVal[] = $temp;
                }
                $recheck = false;
                if ((empty($temp) || (isset($temp['payment_status']) && $temp['payment_status'] != 1)) && !empty($extraInfoArray) && isset($extraInfoArray['order_meta_data']) && !empty($extraInfoArray['order_meta_data']) && array_key_exists($orderId, $extraInfoArray['order_meta_data']) && isset($extraInfoArray['order_meta_data'][$orderId]['immediate_verify']['payment_option_id']) && !empty($extraInfoArray['order_meta_data'][$orderId]['immediate_verify']['payment_option_id'])) {
                    $paymentOptionId = $extraInfoArray['order_meta_data'][$orderId]['immediate_verify']['payment_option_id'];
                    $emiId = (isset($extraInfoArray['order_meta_data'][$orderId]['immediate_verify']['emi_id']) && !empty($extraInfoArray['order_meta_data'][$orderId]['immediate_verify']['emi_id'])) ? $extraInfoArray['order_meta_data'][$orderId]['immediate_verify']['emi_id'] : 0;
                    $paymentId = $this->_paymentDb->getPaymentIdFromPaymentOptionId($orderId, $paymentOptionId, $emiId);
                    unset($paymentOptionId, $emiId);
                    if (!empty($paymentId)) {
                        DI::Map('pgwVerify', 'clues\\model\\pgwVerify\\PGWVerify');
                        $pgwVerifyObject = DI::Singleton('pgwVerify');
                        $recheck = $pgwVerifyObject->verifyEachPendingOrder($orderId, $paymentId);
                        if ($recheck) {
                            $temp = array();
                            $temp = $this->_paymentDb->getLastSuccessPaymentDetailsByOrderId($orderId, $extraInfoArray);
                            if (!empty($temp)) {
                                $retVal[] = $temp;
                            }
                        }
                    }
                }
            }
        }
        return $retVal;
    }

    /**
     * This method is used to get pgw merchant params by payment id and client id
     * @method getPaymentInfoByPaymentIdAndCID
     * @access public
     * @param integer $paymentId payment id
     * @param string $clientID client id
     * @return array
     */
    public function getPaymentInfoByPaymentIdAndCID($paymentId, $clientID = PaymentConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $paramData = $this->getPaymentParams($paymentId, $clientID);
        if ((empty($paramData) || !is_array($paramData)) && ($clientID != PaymentConstants::DEFAULT_CLIENT_ID)) {
            $paramData = $this->getPaymentParams($paymentId, PaymentConstants::DEFAULT_CLIENT_ID);
        }
        return $paramData;
    }

    /**
     * Get Payment Params
     * @method getPaymentParams
     * @access protected
     * @param integer $paymentId
     * @param string $clientID client id
     * @return array
     */
    protected function getPaymentParams($paymentId, $clientID) {
        $paramData = $this->_paymentDb->getPaymentParams($paymentId, $clientID);
        if (!empty($paramData)) {
            $paramData = json_decode($paramData, true);
        }
        return $paramData;
    }

    /**
     * create a unique pup token with appending given salt
     * @method createPupToken
     * @access protected
     * @param string $salt salt to be append in the end of pup token
     * @return string
     */
    protected function createPupToken($salt = '') {
        //check pup token 10 times in table: pup_user_card_info
        for ($retryCount = 1; $retryCount <= 10; $retryCount++) {
            $tempRefId = 'A' . $retryCount . StringOperations::getMd5UniqueId($salt);
            $tempRefId = strtoupper($tempRefId);
            $existCount = $this->_paymentDb->checkPupToken($tempRefId);
            if (empty($existCount)) {
                return $tempRefId;
            }
        }
        return '';
    }

    /**
     * This method is used to save card info
     * @method saveCardInfo
     * @access public
     * @param array $cardDetails
     */
    public function saveCardInfo($cardDetails) {
        try {
            $paymentId = PaymentConstants::Pgw2C2P_PAYMENT_ID;
            $paymentInfo = $this->_paymentDb->getPaymentInfoByPaymentId($paymentId);
            DI::Map('EncryptDecrypt', 'clues\\model\\pgwVerify\\util\\EncryptDecrypt');
            $encryptDecrypt = DI::Singleton('EncryptDecrypt');
            $encryptDecrypt->setKey($paymentInfo['params']['cardAESKey']);
            $tempString = $encryptDecrypt->decrypt($cardDetails, $paymentInfo['params']['cardAESMethod']);
            $encCardData = json_decode($tempString, true);
            if (!isset($encCardData['user_id']) || empty($encCardData['user_id'])) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "User Id missing";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::USER_ID_MISSING, $exceptionData);
            }
            if (!isset($encCardData['card_no']) || empty($encCardData['card_no']) || !isset($encCardData['expiry_month']) || empty($encCardData['expiry_month']) || !isset($encCardData['expiry_year']) || empty($encCardData['expiry_year'])) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Card details missing";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::CARD_DETAILS_MISSING, $exceptionData);
            }
            if(!is_string($encCardData['card_no']) || !is_string($encCardData['expiry_month']) || !is_string($encCardData['expiry_year']) || (isset($encCardData['cvv']) && !empty($encCardData['cvv']) && !is_string($encCardData['cvv']))) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Invalid input";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new CommonException(Status::INVALID_INPUT, null, null, null, $exceptionData);
            }
            $currentYear = (int) date("y");
            $currentMonth = (int) date("m");
            if(strlen($encCardData['expiry_month']) > 2 || (int) $encCardData['expiry_month'] < 1 || (int) $encCardData['expiry_month'] > 12 || strlen($encCardData['expiry_year']) < 2 || strlen($encCardData['expiry_year']) > 4) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Invalid card expiry";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::INVALID_CARD_EXPIRY, $exceptionData);
            }
            if(substr((int) $encCardData['expiry_year'], -2) < substr($currentYear, -2) || (substr((int) $encCardData['expiry_year'], -2) == substr($currentYear, -2) && (int) $encCardData['expiry_month'] < $currentMonth)) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Card expired";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::CARD_EXPIRED, $exceptionData);
            }
            if(strlen($encCardData['expiry_month']) == 1) {
                $encCardData['expiry_month'] = '0' . $encCardData['expiry_month'];
            }
            if(isset($encCardData['cvv']) && !empty($encCardData['cvv']) && (strlen($encCardData['card_no']) + strlen($encCardData['cvv']) != 19)) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Invalid card details";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::INVALID_CARD_DETAILS, $exceptionData);
            }
            if(!preg_match('/^\d{15,16}$/', $encCardData['card_no'])) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Invalid card number";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::INVALID_CARD_NO, $exceptionData);
            }
            if(isset($encCardData['cvv']) && !empty($encCardData['cvv']) && !preg_match('/^\d{3,4}$/', $encCardData['cvv'])) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Invalid card cvv";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::INVALID_CARD_CVV, $exceptionData);
            }
            $encCardData['card_nickname'] = (isset($encCardData['name']) && !empty($encCardData['name'])) ? $encCardData['name'] : 'User';
            $encCardData['payment_id'] = $paymentId;
            $stringToHash = $encCardData['card_no'] . '|User|' . $encCardData['user_id'];
            $cardHash = strtoupper(hash_hmac('sha256', $stringToHash, $paymentInfo['params']['cardHashKey'], false));
            $tokenExists = $this->_paymentDb->checkCardHashExists($encCardData['user_id'], $cardHash);
            if ($tokenExists) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Card already exists";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::CARD_ALREADY_EXISTS, $exceptionData);
            }
            DI::Map('CardMaintainance2C2P', 'clues\\model\\pgw\\CardMaintainance2C2P');
            $cardMaintObj = DI::Singleton('CardMaintainance2C2P');
            $response = $cardMaintObj->addCardDetails($encCardData);
            if (!isset($response['response']) || empty($response['response'])) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($encCardData);
                $exceptionData['error_name'] = "Error in saving card at pgw end";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                if (isset($response['error']) && !empty($response['error']) && $response['error'] == 1 && isset($response['msg']) && !empty($response['msg'])) {
                    $exceptionData['error_name'] = $response['msg'];
                }
                throw new PaymentException(PaymentStatus::CARD_NOT_SAVED_AT_PGW_END, $exceptionData);
            }
            if (isset($response['response']) && !empty($response['response']) && isset($response['response']['respCode']) && !empty($response['response']['respCode']) && !in_array($response['response']['respCode'], PaymentConstants::Pgw2C2P_CARD_SUCCESS_RESPONSE_CODES)) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($response);
                $exceptionData['error_name'] = $response['response']['respReason'] ?? PaymentStatus::CARD_NOT_SAVED_AT_PGW_END;
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::CARD_NOT_SAVED_AT_PGW_END, $exceptionData);
            }
            $addCardDetailsArr = array(
                'user_id' => $encCardData['user_id'],
                'card_nickname' => $encCardData['card_nickname'],
                'default_card' => $encCardData['default_card'] ?? 0,
                'card_token' => $response['response']['storeCardUniqueID'],
                'pup_tk' => $this->createPupToken($encCardData['user_id']),
                'card_hash' => $cardHash,
                'card_hash_2c2p' => $response['response']['hashValue'],
                'masked_card_no' => 'XXXXXXXXXXXX' . substr($encCardData['card_no'], -4),
                'expiry_month' => $encCardData['expiry_month'],
                'expiry_year' => substr($encCardData['expiry_year'], -2)
            );
            $cardId = $this->_paymentDb->addCardDetails($addCardDetailsArr);
            if (empty($cardId)) {
                $exceptionData = array();
                $exceptionData['field1'] = json_encode($addCardDetailsArr);
                $exceptionData['error_name'] = "Error while saving the card";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $additionalInfo['placeholders'] = array('save card');
                $exceptionData['additional_data'] = $additionalInfo;
                throw new CommonException(Status::DATABASE_OPREATION_FAILURE, null, null, null, $exceptionData);
            }
            $this->_cluesHistoryObj->insertIntoHistoryTable(PaymentConstants::USER_CARD_INFO_TABLE, PaymentConstants::USER_CARD_INFO_HISTORY_TABLE, 'id', $cardId);
            if (isset($encCardData['default_card']) && !empty($encCardData['default_card']) && !empty($cardId)) {
                $defaultCardsArr = $this->_paymentDb->getDefaultCardsOfUser($encCardData['user_id'], 1, $cardId);
                $this->_paymentDb->updateDefaultCardsOfUser($encCardData['user_id'], 0, $cardId);
                foreach($defaultCardsArr as $key => $value) {
                    $this->_cluesHistoryObj->insertIntoHistoryTable(PaymentConstants::USER_CARD_INFO_TABLE, PaymentConstants::USER_CARD_INFO_HISTORY_TABLE, 'id', $value['id']);
                }
            }
            return array(
                'card_status' => 200,
                'pup_card_token' => $addCardDetailsArr['pup_tk']
            );
        } catch (\Exception $e) {
            $logData = array('field1' => 'requestData' . json_encode($cardDetails), 'error_name' => 'saveCardException', 'data' => $e->getMessage());
            $this->LoggerObj->log(PaymentConstants::DOMAIN_NAME, PaymentConstants::CARD_OPERATIONS_MODULE_NAME, json_encode($logData), Logger::ERROR);
            unset($logData);
            throw $e;
        }
    }

    /**
     * This method is get cards info of a user
     * @method getUserCardInfo
     * @access public
     * @param array $requestData
     * @return array $responseCards
     */
    public function getUserCardInfo($requestData) {
        if (!isset($requestData['user_id']) || empty($requestData['user_id'])) {
            $exceptionData = array();
            $exceptionData['field1'] = json_encode($requestData);
            $exceptionData['error_name'] = "User Id missing";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PaymentException(PaymentStatus::USER_ID_MISSING, $exceptionData);
        }
        $userId = $requestData['user_id'];
        $responseCards = array();
        $defaultCards = array(0, 1);
        if (isset($requestData['only_default_card']) && !empty($requestData['only_default_card'])) {
            $defaultCards = array(1);
        }
        $responseCards = $this->_paymentDb->getUserCardInfo($userId, $defaultCards);
        return array('cards' => $responseCards);
    }

    /**
     * This method is delete card of a user
     * @method deleteUserCard
     * @access public
     * @param integer $userId
     * @param string $pupToken
     * @return array $response
     */
    public function deleteUserCard($userId, $pupToken) {
        $cardData = $this->_paymentDb->getDataForDeleteCard($userId, $pupToken);
        if (!isset($cardData['id']) || empty($cardData['id'])) {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['field2'] = 'userId' . $userId;
            $exceptionData['error_name'] = "No Card Found";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $additionalInfo['placeholders'] = array('card token', $pupToken);
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PaymentException(PaymentStatus::INVALID_TOKEN, $exceptionData);
        }
        DI::Map('BookingModel', 'clues\\model\\booking\\BookingModel');
        $bookingModelObject = DI::Singleton("BookingModel");
        $isBookingPresent = $bookingModelObject->isBookingPresentwithCardtoken(array('user_id' => $userId, 'pup_card_token' => $pupToken));
        if ($isBookingPresent) {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['field2'] = 'userId' . $userId;
            $exceptionData['error_name'] = "Cannot delete card because booking present on this card";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $additionalInfo['placeholders'] = array('card token', $pupToken);
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PaymentException(PaymentStatus::INVALID_DELETE_BOOKING_ALREADY_PRESENT, $exceptionData);
        }
        $countOfSameCard = $this->_paymentDb->count2c2pCardToken($cardData['card_token'], PaymentConstants::INITIAL_STATUS_FOR_USER_CARD);
        $cardDelId = $this->_paymentDb->deleteUserCard($cardData['id'], PaymentConstants::DELETE_STATUS_FOR_USER_CARD, PaymentConstants::DELETE_CARD_MSG);
        $this->_cluesHistoryObj->insertIntoHistoryTable(PaymentConstants::USER_CARD_INFO_TABLE, PaymentConstants::USER_CARD_INFO_HISTORY_TABLE, 'id', $cardDelId);
        if (!isset($cardDelId) || empty($cardDelId)) {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['field2'] = 'userId' . $userId;
            $exceptionData['field3'] = 'cardId' . $cardDelId;
            $exceptionData['error_name'] = "Error while deleting the card";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $additionalInfo['placeholders'] = array('card deletion');
            $exceptionData['additional_data'] = $additionalInfo;
            throw new CommonException(Status::DATABASE_OPREATION_FAILURE, null, null, null, $exceptionData);
        }
        if ($countOfSameCard == 1) {
            $paymentId = PaymentConstants::Pgw2C2P_PAYMENT_ID;
            DI::Map('CardMaintainance2C2P', 'clues\\model\\pgw\\CardMaintainance2C2P');
            $cardMaintObj = DI::Singleton('CardMaintainance2C2P');
            $response = $cardMaintObj->deleteCard($cardData['card_token'], $paymentId);
            if (!isset($response['response']) || empty($response['response'])) {
                $exceptionData = array();
                $exceptionData['field1'] = 'response: ' . json_encode($response);
                $exceptionData['error_name'] = "Error in deleting card at pgw end";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                if (isset($response['error']) && !empty($response['error']) && $response['error'] == 1 && isset($response['msg']) && !empty($response['msg'])) {
                    $exceptionData['error_name'] = $response['msg'];
                }
                $this->LoggerObj->log(PaymentConstants::DOMAIN_NAME, PaymentConstants::CARD_OPERATIONS_MODULE_NAME, json_encode($exceptionData), Logger::ERROR);
                // throw new PaymentException(PaymentStatus::CARD_NOT_DELETED_AT_PGW_END, $exceptionData);
            }
            if (isset($response['response']) && !empty($response['response']) && isset($response['response']['respCode']) && !empty($response['response']['respCode']) && !in_array($response['response']['respCode'], PaymentConstants::Pgw2C2P_CARD_SUCCESS_RESPONSE_CODES)) {
                $exceptionData = array();
                $exceptionData['field1'] = 'response: ' . json_encode($response);
                $exceptionData['error_name'] = $response['response']['respReason'] ?? PaymentStatus::CARD_NOT_DELETED_AT_PGW_END;
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                $this->LoggerObj->log(PaymentConstants::DOMAIN_NAME, PaymentConstants::CARD_OPERATIONS_MODULE_NAME, json_encode($exceptionData), Logger::ERROR);
                // throw new PaymentException(PaymentStatus::CARD_NOT_DELETED_AT_PGW_END, $exceptionData);
            }
            $cardDelId = $this->_paymentDb->deleteUserCard($cardData['id'], PaymentConstants::DELETE_STATUS_FOR_USER_CARD_AT_PGW_END, PaymentConstants::DELETE_CARD_MSG_AT_PGW_END);
            $this->_cluesHistoryObj->insertIntoHistoryTable(PaymentConstants::USER_CARD_INFO_TABLE, PaymentConstants::USER_CARD_INFO_HISTORY_TABLE, 'id', $cardDelId);
            if (!isset($cardDelId) || empty($cardDelId)) {
                $exceptionData = array();
                $exceptionData['field1'] = 'puptoken' . $pupToken;
                $exceptionData['field2'] = 'userId' . $userId;
                $exceptionData['field3'] = 'cardId' . $cardDelId;
                $exceptionData['error_name'] = "Error while deleting the card";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $additionalInfo['placeholders'] = array('card deletion');
                $exceptionData['additional_data'] = $additionalInfo;
                $this->LoggerObj->log(PaymentConstants::DOMAIN_NAME, PaymentConstants::CARD_OPERATIONS_MODULE_NAME, json_encode($exceptionData), Logger::ERROR);
                // throw new CommonException(Status::DATABASE_OPREATION_FAILURE, null, null, null, $exceptionData);
            }
        }
        $response = array(
            'msg' => PaymentConstants::CARD_DELETE_MSG,
            'status' => Status::OK
        );
        return $response;
    }

    /**
     * This method is used to get payment all details by shopclues order id
     * @method getPaymentDetailsByOrderId
     * @access public
     * @param integer $orderId order id
     * @return array
     */
    public function getPaymentDetailsByOrderId($orderId) {
        return $this->_paymentDb->getPaymentDetailsByOrderId($orderId);
    }

    /**
     * This method is used to check pup token is valid or not
     * @method isValidPupUserToken
     * @access public
     * @param integer $userId
     * @param string $pupToken
     * @return boolean
     */
    public function isValidPupUserToken($userId, $pupToken) {
        if (!isset($userId) || empty($userId)) {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['error_name'] = "User Id missing";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PaymentException(PaymentStatus::USER_ID_MISSING, $exceptionData);
        }
        if (!isset($pupToken) || empty($pupToken)) {
            $exceptionData = array();
            $exceptionData['field1'] = 'userId' . $userId;
            $exceptionData['error_name'] = "Pup Token missing";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PaymentException(PaymentStatus::PUP_TOKEN_MISSING, $exceptionData);
        }
        $response = $this->_paymentDb->getCardInfoFromToken($userId, $pupToken);
        if (!empty($response)) {
            return $response;
        }
        return false;
    }

    /**
     * This method is used to update name and default card status according to operation
     * @method cardOperation
     * @access public
     * @param integer $userId
     * @param string $pupToken
     * @param integer $requestData
     * @return array
     */
    public function cardOperation($userId, $pupToken, $requestData) {
        if (isset($requestData['operation']) && !empty($requestData['operation'])) {
            switch ($requestData['operation']) {
                case 'update_nick_name':
                    if (isset($requestData['nick_name']) && !empty($requestData['nick_name'])) {
                        $response = $this->_setDefaultNickName($userId, $pupToken, $requestData['nick_name']);
                    } else {
                        $exceptionData = array();
                        $exceptionData['field1'] = 'puptoken' . $pupToken;
                        $exceptionData['field2'] = 'userId' . $userId;
                        $exceptionData['error_name'] = "Please provide Cards nick name";
                        $additionalInfo['file'] = __FILE__;
                        $additionalInfo['line'] = __LINE__;
                        $additionalInfo['method'] = __METHOD__;
                        $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                        $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                        $exceptionData['additional_data'] = $additionalInfo;
                        throw new PaymentException(PaymentStatus::PROVIDE_CARD_NICKNAME, $exceptionData);
                    }
                    break;
                case 'set_default_card':
                    $response = $this->_setDefaultCard($userId, $pupToken);
                    break;
                default:
                    $exceptionData = array();
                    $exceptionData['field1'] = 'puptoken' . $pupToken;
                    $exceptionData['field2'] = 'userId' . $userId;
                    $exceptionData['error_name'] = "Invalid operation";
                    $additionalInfo['file'] = __FILE__;
                    $additionalInfo['line'] = __LINE__;
                    $additionalInfo['method'] = __METHOD__;
                    $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                    $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                    $exceptionData['additional_data'] = $additionalInfo;
                    throw new PaymentException(PaymentStatus::INVALID_OPERATION, $exceptionData);
            }
            return $response;
        } else {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['field2'] = 'userId' . $userId;
            $exceptionData['error_name'] = "missing operation";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $exceptionData['additional_data'] = $additionalInfo;
            throw new CommonException(Status::PARAMETER_MISSING, null, null, null, $exceptionData);
        }
    }

    /**
     * This method is used to update name
     * @method _setDefaultNickName
     * @access private
     * @param integer $userId
     * @param string $pupToken
     * @param string $nickName
     * @return array
     */
    private function _setDefaultNickName($userId, $pupToken, $nickName) {
        $nickName = trim($nickName);
        $cardData = $this->_paymentDb->getDataForCurrentToken($userId, $pupToken);
        if (!isset($cardData['id']) || empty($cardData['id'])) {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['field2'] = 'userId' . $userId;
            $exceptionData['error_name'] = "No Card Found";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $additionalInfo['placeholders'] = array('card token', $pupToken);
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PaymentException(PaymentStatus::INVALID_TOKEN, $exceptionData);
        }
        if (!preg_match('/^[a-zA-Z0-9][\s\w\'`.-]+[a-zA-Z0-9]$/', $nickName)) {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['field2'] = 'userId' . $userId;
            $exceptionData['field3'] = 'nick_name' . $nickName;
            $exceptionData['error_name'] = "please provide alpha numeric nickname";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $additionalInfo['placeholders'] = array('nick_name');
            $exceptionData['additional_data'] = $additionalInfo;
            throw new CommonException(Status::INVALID_INPUT, null, null, null, $exceptionData);
        }
        if (isset($nickName) && !empty($nickName) && preg_match('/^[a-zA-Z0-9][\s\w\'`.-]+[a-zA-Z0-9]$/', $nickName)) {
            if ($nickName == $cardData['name']) {
                $exceptionData = array();
                $exceptionData['field1'] = 'puptoken' . $pupToken;
                $exceptionData['field2'] = 'userId' . $userId;
                $exceptionData['error_name'] = "Card name is already set as the given name";
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['method'] = __METHOD__;
                $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
                $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PaymentException(PaymentStatus::CARD_NAME_ALREADY_EXISTS, $exceptionData);
            } else {
                $response = $this->_paymentDb->setDefaultNickName($cardData['id'], $userId, $nickName);
                $this->_cluesHistoryObj->insertIntoHistoryTable(PaymentConstants::USER_CARD_INFO_TABLE, PaymentConstants::USER_CARD_INFO_HISTORY_TABLE, 'id', $cardData['id']);
                if (isset($response) && !empty($response)) {
                    $result = array(
                        'msg' => PaymentConstants::CARD_NICKNAME_MSG,
                        'status' => Status::OK
                    );
                }
            }
        }
        return $result;
    }

    /**
     * This method is used to update default card status 
     * @method _setDefaultCard
     * @access private
     * @param integer $userId
     * @param string $pupToken
     * @return array
     */
    private function _setDefaultCard($userId, $pupToken) {
        $cardData = $this->_paymentDb->getInfoIfDefaultCard($userId, $pupToken);
        if (!isset($cardData['id']) || empty($cardData['id'])) {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['field2'] = 'userId' . $userId;
            $exceptionData['error_name'] = "No Card Found";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $additionalInfo['placeholders'] = array('card token', $pupToken);
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PaymentException(PaymentStatus::INVALID_TOKEN, $exceptionData);
        }
        if (isset($cardData) && !empty($cardData) && $cardData['default_card'] == 0) {
            $this->_paymentDb->updateDefaultCardsOfUser($userId, 0, $cardData['id']);
            $response = $this->_paymentDb->setDefaultCard($cardData['id'], $userId, 1);
            $this->_cluesHistoryObj->insertIntoHistoryTable(PaymentConstants::USER_CARD_INFO_TABLE, PaymentConstants::USER_CARD_INFO_HISTORY_TABLE, 'id', $cardData['id']);
            if (isset($response) && !empty($response)) {
                $response = array(
                    'msg' => PaymentConstants::CARD_SET_DEFAULT_MSG,
                    'status' => Status::OK
                );
            }
        } else {
            $exceptionData = array();
            $exceptionData['field1'] = 'puptoken' . $pupToken;
            $exceptionData['field2'] = 'userId' . $userId;
            $exceptionData['error_name'] = "Card already set as default";
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['domain'] = PaymentConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PaymentConstants::CARD_OPERATIONS_MODULE_NAME;
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PaymentException(PaymentStatus::CARD_ALREADY_SET_AS_DEFAULT, $exceptionData);
        }
        return $response;
    }

}
