<?php

namespace clues\model\pgw\PGWHashManager;

use clues\model\pgw\PGWDriver;
use clues\system\DI;
use clues\model\pgw\PGWHashManager\PGWHashConstants;
use clues\model\pgw\PgwModelConstants;
use clues\system\exception\Status;
use clues\model\pgw\exception\PgwException;

/**
 * This class is used to perform operations related to hash
 * @class PGWHashManager
 * @access public
 * @package model/pgw/PGWHashManager
 * @category PGWHashManager
 */
class PGWHashManager extends PGWDriver {

    /**
     * pgw hash class object
     * @var array
     */
    protected $pgwHashObj;

    /**
     * Used to hold pgw mapping with hash class
     * @var array
     */
    private static $_pgwHashMapping = [
        PGWHashConstants::OLAMONEY_HASH_CLASS => 'OlaMoneyHash',
        PGWHashConstants::MOBIKWIK_HASH_CLASS => 'MobikwikHash',
        PGWHashConstants::MOBIKWIK_SEAMLESS_HASH_CLASS => 'MobikwikSeamlessHash',
        PGWHashConstants::OLAMONEY_HASH_CLASS => 'OlaMoneyHash',
        PGWHashConstants::BILLDESK_HASH_CLASS => 'BillDeskHash',
        PGWHashConstants::PAYTM_HASH_CLASS => 'PaytmHash',
        PGWHashConstants::EPAYLATER_HASH_CLASS => 'EpayLaterHash',
        PGWHashConstants::PAYU_UPI_QR_HASH_CLASS => 'PayuUpiQrHash',
        PGWHashConstants::JIO_WALLET_HASH_CLASS => 'JioWalletHash',
        PGWHashConstants::MOMOE_HASH_CLASS => 'MomoeHash'
    ];

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @return void
     */
    public function __construct(array $data = array()) {
        parent::__construct($data);
    }

    /**
     * This method is used to get hash string of response data
     * @method getRequestHash
     * @access public
     * @param array $param
     * @param string $hashClass
     * @param object $orderInfo
     * @return string $hashString
     */
    public function getRequestHash(array $param, $hashClass, $orderInfo) {
        $this->initializePgwObj($hashClass);
        $hashString = $this->pgwHashObj->generateRequestHash($param, $orderInfo);
        return $hashString;
    }

    /**
     * This method is used to get hash string of response data
     * @method getResponseHash
     * @access public
     * @param array $param
     * @param string $hashClass
     * @param object $orderInfo
     * @return String $hashString
     */
    public function getResponseHash(array $param, $hashClass, $orderInfo) {
        $this->initializePgwObj($hashClass);
        $hashString = $this->pgwHashObj->generateResponseHash($param, $orderInfo);
        return $hashString;
    }

    /**
     * This method is used to get hash string of response data
     * @method getNotifyResponseHash
     * @access public
     * @param array $param
     * @param string $hashClass
     * @param object $orderInfo
     * @return String $hashString
     */
    public function getNotifyResponseHash(array $param, $hashClass, $orderInfo) {
        $this->initializePgwObj($hashClass);
        $hashString = $this->pgwHashObj->generateNotifyResponseHash($param, $orderInfo);
        return $hashString;
    }

    /**
     * This method is used to get hash string of response data
     * @method checkResponseHash
     * @access public
     * @param array $param 
     * @param string $hashClass
     * @param object $orderInfo
     * @return string (True/False)
     */
    public function checkResponseHash(array $param, $hashClass, $orderInfo) {
        $this->initializePgwObj($hashClass);
        return $this->pgwHashObj->checkResponseHash($param, $orderInfo);
    }

    /**
     * Initialize pgw hash object
     * @method initializePgwObj
     * @access protected
     * @param string $pgwClassName
     * @throws Exception
     * @return void
     */
    protected function initializePgwObj($pgwClassName) {
        if (isset(static::$_pgwHashMapping[$pgwClassName]) && !empty(static::$_pgwHashMapping[$pgwClassName])) {
            $tempStr = 'clues\\model\\pgw\\PGWHashManager\\' . $pgwClassName;
            $tempMap = md5($tempStr);
            DI::Map($tempMap, $tempStr);
            $pgwObj = DI::Singleton($tempMap);
            $this->pgwHashObj = $pgwObj;
        } else {
            $exceptionData = array();
            $exceptionData['error_name'] = 'Invalid Payment Gateway Hash Creation';
            $exceptionData['field1'] = $pgwClassName;
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['level'] = Logger::ERROR;
            $additionalInfo['domain'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['module'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['method'] = __METHOD__;
            $additionalInfo['input_parms'] = $pgwClassName;
            $exceptionData['additional_data'] = $additionalInfo;
            throw new PgwException(Status::INVALID_INPUT, $exceptionData);
        }
    }

    
    /**
     * This method is used to call hasClass library methods
     * @method __call
     * @access public
     * @param string $name method name
     * @param array $arguments method arguments
     * @return mixed
     * @throws PgwException
     */
    public function __call($name, $arguments){
        $this->initializePgwObj($arguments[0]['hashClass']);
        $check = method_exists($this->pgwHashObj, $name);
        if ($check) {
            return call_user_func_array(array($this->pgwHashObj, $name), $arguments);
        } else {
            $className = get_class($this->pgwHashObj);
            $additionalInfo = $additionalInfo['additional_data'] = array();
            $additionalInfo['field1'] = $name;
            $additionalInfo['field2'] = 'Exception code:' . Status::METHOD_NOT_EXIST;
            $additionalInfo['additional_data']['file'] = __FILE__;
            $additionalInfo['additional_data']['line'] = __LINE__;
            $additionalInfo['additional_data']['domain'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['additional_data']['module'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['additional_data']['level'] = Logger::ERROR;
            $additionalInfo['additional_data']['method'] = __METHOD__;
            $additionalInfo['additional_data']['input_params'] = $arguments;
            $additionalInfo['additional_data']['placeholders'] = array($name, $className);
            throw new PgwException(Status::METHOD_NOT_EXIST, $additionalInfo);
        }

    }

}
