<?php

namespace clues\model\pgw;

use clues\system\DBOperations;
use clues\model\pgw\PgwModelConstants;

/**
 * This class is used for all payment db CRUD operations
 * @class PaymentService
 * @category Model datasource
 * @package model/pgw
 * @access public
 * @dependencies DBOperations
 */
class PaymentService extends DBOperations {

    /**
     * Soa payment api partial path
     * @var string const
     */
    const SOA_PAYMENT_API = 'soa_payment_api';

    /**
     * used to log Order pre payment log data
     * @method savePrePaymentData
     * @access public
     * @param array $prepaymentData prepayment data
     * @return void
     */
    public function savePrePaymentData($prepaymentData) {
        $query = 'INSERT into pup_order_pgw (client_order_id, amount, payment_gateway, order_data, merchant_params) VALUES (?s, ?d, ?s, ?s, ?s)';
        $query = $this->dbQuote($query, $prepaymentData['order_id'], $prepaymentData['amount'], $prepaymentData['payment_gateway'], $prepaymentData['order_data'], $prepaymentData['merchant_params']);
        return $this->dbQueryInsert($query);
    }

    /**
     * used to log Order post payment log data
     * @method saveAfterPaymentData
     * @access public
     * @param array $insert_array post payment data
     * @return void
     */
    public function saveAfterPaymentData($insert_array) {
        $query = 'INSERT into pup_prepayment_details (direcpayreferenceid, client_order_id, flag, other_details, amount, payment_gateway, 3dstatus) VALUES (?s, ?s, ?s, ?s, ?s, ?s, ?s)';
        $query = $this->dbQuote($query, $insert_array['direcpayreferenceid'], $insert_array['order_id'], $insert_array['flag'], $insert_array['other_details'], $insert_array['amount'], $insert_array['payment_gateway'], $insert_array['status_code']);
        return $this->dbQueryInsert($query);
    }

    /**
     * This method used to get pre payment details
     * @method get_prepayment_details
     * @access public
     * @param integer $order_id order id
     * @return array
     */
    public function get_prepayment_details($order_id) {
        $sql = 'SELECT id, direcpayreferenceid, order_id, flag, other_details, amount, payment_gateway,txn_response, 3dstatus FROM pup_prepayment_details WHERE client_order_id = ?s';
        $sql = $this->dbQuote($sql, $order_id);
        return $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get payment data for given payment option id
     * @method getFirstPriorityPaymentDataByPaymentOptId
     * @access public
     * @param integer $payment_option_id payment option id
     * @return array
     */
    public function getFirstPriorityPaymentDataByPaymentOptId($payment_option_id) {
        $sql = 'SELECT cp.payment_option_id, cp.payment_gateway_id, cp.payment_option_pgw_id, cp.bank_code, cp.priority, cpo.payment_type_id, cp.product_code
                FROM pup_payment_option_pgw cp
                JOIN pup_payment_options cpo ON cpo.payment_option_id = cp.payment_option_id
                WHERE cp.priority=?s and cp.status=?s and cp.payment_option_id=?i';
        $sql = $this->dbQuote($sql, '1', 'A', $payment_option_id);
        $payment_details = $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
        return $payment_details;
    }

    /**
     * This method is used to get emi payment data for given payment option id
     * @method getEmiPaymentDataByPaymentOptId
     * @access public
     * @param integer $payment_option_id payment option id
     * @return array
     */
    public function getEmiPaymentDataByPaymentOptId($payment_option_id) {
        $sql = 'select cpe.payment_option_id, cpe.payment_gateway_id, cpo.payment_type_id
                    from pup_payment_options_emi_pgw cpe
                    join pup_payment_options cpo on cpo.payment_option_id = cpe.payment_option_id
                    where cpe.status=?s and cpe.payment_option_id=?i';
        $sql = $this->dbQuote($sql, 'A', $payment_option_id);
        $payment_details = $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
        return $payment_details;
    }

    /**
     * This method is used to get emi code by emi id
     * @method getEmiCodeByEmiId
     * @access public
     * @param integer $emi_id emi id
     * @return string
     */
    public function getEmiCodeByEmiId($emi_id) {
        $emi_code_sql = 'select emi_code, bank_code from pup_payment_options_emi_pgw where id = ?i';
        $emi_code_sql = $this->dbQuote($emi_code_sql, $emi_id);
        $emi_code = $this->dbGetRow($emi_code_sql, array(), self::MASTER_NOCACHE);
        return $emi_code;
    }

    /**
     * This method is used to payment processor by payment id
     * @method getPaymentProcessor
     * @access public
     * @param integer $payment_id payment id
     * @param string $clientID
     * @return array
     */
    public function getPaymentProcessor($payment_id, $clientID = PgwModelConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PgwModelConstants::DEFAULT_CLIENT_ID;
        }
        $sql = "SELECT processor_id FROM pup_payments WHERE payment_id = ?i AND client_id=?s AND status != ?s";
        $sanitizeSql = $this->dbQuote($sql, $payment_id, $clientID, 'R');
        return $this->dbGetField($sanitizeSql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to payment processor script by processor id
     * @method getPgwFromId
     * @access public
     * @param integer $processor_id processor id
     * @return array
     */
    public function getPgwFromId($processor_id) {
        $sql = "SELECT processor_script FROM pup_payment_processors WHERE processor_id = ?i";
        $sanitizeSql = $this->dbQuote($sql, $processor_id);
        return $this->dbGetField($sanitizeSql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to get pgw details
     * @method getPgwDetails
     * @access public
     * @param integer $payment_option_id payment option id
     * @return array
     */
    public function getPgwDetails($payment_option_id) {
        $sql = "select cpop.payment_option_pgw_id, cpop.payment_option_id, cpop.payment_gateway_id, cpop.priority, cpop.bank_code, cpop.status, cpt.name as payment_type, cpo.name as payment_option from pup_payment_option_pgw cpop join pup_payment_options cpo on (cpop.payment_option_id=cpo.payment_option_id) join pup_payment_types cpt on (cpo.payment_type_id=cpt.payment_type_id) where cpop.payment_option_id=?i and cpop.status = ?s order by priority asc";
        $sql = $this->dbQuote($sql, $payment_option_id, 'A');
        return $this->dbGetRow($sql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * Get Payment Info of order
     * @method getPaymentInfo
     * @access public
     * @param integer $payment_id
     * @param string $clientID
     * @return array
     */
    public function getPaymentInfo($payment_id, $clientID = PgwModelConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PgwModelConstants::DEFAULT_CLIENT_ID;
        }
        $sql = "SELECT 
                    cp.payment_id,
                    cp.usergroup_ids,
                    cp.position,
                    cp.status,
                    cp.template,
                    cp.processor_id,
                    cp.params,
                    cp.a_surcharge,
                    cp.p_surcharge,
                    cp.localization,
                    cpd.payment_id as desc_payment_id,
                    cpd.payment,
                    cpd.description,
                    cpd.instructions,
                    cpd.lang_code,
                    cpp.processor  
                FROM 
                    pup_payments  cp
                LEFT JOIN pup_payment_processors cpp
                        ON cpp.processor_id = cp.processor_id 
                LEFT JOIN pup_payment_descriptions cpd
                        ON cp.payment_id = cpd.payment_id AND cpd.cpd_status != ?i
                WHERE cp.payment_id = ?i AND cp.client_id=?s AND cp.status != ?s";
        $sanitizeSql = $this->dbQuote($sql, 2, $payment_id, $clientID, 'R');
        $payment = $this->dbGetRow($sanitizeSql, array(), self::MASTER_NOCACHE);
        $payment['params'] = (!empty($payment['params'])) ? json_decode($payment['params'], true) : '';
        return $payment;
    }

    /**
     * Get payment id data by payment option id and emi id
     * @method getPaymentId
     * @access public
     * @param integer $paymentOptionId payment option id
     * @param integer $emiId emi id
     * @param string $clientID client id
     * @return mixed
     */
    public function getPaymentId($paymentOptionId, $emiId = 0, $clientID = PgwModelConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PgwModelConstants::DEFAULT_CLIENT_ID;
        }
        if (!empty($paymentOptionId) && !empty($emiId)) {
            $sql = "SELECT 
                        cpoep.id,
                        cpoep.payment_option_id,
                        cpoep.payment_gateway_id
                    FROM
                        pup_payment_options_emi_pgw cpoep
                            JOIN
                        pup_payment_options cpo ON (cpoep.payment_option_id = cpo.payment_option_id)
                            JOIN
                        pup_payment_types cpt ON (cpo.payment_type_id = cpt.payment_type_id)
                            JOIN
                        pup_payments cp ON (cpoep.payment_gateway_id = cp.payment_id)
                    WHERE
                        cpoep.payment_option_id = ?i
                            AND cpoep.status = ?s
                            AND cp.status = ?s
                            AND cp.client_id = ?s
                            AND cpoep.id = ?i";
            $sql = $this->dbQuote($sql, $paymentOptionId, 'A', 'A', $clientID, $emiId);
            $result = $this->dbGetRow($sql, array(), DBOperations::SLAVE_CACHE);
            return $result['payment_gateway_id'];
        } else if (!empty($paymentOptionId)) {
            $sql = "SELECT 
                        cpop.payment_option_pgw_id,
                        cpop.payment_option_id,
                        cpop.payment_gateway_id
                    FROM
                        pup_payment_option_pgw cpop
                            JOIN
                        pup_payment_options cpo ON (cpop.payment_option_id = cpo.payment_option_id)
                            JOIN
                        pup_payment_types cpt ON (cpo.payment_type_id = cpt.payment_type_id)
                            JOIN
                        pup_payments cp ON (cpop.payment_gateway_id = cp.payment_id)
                    WHERE
                        cpop.payment_option_id = ?i
                            AND cpop.status = ?s
                            AND cp.status = ?s
                            AND cp.client_id = ?s
                    ORDER BY priority ASC 
                    LIMIT 1";
            $sql = $this->dbQuote($sql, $paymentOptionId, 'A', 'A', $clientID);
            $result = $this->dbGetRow($sql, array(), self::SLAVE_CACHE);
            return $result['payment_gateway_id'];
        } else {
            return false;
        }
    }

    /**
     * Get Payment details by order id from pup_payment_pgw_mapping
     * @method getPaymentInfoFromPgwMapping
     * @access public
     * @param integer $orderId order id
     * @return array
     */
    public function getPaymentInfoFromPgwMapping($orderId) {
        $sql = 'SELECT payment_id,payment_option_id, pgw_order_id, created FROM pup_payment_pgw_mapping WHERE client_order_id=?s AND status=?i AND prepayment_details_auto_increment_id=?i  ORDER BY id DESC LIMIT ?i';
        $sql = $this->dbQuote($sql, (int) $orderId, 1, 0, 1);
        return $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * Insert Payment details by order id from pup_payment_service_data
     * @method insertSoaUrls
     * @access public
     * @param integer $orderId order id
     * @param string $successUrl success url
     * @param string $failedUrl failed url
     * @return boolean
     */
    public function insertSoaUrls($orderId, $successUrl, $failedUrl) {
        $time = time();
        $source = self::SOA_PAYMENT_API;
        $query = "INSERT into pup_payment_service_data (client_order_id,timestamp,returnurl_success,returnurl_fail, source) value(?s,?i,?s,?s,?s)";
        $sql = $this->dbQuote($query, $orderId, $time, $successUrl, $failedUrl, $source);
        return $this->dbQueryInsert($sql);
    }

    /**
     * Get Payment deails by order id from pup_payment_service_data
     * @method getSoaUrls
     * @access public
     * @param integer $order_id order id 
     * @return array
     */
    public function getSoaUrls($order_id) {
        $source = self::SOA_PAYMENT_API;
        $query = "SELECT returnurl_success, returnurl_fail from pup_payment_service_data where client_order_id = ?s and source = ?s order by `timestamp` desc";
        $query = $this->dbQuote($query, $order_id, $source);
        $data = $this->dbGetArray($query, array(), self::MASTER_NOCACHE);
        return $data;
    }

    /**
     * Get Payment deails by order id from pup_payment_pgw_mapping
     * @method checkPgwOrderMapping
     * @access public
     * @param array $data  
     * @return integer
     */
    public function checkPgwOrderMapping($data) {
        $query = 'SELECT pgw_order_id from pup_payment_pgw_mapping where client_order_id = ?s and status = ?i and payment_id = ?i order by id desc limit 1';
        $query = $this->dbQuote($query, $data['orderId'], 1, $data['paymentId']);
        $pgwOrder = $this->dbGetField($query, array(), self::MASTER_NOCACHE);
        return $pgwOrder;
    }

    /**
     * Insert Payment deails by order id in pup_payment_pgw_mapping
     * @method insertPgwOrderMapping
     * @access public
     * @param array $pgwData  
     * @return integer
     */
    public function insertPgwOrderMapping($pgwData) {
        $time = time();
        $query = 'INSERT into pup_payment_pgw_mapping (client_order_id, pgw_order_id, order_pgw_auto_increment_id,prepayment_details_auto_increment_id, payment_id, payment_option_id, emi_id, status, created, updated, created_by, updated_by, client_id) VALUES (?s, ?s, ?i, ?i, ?i, ?i, ?i, ?i, ?i, ?i, ?i, ?i, ?s)';
        $query = $this->dbQuote($query, $pgwData['orderId'], $pgwData['pgwOrderId'], $pgwData['orderPgwPrimaryId'], $pgwData['prepaymentPgwPrimaryId'], $pgwData['paymentId'], $pgwData['paymentOptionId'], $pgwData['emiId'], 1, time(), time(), $pgwData['userId'], $pgwData['userId'], $pgwData['clientID']);
        return $this->dbQueryInsert($query);
    }

    /**
     * Update Payment details by order id in pup_payment_pgw_mapping
     * @method updateOrderPgwPrimaryKeyByPgwOrderId
     * @access public
     * @param integer $orderPgwPrimaryId
     * @param integer $pgwOrderId  
     * @param integer $userId
     * @param integer $paymentId payment id
     * @return integer
     */
    public function updateOrderPgwPrimaryKeyByPgwOrderId($orderPgwPrimaryId, $pgwOrderId, $userId, $paymentId) {
        $sql = 'UPDATE pup_payment_pgw_mapping SET order_pgw_auto_increment_id = ?i, updated = ?i, updated_by = ?i WHERE pgw_order_id = ?s AND payment_id=?i AND order_pgw_auto_increment_id = ?i';
        $sql = $this->dbQuote($sql, $orderPgwPrimaryId, time(), $userId, $pgwOrderId, $paymentId, 0);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * Get Payment details by order id in pup_payment_pgw_mapping
     * @method getScOrderIdByPgwOrderId
     * @access public
     * @param integer $pgwOrderId  
     * @param integer $paymentId payment id
     * @return integer
     */
    public function getScOrderIdByPgwOrderId($pgwOrderId, $paymentId = 0) {
        if (!empty($paymentId)) {
            $query = "SELECT client_order_id from pup_payment_pgw_mapping where pgw_order_id = ?s and status = ?i and prepayment_details_auto_increment_id = ?i and payment_id = ?i order by created desc limit ?i";
            $query = $this->dbQuote($query, $pgwOrderId, 1, 0, $paymentId, 1);
        } else {
            $query = "SELECT client_order_id from pup_payment_pgw_mapping where pgw_order_id = ?s and status = ?i and prepayment_details_auto_increment_id = ?i order by created desc limit ?i";
            $query = $this->dbQuote($query, $pgwOrderId, 1, 0, 1);
        }
        $scOrder = $this->dbGetField($query, array(), self::MASTER_NOCACHE);
        return $scOrder;
    }

    /**
     * Get Payment details by order id in pup_payment_pgw_mapping
     * @method getOrderIdByPgwOrderId
     * @access public
     * @param integer $pgwOrderId
     * @return integer
     */
    public function getOrderIdByPgwOrderId($pgwOrderId) {
        $query = 'SELECT client_order_id from pup_payment_pgw_mapping where pgw_order_id = ?s and status != ?i order by id desc limit ?i';
        $query = $this->dbQuote($query, $pgwOrderId, 0, 1);
        return $this->dbGetField($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * Update Payment details by order id in pup_payment_pgw_mapping
     * @method updatePrepaymentPrimaryKeyByPgwOrderId
     * @access public
     * @param integer $prepaymentPrimaryId
     * @param integer $pgwOrderId  
     * @param integer $userId
     * @param integer $orderId
     * @return integer
     */
    public function updatePrepaymentPrimaryKeyByPgwOrderId($prepaymentPrimaryId, $pgwOrderId, $userId, $orderId = 0) {
        $extraCond = '';
        if ($orderId > 0) {
            $extraCond = $this->dbQuote(' AND client_order_id = ?s ', $orderId);
        }
        $sql = 'UPDATE pup_payment_pgw_mapping SET prepayment_details_auto_increment_id = ?i, updated = ?i, updated_by = ?i WHERE pgw_order_id = ?s and status = ?i and prepayment_details_auto_increment_id = ?i ?p order by id desc limit ?i';
        $sql = $this->dbQuote($sql, $prepaymentPrimaryId, time(), $userId, $pgwOrderId, 1, 0, $extraCond, 1);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to save post payment response meta data
     * @method savePostPaymentMetaData
     * @access public
     * @param array $metaDataArray post payment meta data
     * @return void
     */
    public function savePostPaymentMetaData($metaDataArray) {
        $query = 'INSERT into pup_prepayment_details_data (`prepayment_id`, `key`, `value`, `status`,`created`, `updated`, `created_by`, `updated_by`) VALUES (?i, ?s, ?s, ?i, ?i, ?i, ?i, ?i)';
        $query = $this->dbQuote($query, $metaDataArray['prepayment_id'], $metaDataArray['key'], $metaDataArray['value'], $metaDataArray['status'], $metaDataArray['created'], $metaDataArray['updated'], $metaDataArray['created_by'], $metaDataArray['updated_by']);
        return $this->dbQueryInsert($query);
    }

    /**
     * This method is used to update payment status of prepayment_data
     * @method updatePaymentStatusPrepaymentData
     * @access public
     * @param integer $update post payment meta data
     * @param integer $prepaymentId
     * @return void
     */
    public function updatePaymentStatusPrepaymentData($update, $prepaymentId) {
        $query = "UPDATE pup_prepayment_details_data set value = ?i where prepayment_id = ?i";
        $query = $this->dbQuote($query, $update, $prepaymentId);
        return $this->dbQueryUpdate($query);
    }

    /**
     * this method is used to get prepayment data
     * @method getPrepaymentData
     * @access public
     * @param string $pgwOrderId
     * @param integer $paymentId
     * @return array
     */
    public function getPrepaymentData($pgwOrderId, $paymentId) {
        if (!empty($pgwOrderId) && !empty($paymentId)) {
            $query = 'select id, client_order_id,prepayment_details_auto_increment_id, payment_option_id, emi_id, payment_id, order_pgw_auto_increment_id from pup_payment_pgw_mapping where pgw_order_id = ?s and payment_id = ?i order by id desc limit 1';
            $sql = $this->dbQuote($query, $pgwOrderId, $paymentId);
            return $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
        } else {
            return array();
        }
    }

    /**
     * this method is used to update prepaymentId in pgw mapping
     * @method updatePrepaymentIdInMapping
     * @access public
     * @param array $whereArr
     * @param integer $prepaymentId
     * @param integer $updatedBy
     * @param boolean $isSingleUpdateRequired
     * @return void
     */
    public function updatePrepaymentIdInMapping($whereArr, $prepaymentId, $updatedBy = 0, $isSingleUpdateRequired = false) {
        $query = 'update pup_payment_pgw_mapping set prepayment_details_auto_increment_id=?i,updated=?i';
        if ($updatedBy > 0) {
            $query .= $this->dbQuote(',updated_by=?i', $updatedBy);
        }
        $query .= ' where pgw_order_id=?s and payment_id=?i';
        $sql = $this->dbQuote($query, $prepaymentId, time(), $whereArr['pgw_order_id'], $whereArr['payment_id']);
        if (!empty($isSingleUpdateRequired)) {
            $sql .= ' order by id desc limit 1;';
        }
        $this->dbQueryUpdate($sql);
    }

    /**
     * used to get payment status from prepayment data
     * @method getPaymentStatusFromPrepaymentData
     * @access public
     * @param integer $prepaymentId
     * @return integer
     */
    public function getPaymentStatusFromPrepaymentData($prepaymentId) {
        $query = 'select value from pup_prepayment_details_data where `key`=?s and prepayment_id=?i';
        $sql = $this->dbQuote($query, 'payment_status', $prepaymentId);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * this method used to update status in pup_payment_pgw_mapping
     * @method updateTransactionStatus
     * @access public
     * @param integer $id
     * @param integer $status
     * @return void 
     */
    public function updateTransactionStatus($id, $status) {
        $sql = 'UPDATE pup_payment_pgw_mapping SET status = ?i,updated=?i WHERE id = ?i';
        $sql = $this->dbQuote($sql, $status, time(), $id);
        $this->dbQueryUpdate($sql);
    }

    /**
     * Get Payment Params
     * @method getPaymentParams
     * @access public
     * @param integer $paymentId
     * @param string $clientID
     * @return string
     */
    public function getPaymentParams($paymentId, $clientID = PgwModelConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PgwModelConstants::DEFAULT_CLIENT_ID;
        }
        $sql = 'SELECT params FROM pup_payments WHERE payment_id = ?i AND client_id = ?s AND status != ?s';
        $sql = $this->dbQuote($sql, $paymentId, $clientID, 'R');
        return $this->dbGetField($sql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to get payment type id for all payment option ids
     * @method getAllPaymentOptionAndPaymentTypeHashArray
     * @access public
     * @return array
     */
    public function getAllPaymentOptionAndPaymentTypeHashArray() {
        $sql = "SELECT payment_option_id,payment_type_id FROM pup_payment_options WHERE status = 'A'";
        return $this->dbGetHashArray($sql, 'payment_option_id', array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * Update pgw request response in clues order pgw
     * @method updatePgwRequestResponseOrderPgw
     * @access public
     * @param string $requestResponse
     * @param integer $id
     * @return void
     */
    public function updatePgwRequestResponseOrderPgw($requestResponse, $id) {
        $sql = 'UPDATE pup_order_pgw SET pgw_request_response = ?s WHERE id = ?i';
        $sql = $this->dbQuote($sql, $requestResponse, $id);
        $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to get all active payment options name
     * @method getAllPaymentOptionsName
     * @access public
     * @return array
     */
    public function getAllPaymentOptionsName() {
        $query = 'SELECT payment_option_id,payment_option_name AS name FROM pup_payment_options WHERE CASE WHEN payment_option_id IN (?n) THEN 1=1 ELSE status=?s END';
        $sql = $this->dbQuote($query, array(PgwModelConstants::MOMOE_CC_PAYMENT_OPTION_ID, PgwModelConstants::MOMOE_DC_PAYMENT_OPTION_ID), 'A');
        return $this->dbGetHashArray($sql, 'payment_option_id', array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to get all active pgw name
     * @method getPgwName
     * @access public
     * @return array
     */
    public function getPgwName() {
        $query = 'SELECT payment_id,payment_gateway_name AS name FROM pup_payments WHERE status=?s';
        $sql = $this->dbQuote($query, 'A');
        return $this->dbGetHashArray($sql, 'payment_id', array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * Get shopclues order id in pup_payment_pgw_mapping by pgwOrderid
     * @method getScOrderIdsByPgwOrderId
     * @access public
     * @param string $pgwOrderId  
     * @return array
     */
    public function getScOrderIdsByPgwOrderId($pgwOrderId) {
        $sql = 'SELECT DISTINCT client_order_id FROM pup_payment_pgw_mapping WHERE pgw_order_id = ?s AND status != ?i';
        $query = $this->dbQuote($sql, $pgwOrderId, 0);
        $scOrderIds = $this->dbGetFields($query, array(), self::MASTER_NOCACHE);
        return $scOrderIds;
    }

    /**
     * Get Payment details by pgw_order id from pup_payment_pgw_mapping
     * @method getPaymentDetailsFromPgwMapping
     * @access public
     * @param integer $pgwOrderId pgw_order id
     * @return array
     */
    public function getPaymentDetailsFromPgwMapping($pgwOrderId) {
        $sql = 'SELECT payment_id,payment_option_id, pgw_order_id, created, client_order_id, order_pgw_auto_increment_id FROM pup_payment_pgw_mapping WHERE pgw_order_id=?s AND status !=?i ORDER BY id DESC LIMIT ?i';
        $sql = $this->dbQuote($sql, $pgwOrderId, 0, 1);
        return $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * checks if phone number numeric or not
     * @method checkPhoneNumberForEncryption
     * @access protected
     * @param mixed $phone
     * @return boolean
     */
    protected function checkPhoneNumberForEncryption($phone) {
        $tr = trim($phone);
        return is_numeric(substr($tr, -3));
    }

    /**
     * get client configurations for given client id
     * @method getClientConfigurations
     * @access public
     * @param string $clientId
     * @return array
     */
    public function getClientConfigurations($clientId) {
        $sql = 'SELECT client_id, client_configurations FROM clues_oauth_clients_details WHERE client_id = ?s AND status = ?i';
        $sanitizedSql = $this->dbQuote($sql, $clientId, 1);
        return $this->dbGetHashArray($sanitizedSql, 'client_id', array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * get payment type id and payment type name from payment option id
     * @method getPaymentTypeDetails
     * @access public
     * @param integer $paymentOptionId
     * @return array
     */
    public function getPaymentTypeDetails($paymentOptionId) {
        $sql = "SELECT 
                    t.payment_type_id,
                    IF(t.payment_type_id > 0,
                        t.`name`,
                        'CB/CB+/GC/Credit') AS pgw_type,
                    IF(o.payment_option_id > 0,
                        o.`name`,
                        'CB/CB+/GC/Credit') AS pgw_option_name
                FROM
                    pup_payment_options o
                        INNER JOIN
                    pup_payment_types t ON t.payment_type_id = o.payment_type_id
                WHERE
                    o.payment_option_id = ?i AND o.status != ?s";
        $sanitizedSql = $this->dbQuote($sql, $paymentOptionId, 'R');
        return $this->dbGetRow($sanitizedSql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * get pgw name from payment id
     * @method getPgwNameFromPaymentId
     * @access public
     * @param integer $paymentId
     * @return array
     */
    public function getPgwNameFromPaymentId($paymentId) {
        $sql = "SELECT 
                    IF(payment_id > 0,
                        payment,
                        'CB/CB+/GC/Credit') AS pgw
                FROM
                    pup_payment_descriptions
                WHERE
                    payment_id = ?i AND cpd_status = ?i";
        $sanitizedSql = $this->dbQuote($sql, $paymentId, 1);
        return $this->dbGetRow($sanitizedSql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * add pre pgw metadata
     * @method addPrePgwMetaData
     * @access public
     * @param array $metaDataArray ['order_pgw_id' = <id>, key = <k>, value = <v>, 'status' = <status>, 'created' = <created>, 'updated' = <updated>, 'created_by' = <created_by>, 'updated_by' = <updated_by>]
     * @return integer
     */
    public function addPrePgwMetaData(array $metaDataArray) {
        $sql = 'INSERT INTO pup_order_pgw_data (order_pgw_id, object_key, object_value, status, created, updated, created_by, updated_by) VALUES (?i, ?s, ?s, ?i, ?i, ?i, ?i, ?i)';
        $quote = $this->dbQuote($sql, $metaDataArray['order_pgw_id'], $metaDataArray['key'], $metaDataArray['value'], $metaDataArray['status'], $metaDataArray['created'], $metaDataArray['updated'], $metaDataArray['created_by'], $metaDataArray['updated_by']);
        return $this->dbQueryInsert($quote);
    }

    /**
     * get object value for a given order id and object key from pup_order_pgw_data
     * @method getPrePgwMetaData
     * @access public
     * @param integer $orderPgwId
     * @param string $objectKey
     * @return string
     */
    public function getPrePgwMetaData($orderPgwId, $objectKey) {
        $sql = 'SELECT object_value FROM pup_order_pgw_data  WHERE order_pgw_id = ?i AND object_key = ?s AND status = ?i';
        $sanitizedSql = $this->dbQuote($sql, $orderPgwId, $objectKey, 1);
        return $this->dbGetField($sanitizedSql, array(), self::SLAVE_NOCACHE);
    }

    /**
     * get object value for a given order id and object key from pup_prepayment_details_data
     * @method getPostPgwMetaData
     * @access public
     * @param integer $prepaymentId
     * @param string $objectKey
     * @return string
     */
    public function getPostPgwMetaData($prepaymentId, $objectKey) {
        $sql = 'SELECT value FROM pup_prepayment_details_data  WHERE prepayment_id = ?i AND `key` = ?s AND status = ?i';
        $sanitizedSql = $this->dbQuote($sql, $prepaymentId, $objectKey, 1);
        return $this->dbGetField($sanitizedSql, array(), self::SLAVE_NOCACHE);
    }

    /**
     * get payment pgw mapping data
     * @method getMappingData
     * @access public
     * @param integer $orderId
     * @param integer $paymentOptionId
     * @param integer $prepaymentId {optional}
     * @return integer
     */
    public function getMappingData($orderId, $paymentOptionId, $prepaymentId = 0) {
        $sql = 'SELECT order_pgw_auto_increment_id FROM pup_payment_pgw_mapping WHERE client_order_id  = ?i AND payment_option_id = ?i ?p LIMIT 1';
        $extraSql = '';
        if (!empty($prepaymentId)) {
            $extraSql = $this->dbQuote(' AND prepayment_details_auto_increment_id = ?i', $prepaymentId);
        } else {
            $extraSql = 'ORDER BY id DESC';
        }
        $sql = $this->dbQuote($sql, $orderId, $paymentOptionId, $extraSql);
        return $this->dbGetField($sql, array(), self::SLAVE_NOCACHE);
    }

    /**
     * get payment channel info of payment option
     * @method getPaymentChannelInfo
     * @access public
     * @param integer $paymentOptionId
     * @param integer $paymentId
     * @return array
     */
    public function getPaymentChannelInfo($paymentOptionId, $paymentId) {
        $sql = 'SELECT bank_code, product_code, sub_product_code FROM pup_payment_option_pgw WHERE payment_option_id = ?i AND payment_gateway_id = ?i AND status = ?s LIMIT 1';
        $sql = $this->dbQuote($sql, $paymentOptionId, $paymentId, 'A');
        return $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * Get order info from pup_order_pgw from order id
     * @method getOrderInfoFromOrderPgw
     * @access public
     * @param integer $orderId order id
     * @return array
     */
    public function getOrderInfoFromOrderPgw($orderId) {
        $sql = 'SELECT order_data FROM pup_order_pgw WHERE client_order_id = ?s ORDER BY id DESC LIMIT ?i';
        $sql = $this->dbQuote($sql, $orderId, 1);
        $data = $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
        $data = json_decode($data['order_data'], true);
        return $data;
    }

    /**
     * Get order info from pup_order_pgw from order pgw id
     * @method getOrderInfoFromOrderPgwById
     * @access public
     * @param integer $id
     * @return array
     */
    public function getOrderInfoFromOrderPgwById($id) {
        $sql = 'SELECT order_data FROM pup_order_pgw WHERE id = ?i ORDER BY id DESC LIMIT ?i';
        $sql = $this->dbQuote($sql, $id, 1);
        $data = $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
        $data = json_decode($data['order_data'], true);
        return $data;
    }

    /**
     * This method is used to get app_client_id details
     * @method getAppClientIdDetails
     * @param string $clientId client id
     * @access public
     * @return array
     */
    public function getAppClientIdDetails($clientId) {
        $sql = 'select c.*,cd.* from oauth_clients as c left join clues_oauth_clients_details as cd'
                . ' on c.client_id=cd.client_id and cd.status=1 where c.status=1 and c.client_id=?s limit 1';
        $query = $this->dbQuote($sql, $clientId);
        return $result = $this->dbGetRow($query, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to check insert data in pup_pgw_notify_mapping
     * @method insertPupNotifyMapping
     * @access public
     * @param array $orderDetails
     * @param string $formDataJson
     * @param integer $userId
     * @return integer
     */
    public function insertPupNotifyMapping($orderDetails, $formDataJson, $userId, $source) {
        $sql = 'INSERT INTO pup_pgw_notify_mapping (client_order_id, ppm_id, source, notify_data, status, created, created_by, updated, updated_by) VALUES (?s, ?i, ?i, ?s, ?i, ?s, ?i, ?s, ?i)';
        $sql = $this->dbQuote($sql, $orderDetails['client_order_id'], $orderDetails['id'], $source, $formDataJson, 1, date('Y:m:d H:i:s'), $userId, date('Y:m:d H:i:s'), $userId);
        return $this->dbQueryInsert($sql);
    }

    /**
     * This method is used to get card info from token
     * @method getCardInfoFromToken
     * @param string $token
     * @param integer $userId
     * @return array
     */
    public function getCardInfoFromToken($token, $userId) {
        $currentYear = (int)date("y");
        $currentMonth = (int)date("m");
        $sql = 'select card_token from pup_user_card_info where pup_tk = ?s and status = ?i and user_id = ?i and expiryYear >= ?i and if(expiryYear > ?i, 1, expiryMonth >= ?i)';
        $query = $this->dbQuote($sql, $token, 1, $userId, $currentYear, $currentYear, $currentMonth);
        return $this->dbGetRow($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to update card token
     * @method updateCardToken
     * @param string $cardToken
     * @param string $clientToken
     * @param string $userId
     * @return array
     */
    public function updateCardToken($cardToken, $clientToken, $userId) {
        $sql = 'UPDATE pup_user_card_info SET card_token = ?s, updated_by = ?i WHERE status = ?i AND pup_tk = ?s';
        $sql = $this->dbQuote($sql, $cardToken, $userId, 1, $clientToken);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to get card info from token
     * @method getDataForCurrentToken
     * @param string $pupToken
     * @return array
     */
    public function getDataForCurrentToken($pupToken) {
        $sql = 'select id from pup_user_card_info where pup_tk = ?s and status = ?i';
        $query = $this->dbQuote($sql, $pupToken, 1);
        return $this->dbGetRow($query, self::MASTER_NOCACHE);
    }
}
