<?php

namespace clues\model\pgwVerify;

use clues\model\CluesModel;
use clues\system\Logger;
use clues\system\DI;
use clues\system\util\StringOperations;
use clues\system\CluesRegistry;
use clues\model\pgwVerify\PgwVerifyConstants;

/**
 * This class is responsible for pgw verify base class
 * @class PGWVerify
 * @access public
 * @category Model class
 * @package model/pgw
 */
class PGWVerify extends CluesModel implements PgwVerifyConstants {

    /**
     * Payment service object
     * @access private
     * @var mixed 
     */
    private $_pgwVerifyDBO;

    /**
     * Class level cache
     * @access private
     * @var mixed 
     */
    private $_cache;

    /**
     * used to hold adminCronPath
     * @var integer
     */
    private $_adminCronPath;

    /**
     * Pgw object
     * @access private
     * @var mixed 
     */
    private $_pgwObj;

    /**
     * Payment params array
     * @access private
     * @var array 
     */
    private $_paymentParams;

    /**
     * used to hold paymentRuleData
     * @var array 
     */
    private $_ruleData = array();

    /**
     * Pgw mapping array
     * @access private
     * @var array 
     */
    private static $_pgwMappingArray = array();

    /**
     * PGW with same pgw_order_id mapping
     * @access private
     * @var array 
     */
    private static $_pgwSameTransactionIdMappingArray = array();

    /**
     * PGW with post promotion update
     * @access private
     * @var array 
     */
    private static $_pgwUpdatePostPromotionMapping = array();

    /**
     * payment object
     * @var object
     */
    protected $PaymentObj;

    /**
     * Set pgw names array
     * @access private
     * @var array
     */
    private $_paymentIdPaymentOptionIdPgwName = array();

    /*
     * Stores the file Name of executable script
     * @access private
     * @var array
     */
    private $_fileName;

    /**
     * prepayment id
     * @access private
     */
    private $_prepaymentId;

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @return void
     */
    public function __construct(array $data = array()) {
        parent::__construct();
        DI::Map('PgwVerifyDBOperations', 'clues\\model\\pgwVerify\\PgwVerifyDBOperations');
        $this->_pgwVerifyDBO = DI::Singleton('PgwVerifyDBOperations');
        DI::Map('PaymentObj', 'clues\\model\\payment\\Payment');
        $this->PaymentObj = DI::Singleton('PaymentObj');
        if (isset($data['adminCronPath']) && !empty($data['adminCronPath'])) {
            $this->_adminCronPath = $data['adminCronPath'];
        } else {
            $this->_adminCronPath = $this->CluesConfigObj->getKeyValue('admin_dir_path', 'payment', 'backend') . 'pgw_verify_cron.php';
        }
        $tempPath = explode('/', $this->_adminCronPath);
        $this->_fileName = end($tempPath);
        $pgwMappingJson = $this->CluesConfigObj->getKeyValue('pgw_verify_cron_mapping', 'payment', 'backend');
        self::$_pgwMappingArray = json_decode($pgwMappingJson, true);
    }

    /**
     * Verify each pending order (older ones also can be verified)
     * @method verifyEachPendingOrder
     * @access public
     * @param integer $orderId
     * @param integer $paymentId
     * @return boolean $verifyResult
     */
    public function verifyEachPendingOrder($orderId, $paymentId) {
        $verifyResult = false;
        $daysBeforeForImmediatePgwVerify = $this->CluesConfigObj->getKeyValue('days_before_for_immediate_pgw_verify', 'payment', 'api');
        $daysBeforeForImmediatePgwVerify = !empty($daysBeforeForImmediatePgwVerify) ? $daysBeforeForImmediatePgwVerify : 730;
        $initialTimeForImmediatePgwVerify = $this->CluesConfigObj->getKeyValue('initial_time_for_immediate_pgw_verify', 'payment', 'api');
        $initialTimeForImmediatePgwVerify = !empty($initialTimeForImmediatePgwVerify) ? $initialTimeForImmediatePgwVerify : 1568529000;
        $this->requestConsInputParams['immediate_pgw_verify']['days_before'] = $daysBeforeForImmediatePgwVerify;
        $this->requestConsInputParams['immediate_pgw_verify']['initial_time'] = time() - $initialTimeForImmediatePgwVerify;
        unset($daysBeforeForImmediatePgwVerify, $initialTimeForImmediatePgwVerify);
        $this->requestConsInputParams['immediate_pgw_verify']['final_time'] = time();
        $this->requestConsInputParams['immediate_pgw_verify']['debug'] = 0;
        $verifyResultArray = $this->verifyPendingTransactions($this->requestConsInputParams['immediate_pgw_verify']['initial_time'], $this->requestConsInputParams['immediate_pgw_verify']['final_time'], $paymentId, $orderId, $this->requestConsInputParams['immediate_pgw_verify']['debug']);
        $verifyResult = (isset($verifyResultArray[$orderId]) && !empty($verifyResultArray[$orderId])) ? $verifyResultArray[$orderId] : $verifyResult;
        return $verifyResult;
    }

    /**
     * Verify pending transactions base method
     * @method verifyPendingTransactions
     * @access public
     * @param integer $initialTime
     * @param integer $finalTime
     * @param integer $paymentId
     * @param integer $orderId
     * @param integer $debug
     * @return array $verifyResultForAllOrderIds
     */
    public function verifyPendingTransactions($initialTime, $finalTime, $paymentId, $orderId, $debug) {
        $this->_validatePaymentId($paymentId);
        $verifyResultForAllOrderIds = array();
        $daysBefore = $this->CluesConfigObj->getKeyValue('daysBeforePendingOrder', 'payment', 'backend');
        $daysBefore = (isset($daysBefore) && !empty($daysBefore)) ? $daysBefore : 7;
        $daysBefore = (isset($this->requestConsInputParams['immediate_pgw_verify']['days_before']) && !empty($this->requestConsInputParams['immediate_pgw_verify']['days_before'])) ? $this->requestConsInputParams['immediate_pgw_verify']['days_before'] : $daysBefore;
        $orderList = $this->_pgwVerifyDBO->getPendingOrders($initialTime, $finalTime, $paymentId, $orderId, $daysBefore);
        $isVerifyFailedData = $this->CluesConfigObj->getKeyValue('verifyFailedData', 'payment', 'backend');
        if (!empty($isVerifyFailedData) && $isVerifyFailedData) {
            $extra_data = array();
            $extra_data = $this->CluesConfigObj->getKeyValue('ExcludePaymentOptionFromPGWVerify', 'payment', 'backend');
            $extra_data = (isset($extra_data) && !empty($extra_data)) ? json_decode($extra_data, true) : array();
            $failedOrdersfromPGW = $this->picFailedOrdersOfPgw($initialTime, $finalTime, $paymentId, $orderId, $extra_data);
            $orderList = array_merge($orderList, $failedOrdersfromPGW);
        }
        if (!empty($orderList)) {
            $this->_initializePgwObj($paymentId);
            foreach ($orderList as $order) {
                $clientId = !empty($order['client_id']) ? $order['client_id'] : PgwVerifyConstants::DEFAULT_CLIENT_ID;
                $order['pgwName'] = $this->_getPgwName($order);
                $order['payment_params'] = $this->_setPaymentParams($paymentId, $order['client_id'], $order['merchant_params']);
                if (!isset($this->_ruleData[$clientId]) || empty($this->_ruleData[$clientId])) {
                    $this->_ruleData[$clientId] = $this->getRuleDetails($clientId);
                }
                $transactionVerified = $this->_checkSamePgwTransactionsid($order, $paymentId);
                $verifyResult = false;
                if (!$transactionVerified) {
                    $verifyResult = $this->_verifyPgwTransaction($order, $debug, $paymentId);
                }
                $verifyResultForAllOrderIds[$order['client_order_id']] = $verifyResult;
            }
        }
        return $verifyResultForAllOrderIds;
    }

    /**
     * Verify PGW Transaction
     * @method _verifyPgwTransaction
     * @access private
     * @param array $order
     * @param integer $debug
     * @param integer $paymentId
     * @return boolean $verifyStatus
     */
    private function _verifyPgwTransaction($order, $debug, $paymentId) {
        $verifyStatus = false;
        $pgTransaction = $this->_pgwObj->getPgwTransaction($order);
        $prepaymentData = $this->_pgwObj->createPrepaymentData($pgTransaction, $order);
        $isTransactionSuccess = $this->_insertPgTransaction($pgTransaction, $order, $prepaymentData, self::ADMIN_USER_ID);
        if ($isTransactionSuccess) {
            $isPaymentMarkSuccess = CluesRegistry::getObject(PgwVerifyConstants::PAYMENT_MARKED_SUCCESSFUL .'_'. $order['id']);
            if (empty($isPaymentMarkSuccess)) {
                $isTransactionSuccess = 0; //mark payment failed
            }
        }
        if ($isTransactionSuccess == 1) {
            $verifyStatus = true;
        }
        $responseNotifyArray = $this->prepareAndInsertNotifyData($order);
        return $verifyStatus;
    }

    /**
     * Initialize pgw object
     * @method _initializePgwObj
     * @access private
     * @param type $paymentId
     * @throws \Exception
     * @return void
     */
    private function _initializePgwObj($paymentId) {
        $pgwObj = $this->_mapPgwObject($paymentId);
        if (empty($pgwObj)) {
            throw new \Exception('Invalid Payment Gateway');
        }
        $this->_pgwObj = $pgwObj;
    }

    /**
     * Map pgw object
     * @method _mapPgwObject
     * @access private
     * @param integer $paymentId
     * @return mixed
     */
    private function _mapPgwObject($paymentId) {
        $pgwObj = null;
        if (isset(self::$_pgwMappingArray[$paymentId]) && !empty(self::$_pgwMappingArray[$paymentId])) {
            $additionalData = $this->_additionalPgwData($paymentId);
            $mapStr = 'clues\\model\\pgwVerify\\' . self::$_pgwMappingArray[$paymentId];
            DI::Map(self::$_pgwMappingArray[$paymentId], $mapStr);
            $pgwObj = DI::Singleton(self::$_pgwMappingArray[$paymentId], $additionalData);
            unset($additionalData, $mapStr);
        }
        return $pgwObj;
    }

    /**
     * pic Failed transactions of pgw
     * @method picFailedOrdersOfPgw
     * @access protected
     * @param integer $initialTime
     * @param integer $finalTime
     * @param integer $paymentId
     * @param integer $orderId
     * @param array extra_data
     * @return array
     */
    protected function picFailedOrdersOfPgw($initialTime, $finalTime, $paymentId, $orderId = 0, array $extra_data = array()) {
        return $this->_pgwVerifyDBO->picFailedOrdersOfPgw($initialTime, $finalTime, $paymentId, $orderId, $extra_data);
    }

    /**
     * Insert pg transaction
     * @method _insertPgTransaction
     * @param mixed $pgTransaction
     * @param array $order
     * @param array $prepaymentData
     * @param integer $userId
     * @return integer
     */
    private function _insertPgTransaction($pgTransaction, $order, $prepaymentData, $userId) {
        if (empty($pgTransaction)) {
            return;
        }
        $clientId = !empty($order['client_id']) ? $order['client_id'] : PgwVerifyConstants::DEFAULT_CLIENT_ID;
        $retryTimeGap = !empty($this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['time_gap']) ? $this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['time_gap'] : self::RETRY_TIME_GAP;
        $oldPrepaymentId = $this->_pgwVerifyDBO->getPrepaymentIdFromMapping($order['id']);
        $prepaymentDataValue = $this->_pgwObj->verifyPgTransaction($pgTransaction, array('payment_params' => $this->_paymentParams, 'amount' => $order['amount'], 'client_order_id' => $order['client_order_id'], 'pgw_order_id' => $order['pgw_order_id']));
        if (!empty($oldPrepaymentId)) {
            $this->_prepaymentId = $oldPrepaymentId;
            $currentPaymentStatus = $this->_pgwVerifyDBO->getPaymentStatusFromPrepaymentData($oldPrepaymentId);
        }
        if (empty($oldPrepaymentId) || (isset($currentPaymentStatus) && $currentPaymentStatus == 0 && $prepaymentDataValue == 1)) {
            $cluesPrepaymentId = $this->insertPostMetaData($order, $clientId);
            $this->_prepaymentId = $cluesPrepaymentId;
            if (!empty($oldPrepaymentId)) {
                $this->_pgwVerifyDBO->updateVerifiedTransactionInMapping($cluesPrepaymentId, $order['client_order_id'], $userId, $order['payment_id'], $order['id'], 3);
                $this->_pgwVerifyDBO->updateTransactionStatus($order['id'], 3);
            } else {
                if (!empty($prepaymentDataValue)) {
                    $this->_pgwVerifyDBO->updateVerifiedTransactionInMapping($cluesPrepaymentId, $order['client_order_id'], $userId, $order['payment_id'], $order['id'], 3);
                } else {
                    $this->_pgwVerifyDBO->updateVerifiedTransactionInMapping($cluesPrepaymentId, $order['client_order_id'], $userId, $order['payment_id'], $order['id'], 1);
                }
            }
            $this->_notifyRefund($cluesPrepaymentId, $order['client_order_id'], $order['payment_id']);
        } else if (!($prepaymentDataValue xor $currentPaymentStatus)) {
            $this->_pgwVerifyData($order);
        } else {
            $this->_prepaymentId = $cluesPrepaymentId;
            $metaData = $this->_getPostPaymentMetaData($cluesPrepaymentId, json_encode($prepaymentData['prepayment_dump']), $userId, self::PGW_VERIFY_RESPONSE_KEY);
            $this->_pgwVerifyDBO->savePostPaymentMetaData($metaData);
        }
        return $prepaymentDataValue;
    }

    /**
     * This method is used to notify refund about success from pgw
     * @method _notifyRefund
     * @access private
     * @param integer $cluesPrepaymentId
     * @param integer $scOrderId
     * @param integer $paymentId
     * @note this part has to be handled later on 
     * @return mixed
     */
    private function _notifyRefund($cluesPrepaymentId, $scOrderId, $paymentId) {
        return true;
        try {
            DI::Map('RefundUtil', 'clues\\model\\refund\\util\\RefundUtil');
            $RefundUtilObj = DI::Singleton('RefundUtil');
            return $RefundUtilObj->insertPrePaymentDetails($cluesPrepaymentId, $scOrderId, $paymentId);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Save post payment meta data
     * @method _getPostPaymentMetaData
     * @param integer $cluesPrepaymentId
     * @param integer $value
     * @param integer $userId
     * @param string $key key for prepaymentmeta data
     * @return array
     */
    private function _getPostPaymentMetaData($cluesPrepaymentId, $value, $userId, $key = 'payment_status') {
        $response = array();
        $response['prepayment_id'] = $cluesPrepaymentId;
        $response['key'] = $key;
        $response['value'] = $value;
        $response['status'] = 1;
        $response['created'] = $response['updated'] = time();
        $response['created_by'] = $response['updated_by'] = $userId;
        return $response;
    }

    /**
     * Validate payment id
     * @method _validatePaymentId
     * @access private
     * @param integer $paymentId
     * @throws \Exception
     * @return void
     */
    private function _validatePaymentId($paymentId) {
        if (!isset(self::$_pgwMappingArray[$paymentId]) || empty(self::$_pgwMappingArray[$paymentId])) {
            throw new \Exception('Invalid Payment Gateway');
        }
    }

    /**
     * this method is used get payment id's for cron
     * @method getAllSupportingPaymentId
     * @access public
     * @return array payment_ids
     */
    public function getAllSupportingPaymentId() {
        return array_keys(self::$_pgwMappingArray);
    }

    /**
     * this method is used to check payment id supports same txn or not
     * @method _checkSamePgwTransactionsid
     * @access private
     * @param array $order
     * @param integer $paymentId
     * @return boolean
     */
    private function _checkSamePgwTransactionsid($order, $paymentId) {
        $clientId = !empty($order['client_id']) ? $order['client_id'] : PgwVerifyConstants::DEFAULT_CLIENT_ID;
        if (!in_array($paymentId, self::$_pgwSameTransactionIdMappingArray)) {
            return false;
        }
        if ($this->_checkClientRuleAvailablity($clientId, self::PGW_VERIFY_RETRY_CHECK_RULE)) {
            $verifiedTransactions = $this->_pgwVerifyDBO->getTransactionsByOrderId($order['client_order_id'], $paymentId);
            if (empty($verifiedTransactions)) {
                return false;
            }
            if ($verifiedTransactions['status'] == 3) {
                $this->_pgwVerifyDBO->updateVerifiedTransactionInMapping($verifiedTransactions['prepayment_details_auto_increment_id'], $order['client_order_id'], self::ADMIN_USER_ID, $paymentId, $order['id'], 3);
                return true;
            }
            if ($verifiedTransactions['status'] == 1) {
                $pgTransaction = $this->_pgwObj->getPgwTransaction($order);
                $prepaymentData = $this->_pgwObj->createPrepaymentData($pgTransaction, $order);
                $prepaymentDataValue = $this->_pgwObj->verifyPgTransaction($pgTransaction, array('payment_params' => $this->_paymentParams, 'amount' => $order['amount'], 'client_order_id' => $order['client_order_id'], 'pgw_order_id' => $order['pgw_order_id']));
                if (!$prepaymentDataValue && !$verifiedTransactions['value']) {
                    return false;
                } else {
                    $this->_pgwVerifyDBO->updateVerifiedTransactionInMapping($verifiedTransactions['prepayment_details_auto_increment_id'], $order['client_order_id'], self::ADMIN_USER_ID, $paymentId, $order['id'], 3);
                    $this->_pgwVerifyDBO->updateTransactionStatus($verifiedTransactions['id'], 3);
                    return true;
                }
            } else {
                return false;
            }
        }
        return false;
    }

    /**
     * method will open parallel crons
     * @method  startVerifyTransaction
     * @access public
     * @return void
     */
    public function startVerifyTransaction() {
        foreach (self::$_pgwMappingArray as $paymentId => $value) {
            if (!empty($paymentId)) {
                $this->_shellExec($this->_adminCronPath, 'verify_pgw', $paymentId);
            } else {
                break;
            }
        }
    }

    /**
     * this method is used for executing shell command
     * @method _shellExec
     * @access private
     * @param string $path path of php file to execute
     * @param string $param1 cron param1
     * @param integer $param2 cron param2
     * @return void
     */
    private function _shellExec($path, $param1, $param2) {
        $fileName = $this->_fileName;
        $fileName .= '_' . $param1 . '_' . $param2 . '.lock';
        shell_exec("/usr/bin/flock -w 0  /var/tmp/$fileName php $path $param1 $param2  > /dev/null 2>&1 &");
    }

    /**
     * this method is used to pass any additional data to pgw
     * @method _additionalPgwData
     * @access private
     * @param integer $paymentId
     * @return array 
     */
    private function _additionalPgwData($paymentId) {
        $additinalData = array();
        if (isset($paymentId) && !empty($paymentId)) {
            switch ($paymentId) {
                case self::Pgw2C2P_PAYMENT_ID:
                    $additinalData = array();
                    break;
                default :
                    break;
            }
        }
        return $additinalData;
    }

    /**
     * This method is used to set payment params
     * @method _setPaymentParams
     * @access private
     * @param integer $paymentId payment id
     * @param string $clientId
     * @param string  $orderReqMerchantParam
     * @return void
     */
    private function _setPaymentParams($paymentId, $clientId, $orderReqMerchantParam) {
        $temp = array();
        if (!empty($orderReqMerchantParam)) {
            $temp = json_decode($orderReqMerchantParam, true);
        }
        if (empty($temp) || !is_array($temp)) {
            $temp = $this->PaymentObj->getPaymentInfoByPaymentIdAndCID($paymentId, $clientId);
        }
        $this->_paymentParams = $temp;
        return $this->_paymentParams;
    }

    /**
     * This method is used to get pgw data of all payment options
     * @method _getPgwData
     * @access private
     * @return array
     */
    private function _getPgwData() {
        $returnData = array();
        $returnData['payment_pgw_names'] = $this->_pgwVerifyDBO->getPgwName();
        $returnData['payment_option_pgw_names'] = $this->_pgwVerifyDBO->getAllPaymentOptionsName();
        return $returnData;
    }

    /**
     * This method is used to get pgwName data
     * @method _getPgwName
     * @access private
     * @param array $orderData
     * @return string
     */
    private function _getPgwName(array $orderData) {
        $pgwName = '';
        if (isset($orderData['payment_id']) && !empty($orderData['payment_id']) &&
                isset($orderData['payment_option_id']) && !empty($orderData['payment_option_id'])) {
            $nameKey = $orderData['payment_id'] . '_' . $orderData['payment_option_id'];
            if (isset($this->_paymentIdPaymentOptionIdPgwName[$nameKey]) && !empty($this->_paymentIdPaymentOptionIdPgwName[$nameKey])) {
                $pgwName = $this->_paymentIdPaymentOptionIdPgwName[$nameKey];
            } else {
                $paymentData = $this->_getPgwData();
                if (!empty($paymentData['payment_pgw_names'][$orderData['payment_id']]['name']) && !empty($paymentData['payment_option_pgw_names'][$orderData['payment_option_id']]['name'])) {
                    $pgwName = $this->_paymentIdPaymentOptionIdPgwName[$nameKey] = $paymentData['payment_pgw_names'][$orderData['payment_id']]['name'] . '_' . $paymentData['payment_option_pgw_names'][$orderData['payment_option_id']]['name'];
                }
            }
        }
        return $pgwName;
    }

    /**
     * It provides rule details
     * @method getRuleDetails
     * @access public
     * @param string $clientId clientId
     * @return array $ruleData
     */
    public function getRuleDetails($clientId) {
        $clientId = !empty($clientId) ? $clientId : PgwVerifyConstants::DEFAULT_CLIENT_ID;
        if (!isset($this->_ruleData[$clientId]) || empty($this->_ruleData[$clientId]) || !is_array($this->_ruleData[$clientId])) {
            $rulesFromDb = $this->_pgwVerifyDBO->getPgwVerifyRulesForClientId($clientId, self::PAYMENT_MODULE_ID);
            if ((!isset($rulesFromDb) || empty($rulesFromDb) || !is_array($rulesFromDb)) && strtolower($clientId) != PgwVerifyConstants::DEFAULT_CLIENT_ID) { // no client rules found in db
                if (isset($this->_ruleData[PgwVerifyConstants::DEFAULT_CLIENT_ID]) && !empty($this->_ruleData[PgwVerifyConstants::DEFAULT_CLIENT_ID])) { // check if rules exist for default client, return if found
                    return $this->_ruleData[PgwVerifyConstants::DEFAULT_CLIENT_ID];
                } else { // fetch rules for default client from db
                    $rulesFromDb = $this->_pgwVerifyDBO->getPgwVerifyRulesForClientId(PgwVerifyConstants::DEFAULT_CLIENT_ID, self::PAYMENT_MODULE_ID);
                }
            }
            if (isset($rulesFromDb) && !empty($rulesFromDb)) { // rules found in db for given client id or default client id
                foreach ($rulesFromDb as $rule => $ruleInfo) {
                    $rulesFromDb[$rule]['rule_configurations'] = (isset($ruleInfo['rule_configurations']) && !empty($ruleInfo['rule_configurations'])) ? StringOperations::jsonDecodeOrUnserialize($ruleInfo['rule_configurations']) : array();
                }
            } else { // no rules found for either given client id or default client id
                return array();
            }
            return $rulesFromDb;
        }
        return $this->_ruleData[$clientId];
    }

    /**
     * check rule available or not 
     * @method _checkClientRuleAvailablity
     * @access private
     * @param string $ruleName rule name
     * @param string $clientId client id
     * @return integer 
     */
    private function _checkClientRuleAvailablity($clientId, $ruleName) {
        if (isset($this->_ruleData[$clientId][$ruleName]) && !empty($this->_ruleData[$clientId][$ruleName])) {
            return $this->_ruleData[$clientId][$ruleName]['rule_status'];
        }
        return 0;
    }

    /**
     * Retry and backfill data in tables
     * @method _pgwVerifyData
     * @access private
     * @param array $order
     * @return boolean
     */
    private function _pgwVerifyData($order) {
        $clientId = !empty($order['client_id']) ? $order['client_id'] : PgwVerifyConstants::DEFAULT_CLIENT_ID;
        $count = $order['retry_count'];
        $retryCountLimit = !empty($this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['count']) ? $this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['count'] : self::RETRY_COUNT_VALUE;
        $retryTimeGap = !empty($this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['time_gap']) ? $this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['time_gap'] : self::RETRY_TIME_GAP;
        $status = $this->_pgwVerifyDBO->fetchPaymentPgwMappingStatus($order['id']);
        $flag = $this->_pgwVerifyDBO->getPaymentFlagByPrepaymentId($order['prepayment_details_auto_increment_id']);
        $this->_prepaymentId = $order['prepayment_details_auto_increment_id'];
        $check = $this->_pgwObj->checkVerifyValidationByPPDFlag($flag);
        if ($count < $retryCountLimit && empty($check)) {
            $this->_pgwVerifyDBO->updateVerifiedTransactionInMapping($order['prepayment_details_auto_increment_id'], $order['client_order_id'], self::ADMIN_USER_ID, $paymentId, $order['id'], 3);
            $this->_pgwVerifyDBO->updateTransactionStatus($order['id'], 3);
        } elseif ($count < $retryCountLimit && $status != 3) {
            $cluesPrepaymentId = $this->insertPostMetaData($order, $clientId);
        } else {
            $this->_pgwVerifyDBO->updateVerifiedTransactionInMapping($order['prepayment_details_auto_increment_id'], $order['client_order_id'], self::ADMIN_USER_ID, $paymentId, $order['id'], 3);
            $this->_pgwVerifyDBO->updateTransactionStatus($order['id'], 3);
        }
        return true;
    }

    /**
     * Retry and backfill data in tables
     * @method insertPostMetaData
     * @access protected
     * @param array $order
     * @param $clientId
     * @return boolean
     */
    protected function insertPostMetaData($order, $clientId) {
        $retryCountLimit = !empty($this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['count']) ? $this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['count'] : self::RETRY_COUNT_VALUE;
        $retryTimeGap = !empty($this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['time_gap']) ? $this->_ruleData[$clientId][self::PGW_VERIFY_RETRY_CHECK_RULE]['rule_configurations'][$order['payment_id']]['time_gap'] : self::RETRY_TIME_GAP;
        $oldPrepaymentId = $this->_pgwVerifyDBO->getPrepaymentIdFromMapping($order['id']);
        $pgTransaction = $this->_pgwObj->getPgwTransaction($order);
        $prepaymentData = $this->_pgwObj->createPrepaymentData($pgTransaction, $order);
        $prepaymentDataValue = $this->_pgwObj->verifyPgTransaction($pgTransaction, array('payment_params' => $this->_paymentParams, 'amount' => $order['amount'], 'client_order_id' => $order['client_order_id'], 'pgw_order_id' => $order['pgw_order_id']));
        $pgwRetryReqId = $this->_pgwVerifyDBO->insertCluesOrderPgw($order);
        $ipData = $this->RequestObj->getIp();
        $metaDataArray = ['order_pgw_id' => $pgwRetryReqId, 'key' => 'request_ip', 'value' => $ipData, 'status' => 1];
        $metaDataArray['created'] = $metaDataArray['updated'] = time();
        $metaDataArray['created_by'] = $metaDataArray['updated_by'] = PgwVerifyConstants::ADMIN_USER_ID;
        $this->_pgwVerifyDBO->addPrePgwMetaData($metaDataArray);
        unset($metaDataArray);
        $cluesPrepaymentId = $this->_pgwVerifyDBO->insertCluesPrepaymentDetails($prepaymentData['prepayment_data']['direcpayreferenceid'], $order['client_order_id'], $prepaymentData['prepayment_data']['flag'], json_encode($prepaymentData['prepayment_dump']), $prepaymentData['prepayment_data']['amount'], $prepaymentData['prepayment_data']['payment_gateway']);
        $this->_prepaymentId = $cluesPrepaymentId;
        $pgwData = array(self::PGW_ORDER_REQ_ID => $pgwRetryReqId, self::PAYMENT_STATUS_KEY => $prepaymentDataValue, self::RETRY_COUNT => $order['retry_count'] + 1, self::OLD_PREPAYMENT_ID_KEY => $oldPrepaymentId, self::PGW_VERIFY_NEXT_PICKUP => $retryTimeGap + time());
        $metaData = $this->_pgwObj->getPgwMetaData($pgTransaction);
        if (!empty($metaData)) {
            $pgwData = array_merge($pgwData, $metaData);
        }
        $pgwData['response_ip'] = $ipData;
        foreach ($pgwData as $key => $value) {
            if (empty($oldPrepaymentId) && $key == self::OLD_PREPAYMENT_ID_KEY) {
                continue;
            }
            if (!empty($prepaymentDataValue) && $key == self::PGW_VERIFY_NEXT_PICKUP) {
                continue;
            }
            $metaData = $this->_getPostPaymentMetaData($cluesPrepaymentId, $value, self::ADMIN_USER_ID, $key);
            $insertMetaDataId = $this->_pgwVerifyDBO->savePostPaymentMetaData($metaData);
            if (($key == PgwVerifyConstants::PAYMENT_STATUS_KEY) && ($value == PgwVerifyConstants::PAYMENT_SUCCESS_STATUS) && !empty($insertMetaDataId) && !empty($order['id'])) {
                CluesRegistry::addObject(PgwVerifyConstants::PAYMENT_MARKED_SUCCESSFUL . '_' . $order['id'], PgwVerifyConstants::PAYMENT_SUCCESS_STATUS);
            }
        }
        $this->_pgwVerifyDBO->updatePgwVerifyDetails($order['id'], $order['retry_count'] + 1, self::ADMIN_USER_ID, $cluesPrepaymentId);
        return $cluesPrepaymentId;
    }

    /**
     * this method is used to prepare and insert notify data
     * @method prepareAndInsertNotifyData
     * @access protected
     * @param array $orderDetails ['orderId' => <oid>, 'toStatus' => <status>, 'paymentOptionId' => <opt_id>, 'paymentId' => <payment_id>]
     * @return string
     */
    protected function prepareAndInsertNotifyData(array $orderDetails) {
        $formDataJson = '';
        $notifyId = 0;
        DI::Map('Payment', 'clues\\model\\payment\\Payment');
        $PaymentObj = DI::Singleton('Payment');
        $paymentDetails = $PaymentObj->getPaymentDetailsByOrderId($orderDetails['client_order_id']);
        if (!empty($paymentDetails) && is_array($paymentDetails)) {
            $keys = PgwVerifyConstants::NOTIFY_DATA_KEYS;
            $paymentDetails = array_pop($paymentDetails);
            $tempDetails = [];
            foreach ($keys as $key) {
                if (!isset($paymentDetails[$key])) {
                    $tempDetails[$key] = '';
                } else {
                    $tempDetails[$key] = $paymentDetails[$key];
                    unset($paymentDetails[$key]);
                }
            }
            if (strtoupper($orderDetails['toStatus']) == 'F') {
                $tempDetails['payment_status'] = 0;
                $tempDetails['flag'] = 'failure';
                $tempDetails['udf_payment_type'] = $tempDetails['pgw_txn_id'] = $tempDetails['card_number'] = $tempDetails['pgw_order_id'] = $tempDetails['prepayment_details_auto_increment_id'] = $tempDetails['direcpayreferenceid'] = '';
            }
            $paymentDetails = $tempDetails;
        } else {
            $paymentTypeDetails = $this->paymentService->getPaymentTypeDetails($orderDetails['payment_option_id']);
            $pgwName = $this->paymentService->getPgwNameFromPaymentId($orderDetails['payment_id']);
            $paymentDetails = array(
                'order_id' => $orderDetails['client_order_id'],
                'client_order_id' => $orderDetails['client_order_id'],
                'payment_id' => $orderDetails['payment_id'],
                'payment_option_id' => $orderDetails['payment_option_id'],
                'payment_status' => 0,
                'payment_status_info' => 'Payment Not Received',
                'flag' => 'failure',
                'emi_id' => 0,
                'emi_name' => NULL,
                'status' => 1,
                'pgw_type' => $paymentTypeDetails['pgw_type'],
                'pgw_option_name' => $paymentTypeDetails['pgw_option_name'],
                'payment_type_id' => $paymentTypeDetails['payment_type_id'],
                'pgw' => $pgwName['pgw'],
            );
            $paymentDetails['created_by'] = $paymentDetails['updated_by'] = PgwVerifyConstants::ADMIN_USER_ID;
            $paymentDetails['created_timestamp'] = $paymentDetails['updated'] = time();
            $paymentDetails['created'] = date('d M Y H:i:s', time());
            foreach ($keys as $key) {
                if (!isset($paymentDetails[$key])) {
                    $paymentDetails[$key] = '';
                }
            }
        }
        $ipDetails = $this->_getIpDetailsForOrderId($paymentDetails['order_id'], $paymentDetails['payment_option_id']);
        $paymentDetails['request_ip'] = $ipDetails['request_ip'];
        $paymentDetails['response_ip'] = $ipDetails['response_ip'];
        $formDataJson = json_encode($paymentDetails);
        $notifyId = $this->_pgwVerifyDBO->insertPupNotifyMapping($orderDetails, $formDataJson, PgwVerifyConstants::ADMIN_USER_ID, PgwVerifyConstants::ATOM_VERIFY_SOURCE);
        return ['notifyData' => $formDataJson, 'notifyId' => $notifyId];
    }

    /**
     * get ip details for order id
     * @method _getIpDetailsForOrderId
     * @access private
     * @param integer $orderId
     * @param integer $paymentOptionId
     * @param integer $prepaymentId | OPTIONAL
     * @return array $ip ['request_ip' => <ip_addr>, 'response_ip' => <ip_addr>]
     */
    private function _getIpDetailsForOrderId($orderId, $paymentOptionId, $prepaymentId = 0) {
        $ip = ['request_ip' => '', 'response_ip' => ''];
        if (empty($prepaymentId) && isset($this->_prepaymentId) && !empty($this->_prepaymentId)) { // handling prepayment id for call from refund controller | ADMIN
            $prepaymentId = $this->_prepaymentId;
        }
        $orderPgwId = $this->_pgwVerifyDBO->getMappingData($orderId, $paymentOptionId, $prepaymentId);
        if (!empty($orderPgwId)) {
            $ip['request_ip'] = $this->_pgwVerifyDBO->getPrePgwMetaData($orderPgwId, 'request_ip');
        }
        if (!empty($prepaymentId)) {
            $ip['response_ip'] = $this->_pgwVerifyDBO->getPostPgwMetaData($prepaymentId, 'response_ip');
        } else {
            $ip['response_ip'] = $this->RequestObj->getIp();
        }
        return $ip;
    }

    /**
     * This method is used to hit curl
     * @method _curlRequest
     * @access private
     * @param integer $
     * @param string $url url
     * @param array $post_data data
     * @param string $errorPrefix
     * @return array response
     */
    private function _curlRequest($orderId, $url, array $post_data = array(), $errorPrefix = 'VERIFY_CRON_') {
        $timeoutTimeSeconds = $this->CluesConfigObj->getKeyValue('tiny_url_curl_timeout', 'payment', 'api');
        if (empty($timeoutTimeSeconds)) {
            $timeoutTimeSeconds = 30;
        }
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeoutTimeSeconds);
        if (!empty($post_data)) {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $post_data);
        }
        $response = curl_exec($ch);
        $curl_errno = curl_errno($ch);
        $curlError = curl_error($ch);
        $res = json_decode($response, true);
        $res['curl_error'] = 0;
        if ($curl_errno > 0) {
            $logData = array();
            $logData['field1'] = 'order_id:' . $orderId;
            $logData['field2'] = $url;
            $logData['error_name'] = $errorPrefix . 'CURL_TIMEOUT';
            $additionalInfo = array();
            $additionalInfo['input_params'] = $curlArray;
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['domain'] = PgwVerifyConstants::DOMAIN_NAME;
            $additionalInfo['module'] = 'pgw_verify_cron';
            $additionalInfo['level'] = Logger::ERROR;
            $logData['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwVerifyConstants::DOMAIN_NAME, 'pgw_verify_cron_curl_timeout', $logData, Logger::ERROR);
            $res['curl_error'] = 1;
        }
        if (!empty($curlError)) {
            $exception_data = array();
            $exception_data['field1'] = 'order_id:' . $orderId;
            $exception_data['field2'] = 'curl_response:' . $response;
            $exception_data['error_name'] = $errorPrefix . 'CURL_ERROR';
            $additionalInfo = array();
            $additionalInfo['input_params'] = $post_data;
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['domain'] = PgwVerifyConstants::DOMAIN_NAME;
            $additionalInfo['module'] = 'pgw_verify_cron';
            $additionalInfo['level'] = Logger::ERROR;
            $exception_data['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwVerifyConstants::DOMAIN_NAME, 'pgw_verify_cron_curl_error', $exception_data, Logger::ERROR);
            $res['curl_error'] = 1;
        }
        return $res;
    }

    /**
     * This method is used to update the post payment metadata in clues_payment_pgw_mapping and clues_prepayment_details_data table for retry logic
     * @method _updatePaymentMetaData
     * @access private
     * @param array $orderInfo
     * @param string $momoeOrderId
     * @return void
     */
    private function _updatePaymentMetaData($orderInfo, $momoeOrderId, $paymentId, $metaDataValues, $prepaymentPrimaryId) {
        $momoeOrderData = $this->_pgwVerifyDBO->checkMomoeOrderPgwMapping($momoeOrderId, $paymentId);
        $oldPrepaymentId = (isset($momoeOrderData['prepayment_id']) && !empty($momoeOrderData['prepayment_id'])) ? $momoeOrderData['prepayment_id'] : 0;
        $retryCount = (isset($momoeOrderData['retry_count']) && !empty($momoeOrderData['retry_count'])) ? $momoeOrderData['retry_count'] : 0;
        $momoeOrderexist = !empty($momoeOrderData) ? true : false;
        $orderPgwPrimaryId = $this->_pgwVerifyDBO->getCluesOrderIdForMomoeOrder($orderInfo['order_id']);
        if ($momoeOrderexist == true) {
            $this->_pgwVerifyDBO->updateOrderPgwPrimaryKeyByPgwOrderId($orderPgwPrimaryId, $momoeOrderId, $orderInfo['user_id'], $paymentId, $prepaymentPrimaryId);
        } else {
            $mappingCreated = true;
            $pgwData = array('orderId' => $orderInfo['order_id'], 'pgwOrderId' => $momoeOrderId, 'orderPgwPrimaryId' => $orderPgwPrimaryId, 'prepaymentPrimaryId' => $prepaymentPrimaryId, 'paymentId' => $paymentId, 'paymentOptionId' => $orderInfo['payment_option_id'], 'emiId' => 0, 'userId' => $orderInfo['user_id']);
            $this->_pgwVerifyDBO->insertPgwOrderMappingForRetry($pgwData);
        }
        $metaDataValues[PgwVerifyConstants::OLD_PREPAYMENT_ID] = $oldPrepaymentId;
        $metaDataValues[PgwVerifyConstants::RETRY_COUNT] = $retryCount + 1;
        $metaDataValues[PgwVerifyConstants::PGW_VERIFY_NEXT_PICKUP] = PgwVerifyConstants::RETRY_TIME_GAP + time();
        if (empty($mappingCreated)) {
            $this->_pgwVerifyDBO->updateVerifyRetryCount($momoeOrderId, $prepaymentPrimaryId, $metaDataValues[PgwVerifyConstants::RETRY_COUNT]);
        }
        $this->_checkAndAddPrepaymentIdMetaData($orderInfo, $metaDataValues, $prepaymentPrimaryId, array(PgwVerifyConstants::PAYMENT_STATUS_KEY, PgwVerifyConstants::PGW_TXN_ID, PgwVerifyConstants::PG_FORWARDED_NAME, PgwVerifyConstants::CARD_NUMBER, PgwVerifyConstants::PG_CARD_TYPE, PgwVerifyConstants::PG_CARD_MODE, PgwVerifyConstants::PG_IS_SAVED_CARD, PgwVerifyConstants::PG_BANK_CODE, PgwVerifyConstants::PG_BANK_REF_ID, PgwVerifyConstants::PGW_VERIFY_NEXT_PICKUP, PgwVerifyConstants::RETRY_COUNT, PgwVerifyConstants::OLD_PREPAYMENT_ID,
            PgwVerifyConstants::UPI_VPA));
    }

    /**
     * This method is used to check and add momoe pgw txn meta data
     * @method _checkAndAddPrepaymentIdMetaData
     * @access private
     * @param array $orderInfo order info
     * @param array $metaDataValues
     * @param integer $prepaymentPrimaryId
     * @param array $checkKeys
     * @return void
     */
    private function _checkAndAddPrepaymentIdMetaData(array $orderInfo, array $metaDataValues, $prepaymentPrimaryId, array $checkKeys) {
        if (!empty($checkKeys)) {
            foreach ($checkKeys as $metaKey) {
                $prepaymentIdExists = false;
                if (array_key_exists($metaKey, $metaDataValues)) {
                    if (empty($metaDataValues[PgwVerifyConstants::OLD_PREPAYMENT_ID]) && $metaKey == PgwVerifyConstants::OLD_PREPAYMENT_ID) {
                        continue;
                    }
                    if (!empty($metaDataValues[PgwVerifyConstants::PAYMENT_STATUS_KEY]) && $metaKey == PgwVerifyConstants::PGW_VERIFY_NEXT_PICKUP) {
                        continue;
                    }
                    $prepaymentIdExists = $this->_pgwVerifyDBO->checkPrepaymentIdMetaData($prepaymentPrimaryId, $metaKey);
                    if ($prepaymentIdExists == true) {
                        $this->_pgwVerifyDBO->updatePostPaymentMetaData($metaDataValues[$metaKey], $prepaymentPrimaryId, $orderInfo['user_id'], $metaKey);
                    } else {
                        $metaDataArray = array('prepayment_id' => $prepaymentPrimaryId, 'key' => $metaKey, 'value' => $metaDataValues[$metaKey], 'status' => 1, 'created' => time(), 'updated' => time(), 'created_by' => $orderInfo['user_id'], 'updated_by' => $orderInfo['user_id']);
                        $this->_pgwVerifyDBO->savePostPaymentMetaData($metaDataArray);
                    }
                }
            }
        }
    }
}
