<?php

namespace clues\model\payment;

use clues\system\DBOperations;
use clues\model\payment\PaymentConstants;
use clues\system\util\StringOperations;
use clues\system\CluesConstants;
use Registry;

/**
 * This class is used to perform all CRUD operations
 * @class PaymentDbData
 * @access public
 * @category Model datasource
 * @package model/payment
 */
class PaymentDbData extends DBOperations {

    /**
     * input params 
     * @var constant
     */
    protected $inputParams;

    /**
     * Class Constructor
     * @method __construct
     * @access public
     * @return void
     */
    public function __construct($inputParams = array()) {
        $this->inputParams = $inputParams;
        parent::__construct();
    }

    /**
     * This method is used to check payment status
     * @method checkPaymentStatus
     * @access public
     * @param string $token
     * @return integer order id
     */
    public function checkPaymentStatus($token) {
        $sql = 'SELECT client_order_id FROM pup_payment_service_data WHERE token = ?s AND expirestatus = ?i';
        $sanitizeSql = $this->dbQuote($sql, $token, 1);
        return $this->dbGetField($sanitizeSql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to insert url details
     * @method insertUrlDetails
     * @access public
     * @param integer $orderId
     * @param string $longUrl
     * @param string $shortUrl
     * @param string $expireTime
     * @param integer $retry
     * @param string $urlGenerationEvent
     * @param integer $userId
     * @param string $token
     * @param string $source
     * @param string $extraFields
     * @return void
     */
    public function insertUrlDetails($orderId, $longUrl, $shortUrl, $expireTime, $retry = 0, $urlGenerationEvent, $userId, $token, $source, $extraFields) {
        $query = 'UPDATE clues_url_details SET status = ?i, updated_at = ?i WHERE object_id = ?i AND url_generation_event = ?s AND type = ?i';
        $updateSql = $this->dbQuote($query, 0, time(), $orderId, $urlGenerationEvent, 1);
        $this->dbQueryUpdate($updateSql);
        $insertQuery = 'INSERT INTO clues_url_details (object_id,type,short_url,original_url,expire_time,retry,url_generation_event,status, added_at, updated_at, created_by, updated_by, token, source, extra_fields) VALUES (?i, 1 ,?s , ?s, ?s, ?i, ?s, 1, ?i, ?i, ?i, ?i, ?s, ?s, ?s)';
        $sql = $this->dbQuote($insertQuery, $orderId, $shortUrl, $longUrl, $expireTime, $retry, $urlGenerationEvent, time(), time(), $userId, $userId, $token, $source, $extraFields);
        $this->dbQueryInsert($sql);
    }

    /**
     * This method is used to update url details
     * @method updateUrlDetails
     * @access public
     * @param integer $orderId
     * @param integer $count
     * @return void
     */
    public function updateUrlDetails($orderId, $count) {
        $query = 'UPDATE clues_url_details SET retry = ?i WHERE object_id = ?i';
        $sql = $this->dbQuote($query, $count, $orderId);
        $this->dbQueryUpdate($sql);
    }

    /**
     * Rto statuses from clues_status_typess
     * @method rtoStatuses
     * @access public
     * @return array all rto statuses
     */
    public function rtoStatuses() {
        $sql = "SELECT status FROM clues_status_types WHERE status_group=?s";
        $sql = $this->dbQuote($sql, 'X');
        return $this->dbGetArray($sql);
    }

    /**
     * Get category ids of product
     * @method getProductCategoryIds
     * @access public
     * @param array $productList List of product ids
     * @return array Product category ids
     */
    public function getProductCategoryIds($productList) {
        $query = 'SELECT distinct pc.category_id,c.id_path FROM cscart_products_categories AS pc INNER JOIN '
                . 'cscart_categories as c on pc.category_id=c.category_id WHERE pc.link_type IN (?a) AND pc.product_id in (?n)';
        $sql = $this->dbQuote($query, array('M', 'A'), $productList);
        $data = $this->dbGetArray($sql, array(), self::MASTER_NOCACHE);
        $allCatIds = array();
        if (!empty($data)) {
            foreach ($data as $value) {
                $temp = array();
                $temp = explode('/', $value['id_path']);
                foreach ($temp as $v) {
                    $allCatIds[$v] = $v;
                }
            }
        }
        return $allCatIds;
    }

    /**
     * This method is used to get rto risk count by email id, shipping phone and visitor id
     * @method getRTOLevelByUserData
     * @access public
     * @param string $email email id
     * @param string $mobile shipping phone
     * @param string $visitorId visitor id
     * @return array 
     */
    public function getRTOLevelByUserData($email = '', $mobile = '', $visitorId = '') {
        $whereArray = $response = array();
        if (!empty($email)) {
            $whereArray[] = $this->dbQuote('(crrud.object_value=?s AND crrud.object_key=1)', $email);
        }
        if (!empty($mobile)) {
            $whereArray[] = $this->dbQuote('(crrud.object_value=?s AND crrud.object_key=2)', $mobile);
        }
        if (!empty($visitorId)) {
            $whereArray[] = $this->dbQuote('(crrud.object_value=?s AND crrud.object_key=3)', $visitorId);
        }
        if (!empty($whereArray)) {
            $where = implode(' OR ', $whereArray);
            $query = "SELECT crrud.object_key, crrud.object_value, crrud.risk_level, CASE WHEN crrud.object_key = 1 THEN 'Email' WHEN crrud.object_key = 2 THEN 'Mobile' ELSE 'Visitor Id' END AS object_type_description FROM clues_rto_risk_user_details AS crrud INNER JOIN clues_rto_risk_level AS crrl ON crrl.clues_rto_risk_level_id=crrud.risk_level WHERE (?p) AND crrl.status=1 AND crrl.is_active=1 AND crrud.status=1 ORDER BY crrud.risk_level DESC LIMIT 1";
            $query = $this->dbQuote($query, $where);
            $response = $this->dbGetRow($query);
        }
        return $response;
    }

    /**
     * This method is used to payment options data
     * @method getPaymentOptions
     * @access public
     * @param string $paymentTypeId payment type id
     * @param integer $platform platfom i.e D/W/A/M/I
     * @param integer $environment
     * @param array $requestData
     * @return array
     */
    public function getPaymentOptions($paymentTypeId, $platform, $environment = 0, array $requestData = array()) {
        $paymentTypeId = (int) $paymentTypeId;
        //for old legacy
        if (empty($requestData['clientId'])) {
            $requestData['clientId'] = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $sql = 'SELECT cpo.payment_type_id, cpo.name, cpo.description, cpo.image_url,cpo.m_image_url,cpo.position, cpop.payment_option_pgw_id,cpop.payment_option_id, cpop.payment_gateway_id,cpop.bank_code, cpop.status,cpo.flow_type, group_concat(cpop.payment_gateway_id) as served_by FROM pup_payment_options cpo 
            join pup_payment_option_pgw cpop on (cpo.payment_option_id=cpop.payment_option_id) 
            join pup_payments cp on (cpop.payment_gateway_id=cp.payment_id) 
            left join clues_envionment_platform_dependencies cep on cpo.payment_option_id=cep.object_id and cep.object_type= 0 and cep.platform= ?i
            and cep.environment= ?i WHERE cpo.payment_type_id = ?i and cp.`status` = ?s and cpop.`status` = ?s and cpop.priority = ?i and cep.object_id is null and cp.client_id=?s';
        $query = $this->dbQuote($sql, $platform, $environment, $paymentTypeId, 'A', 'A', 1, $requestData['clientId']);
        $query = !empty($requestData) && isset($requestData['flow_type']) && !empty($requestData['flow_type']) ? $query . ' and case when cpo.payment_option_id in (162,163) then 1=1 else cpo.status=?s end group by cpo.payment_option_id order by position asc' : $query . 'and cpo.status=?s group by cpo.payment_option_id order by position asc';
        $query = $this->dbQuote($query, 'A');
        return $this->dbGetArray($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to emi options data
     * @method getEMIOptions
     * @access public
     * @param integer $paymentTypeId payment type id
     * @param string $platform platfom i.e D/W/A/M/I
     * @param integer $environment
     * @param array $requestData
     * @return array
     */
    public function getEMIOptions($paymentTypeId, $platform, $environment = 0, array $requestData = array()) {
        $paymentTypeId = (int) $paymentTypeId;
        //for old legacy
        if (empty($requestData['clientId'])) {
            $requestData['clientId'] = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $cartTotal = $requestData['get_payment_options_data']['totalPayment'];
        $emiEligibilityByTotal = isset($cartTotal) && !empty($cartTotal) && $cartTotal > 0 ? $this->dbQuote(' and cpoep.min_amount <= ?i', $cartTotal) : '';
        $sql = 'SELECT 
                cpo.payment_option_id,
                cpo.payment_type_id,
                cpo.name,
                cpo.description,
                cpo.image_url,
                cpo.m_image_url,
                cpo.position
            FROM
                pup_payment_options cpo
                    join
                pup_payment_options_emi_pgw cpoep ON (cpo.payment_option_id = cpoep.payment_option_id)
                    join
                pup_payments cp ON (cpoep.payment_gateway_id = cp.payment_id)
                    left join
                clues_envionment_platform_dependencies cep ON cpo.payment_option_id = cep.object_id
                    and cep.object_type = 0
                    and cep.platform = ?i
                    and cep.environment = ?i
            WHERE
                cpo.status = ?s and cp.status = ?s
                    and cpoep.status = ?s
                    and cep.object_id is null
                    and cpo.payment_type_id = ?i
                    and cp.client_id = ?s
		    ?p
            group by cpo.name';
        $query = $this->dbQuote($sql, $platform, $environment, 'A', 'A', 'A', $paymentTypeId, $requestData['clientId'], $emiEligibilityByTotal);
        return $this->dbGetArray($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to emi option details
     * @method getEMIDetailsDB
     * @access public
     * @param integer $paymentOptionId payment option id
     * @param integer $total order/cart total
     * @param string $clientId
     * @return array
     */
    public function getEMIDetailsDB($paymentOptionId, $total, $clientId = PaymentConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientId)) {
            $clientId = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $paymentOptionId = (int) $paymentOptionId;
        $sql = 'select 
                    cpoep.id, cpoep.payment_option_id, cpoep.payment_gateway_id,
                    cpoep.interest_rate,
                    cpoep.period, cpoep.fee, cpoep.promo_fee, cpoep.promo_end_date, 
                    cpoep.name, cpoep.status, cpt.name as payment_type,
                    cpo.name as payment_option ,cpoep.min_amount
                from 
                    pup_payment_options_emi_pgw cpoep 
                        join 
                    pup_payment_options cpo on (cpoep.payment_option_id=cpo.payment_option_id)
                        join 
                    pup_payment_types cpt on (cpo.payment_type_id=cpt.payment_type_id)
                        join 
                    pup_payments cp on (cpoep.payment_gateway_id=cp.payment_id) 
                where 
                    cpoep.payment_option_id=?i and cpoep.status = ?s
                        and cp.status = ?s and cpoep.min_amount <= ?i and cp.client_id = ?s';
        $sql = $this->dbQuote($sql, $paymentOptionId, 'A', 'A', $total, $clientId);
        return $this->dbGetArray($sql);
    }

    /**
     * This method is used to get products category ids
     * @method getCategoryId
     * @access public
     * @param integer $emiId emi id
     * @return array
     */
    public function getEMIduration($emiId) {
        $emiId = (int) $emiId;
        $sql = 'SELECT name from pup_payment_options_emi_pgw where id = ?i';
        $sql = $this->dbQuote($sql, $emiId);
        return $this->dbGetField($sql);
    }

    /**
     * This method is used to check cashcare serviceability on given pincode
     * @method checkCashcarePincodeServiceability
     * @param integer $pincode pincode
     * @access public
     * @return boolean true/false
     */
    public function checkCashcarePincodeServiceability($pincode) {
        $returnVal = false;
        $sql = 'SELECT count(serviceability_status) FROM  cashcare_serviceable_pincodes '
                . 'WHERE status=?i AND pincode=?i AND serviceability_status=?i';
        $query = $this->dbQuote($sql, 1, $pincode, 1);
        $data = $this->dbGetField($query);
        if (!empty($data)) {
            $returnVal = true;
        }
        return $returnVal;
    }

    /**
     * This method is used to get paymnent types
     * @method getPaymentTypes
     * @access public
     * @return array
     */
    public function getPaymentTypes() {
        $sql = 'SELECT payment_type_id as payment_method_type_id ,name as payment_method_name,image_url as payment_method_image_url,m_image_url as payment_method_m_image_url FROM pup_payment_types WHERE status = ?s order by position asc';
        $sql = $this->dbQuote($sql, 'A');
        return $this->dbGetHashArray($sql, 'payment_method_type_id', array(), self::MASTER_NOCACHE);
    }

    /**
     * Get Payment Info of order
     * @method getPaymentInfoByPaymentId
     * @access public
     * @param integer $paymentId
     * @return array array of payment Info
     */
    public function getPaymentInfoByPaymentId($paymentId) {
        $queryGetServerType = Registry::get('config.query_move_from_master_to_slave');
        $queryServerType = (!empty($queryGetServerType) ? self::SLAVE_NOCACHE : self::MASTER_NOCACHE);
        $sql = 'SELECT 
                    cp.payment_id,
                    cp.usergroup_ids,
                    cp.position,
                    cp.status,
                    cp.template,
                    cp.processor_id,
                    cp.params,cp.a_surcharge,
                    cp.p_surcharge,
                    cp.localization,
                    cpd.payment_id as desc_payment_id,
                    cpd.payment,
                    cpd.description,
                    cpd.instructions,
                    cpd.lang_code,
                    cpp.processor  
                FROM 
                    pup_payments  cp
                LEFT JOIN pup_payment_processors cpp
                        ON cpp.processor_id = cp.processor_id 
                LEFT JOIN pup_payment_descriptions cpd
                        ON cp.payment_id = cpd.payment_id AND cpd.cpd_status != ?i 
                WHERE cp.payment_id = ?i AND cp.status != ?s';
        $sanitizeSql = $this->dbQuote($sql, 2, $paymentId, 'R');
        $payment = $this->dbGetRow($sanitizeSql, array(), $queryServerType);
        $payment['params'] = (!empty($payment['params'])) ? json_decode($payment['params'], true) : '';
        return $payment;
    }

    /**
     * used to get payment gateway  Emi ID
     * @method getPriorityPGWEmi
     * @access public
     * @param  integer $paymentOptionId 
     * @param  integer $emiId      
     * @param string $clientID client id for merchant like ezonow/naaptol
     * @return integer payment gateway id        
     */
    public function getPriorityPGWEmi($paymentOptionId, $emiId, $clientID = PaymentConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $sql = 'SELECT 
                        cpoep.payment_gateway_id
                    FROM
                        pup_payment_options_emi_pgw cpoep
                            JOIN
                        pup_payment_options cpo ON (cpoep.payment_option_id = cpo.payment_option_id)
                            JOIN
                        pup_payment_types cpt ON (cpo.payment_type_id = cpt.payment_type_id)
                            JOIN
                        pup_payments cp ON (cpoep.payment_gateway_id = cp.payment_id)
                    WHERE
                        cpoep.payment_option_id = ?i
                            AND cpoep.status = ?s
                            AND cp.status = ?s
                            AND cpoep.id = ?i 
                            AND cp.client_id = ?s';
        $sanitizeSql = $this->dbQuote($sql, $paymentOptionId, 'A', 'A', $emiId, $clientID);
        $result = $this->dbGetField($sanitizeSql);
        return $result;
    }

    /**
     * used to get payment gateway
     * @method getPriorityPGW
     * @access public
     * @param integer $paymentOptionId 
     * @param string $clientID client id for merchant like ezonow/naaptol
     * @return integer payment gateway id
     */
    public function getPriorityPGW($paymentOptionId, $clientID = PaymentConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $sql = 'SELECT 
                        cpop.payment_gateway_id
                    FROM
                        pup_payment_option_pgw cpop
                            JOIN
                        pup_payment_options cpo ON (cpop.payment_option_id = cpo.payment_option_id)
                            JOIN
                        pup_payment_types cpt ON (cpo.payment_type_id = cpt.payment_type_id)
                            JOIN
                        pup_payments cp ON (cpop.payment_gateway_id = cp.payment_id)
                    WHERE
                        cpop.payment_option_id = ?i
                            AND cpop.status = ?s
                            AND cp.status = ?s
                            AND cp.client_id = ?s
                    ORDER BY priority ASC 
                    LIMIT 1';
        $sanitizeSql = $this->dbQuote($sql, $paymentOptionId, 'A', 'A', $clientID);
        $result = $this->dbGetField($sanitizeSql);
        return $result;
    }

    /**
     * used to get prepaid order status
     * @method getPrepaidOrderStatus
     * @access public
     * @param integer $orderId
     * @return string order status
     */
    public function getPrepaidOrderPaymentStatus($orderId) {
        $sql = "SELECT (CASE s.value when 1 then ?s when 2 then ?s when 3 then ?s ELSE ?s END) as payment_status FROM pup_prepayment_details "
                . "as m inner join pup_prepayment_details_data as s on (m.id=s.prepayment_id and s.key='payment_status') "
                . "where m.order_id=?i order by m.id desc limit 1";
        $sanitizeSql = $this->dbQuote($sql, PaymentConstants::PAYMENT_SUCCESS, PaymentConstants::PAYMENT_MANUALLY_REFUNDED, PaymentConstants::PAYMENT_PARTIALLY_REFUNDED, PaymentConstants::PAYMENT_FAILURE, $orderId);
        $result = $this->dbGetField($sanitizeSql, array(), self::MASTER_NOCACHE);
        if (!empty($result)) {
            return $result;
        } else {
            return 'no_transaction_found';
        }
    }

    /**
     * used to get payment ids for latest successful order of userId 
     * @method getLastSuccessPaymentIds
     * @access public
     * @param integer $userId
     * @param array $ignorePaymentEntities payment ids/types/options to be ignored when picking last success order payment id | OPTIONAL
     * @return array(payment ids)
     */
    public function getLastSuccessPaymentIds($userId, array $ignorePaymentEntities = array()) {
        $ignorePaymentEntitiesCond = '';
        if (isset($ignorePaymentEntities['payment_ids']) && count($ignorePaymentEntities['payment_ids']) > 0) {
            $ignorePaymentEntitiesCond .= $this->dbQuote(' AND cso.payment_id NOT IN (?n) ', $ignorePaymentEntities['payment_ids']);
        }
        if (isset($ignorePaymentEntities['payment_option_ids']) && count($ignorePaymentEntities['payment_option_ids']) > 0) {
            $ignorePaymentEntitiesCond .= $this->dbQuote(' AND cso.payment_option_id NOT IN (?n) ', $ignorePaymentEntities['payment_option_ids']);
        }
        if (isset($ignorePaymentEntities['payment_type_ids']) && count($ignorePaymentEntities['payment_type_ids']) > 0) {
            $ignorePaymentEntitiesCond .= $this->dbQuote(' AND cpo.payment_type_id NOT IN (?n) ', $ignorePaymentEntities['payment_type_ids']);
        }
        $sql = "SELECT 
                    cpo.payment_type_id, cso.payment_id, cso.payment_option_id
                FROM
                    cscart_orders cso
                        LEFT JOIN
                    pup_payment_options cpo ON cso.payment_option_id = cpo.payment_option_id
                        INNER JOIN
                    clues_order_history coh ON cso.order_id = coh.order_id
                WHERE
                    coh.to_status IN ('P' , 'Q', '92', '93')
                        AND cso.user_id = ?i
                        AND CASE
                        WHEN
                            (cso.payment_id = ?i
                                OR cso.payment_id <= 0)
                        THEN
                            1 = 1
                        ELSE cpo.status = ?s
                    END
                    ?p
                ORDER BY cso.order_id DESC
                LIMIT 1";
        $query = $this->dbQuote($sql, $userId, 73, 'A', $ignorePaymentEntitiesCond);
        $fetchedResult = $this->dbGetRow($query);
        return $fetchedResult;
    }

    /**
     * This method is used to payment options data
     * @method getPaymentOptionsByOptionId
     * @access public
     * @param integer $paymentOptionId 
     * @param integer $paymentId
     * @param integer $platform platform i.e D/W/A/M/I
     * @param integer $environment (default=0(production) LFDSP)
     * @return array (payment Info)
     */
    public function getAllPaymentInfoByOptionId($paymentOptionId, $paymentId, $platform, $environment = 0) {
        $sql = 'SELECT cpt.name as payment_method_name,cp.payment_id,cpo.payment_type_id, cpo.name,cpt.image_url as payment_method_image_url,
            cpt.m_image_url as payment_method_m_image_url, cpo.description, cpo.image_url,cpo.m_image_url, cpop.payment_option_pgw_id,cpop.payment_option_id, 
            cpop.payment_gateway_id,cpop.bank_code, cpop.status, group_concat(cpop.payment_gateway_id) as served_by,cpo.flow_type FROM pup_payment_options cpo 
            inner join pup_payment_option_pgw cpop on (cpo.payment_option_id=cpop.payment_option_id) 
            inner join pup_payments cp on (cpop.payment_gateway_id=cp.payment_id) 
            inner join pup_payment_types cpt on (cpt.payment_type_id=cpo.payment_type_id)
            left join clues_envionment_platform_dependencies cep on cpo.payment_option_id=cep.object_id and cep.object_type= ?i and cep.platform= ?i
            and cep.environment= ?i WHERE  cpo.payment_option_id = ?i and case when cp.payment_id=?i then 1=1 else
            cpo.status=?s and cp.status=?s end and cp.payment_id=?i and cep.object_id is null group by cpo.payment_option_id order by cpo.position asc';
        $query = $this->dbQuote($sql, 0, $platform, $environment, $paymentOptionId, 73, 'A', 'A', $paymentId);
        return $this->dbGetRow($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get payment type id's data
     * @method getDefaultPaymentTypeIdInfo
     * @access public
     * @param integer $paymentTypeId 
     * @return array (payment Info)
     */
    public function getDefaultPaymentTypeIdInfo($paymentTypeId) {
        $sql = 'select payment_type_id,name as payment_method_name,image_url as payment_method_image_url,m_image_url as payment_method_m_image_url,description,short_text_bottom,short_text_top from pup_payment_types where payment_type_id=?i and status=?s';
        $query = $this->dbQuote($sql, $paymentTypeId, 'A');
        return $this->dbGetRow($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * Get current down payment options
     * @method getCurrentDownPaymentOptions
     * @access public
     * @param string $source
     * @return array
     */
    public function getCurrentDownPaymentOptions($source) {
        $sql = 'SELECT cpo.name as payment_option_name, cpdn.sc_payment_option_id as payment_option_id, cpdn.payment_option_status_message, cpo.payment_type_id FROM clues_payment_downtime_notifications cpdn INNER JOIN pup_payment_options cpo on cpdn.sc_payment_option_id = cpo.payment_option_id WHERE cpo.status = ?s AND cpdn.payment_option_status = 0 AND cpdn.source = ?s ORDER BY cpo.payment_type_id';
        $sql = $this->dbQuote($sql, 'A', $source);
        return $this->dbGetHashMultiArray($sql, array('payment_type_id', 'payment_option_id'), array(), self::MASTER_NOCACHE);
    }

    /**
     * method to fetch pgw error message 
     * @method getPgwErrorMessage
     * @access public
     * @param integer $orderId
     * @param integer $paymentId
     * @return array containing error message 
     */
    public function getPgwErrorMessage($orderId, $paymentId) {
        $sql = 'SELECT cpemm.display_msg FROM pup_payment_pgw_mapping cppm INNER JOIN pup_prepayment_details cpd ON (cppm.prepayment_details_auto_increment_id = cpd.id
            AND cppm.prepayment_details_auto_increment_id != 0) INNER JOIN clues_pgw_error_message_mapping cpemm ON (cpd.3dstatus = cpemm.error_code
            AND cpd.3dstatus IS NOT NULL) WHERE cppm.client_order_id = ?s AND cpemm.status= ?i AND cpemm.payment_id = ?i ORDER BY cppm.id DESC LIMIT 1';
        $query = $this->dbQuote($sql, $orderId, 1, $paymentId);
        return $this->dbGetField($query);
    }

    /**
     * This method is used to get payment all details by client order id
     * @method getPaymentDetailsByOrderId
     * @access public
     * @param integer $orderId order id
     * @return array
     */
    public function getPaymentDetailsByOrderId($orderId) {
        $sql = "SELECT 
                m.client_order_id,
                m.client_order_id as order_id,
                m.payment_id,
                m.payment_option_id,
                m.emi_id,
                e.`name` AS emi_name,
                t.payment_type_id,
                IF(t.payment_type_id > 0,
                    t.`name`,
                    'CB/CB+/GC/Credit') AS pgw_type,
                IF(m.payment_id > 0,
                    s.payment,
                    'CB/CB+/GC/Credit') AS pgw,
                IF(m.payment_option_id > 0,
                    o.`name`,
                    'CB/CB+/GC/Credit') AS pgw_option_name,
                (CASE md.value when 1 then 1 when 2 then 2 when 3 then 3 ELSE 0 END) AS payment_status,
                (CASE md.value when 1 then 'Payment Received' when 2 then 'Payment Fully Refunded Manually' when 3 then 'Payment Partially Refunded Manually' ELSE 'Payment Not Received' END) AS payment_status_info,
                IF(IFNULL(md1.`value`, '') != '',
                    md1.`value`,
                    'NA') AS pgw_txn_id,
                m.pgw_order_id,
                m.client_order_id,
                m.prepayment_details_auto_increment_id,
                m.`status`,
                m.created AS created_timestamp,
                DATE_FORMAT(FROM_UNIXTIME(m.created),
                        '%d %M %Y %H:%i:%s') AS created,
                IF(md.value = 1,
                    DATE_FORMAT(FROM_UNIXTIME(md.updated),
                            '%d %M %Y %H:%i:%s'),
                    'NA') AS payment_received_date,
                m.updated,
                m.updated_by,
                m.created_by,
                dd.amount AS Payment_received_amt,
                dd.amount AS payment_received_amt,
                dd.direcpayreferenceid,
                dd.other_details,
                dd.flag,
                md.updated_by as updatedBy,
                md2.value as card_number
            FROM
                pup_payment_pgw_mapping AS m
                    LEFT JOIN
                pup_payment_descriptions AS s ON m.payment_id = s.payment_id
                    LEFT JOIN
                pup_payment_options as o ON o.payment_option_id = m.payment_option_id
                    left join
                pup_payment_types t ON t.payment_type_id = o.payment_type_id
                    left join
                pup_payment_options_emi_pgw e ON (e.id = m.emi_id and t.payment_type_id = 6)
                    left join
                pup_prepayment_details as dd ON (dd.id = m.prepayment_details_auto_increment_id)
                    left join
                pup_prepayment_details_data as md ON (md.prepayment_id = m.prepayment_details_auto_increment_id and md.`key` = 'payment_status')
                    left join
                pup_prepayment_details_data as md1 ON (md1.prepayment_id = m.prepayment_details_auto_increment_id and md1.`key` = 'pgw_txn_id')
                    left join
                pup_prepayment_details_data as md2 ON (md2.prepayment_id = m.prepayment_details_auto_increment_id and md2.`key` = 'card_number')
                where
                    m.client_order_id = ?s";
        $sanitizeSql = $this->dbQuote($sql, $orderId);
        return $this->dbGetArray($sanitizeSql, array(), self::MASTER_NOCACHE);
    }

    /**
     * Get Payment details by order id in pup_payment_pgw_mapping
     * @method getScOrderIdByPgwOrderIds
     * @access public
     * @param array $pgwOrderIds  
     * @param integer $paymentId payment id
     * @return array
     */
    public function getScOrderIdByPgwOrderIds($pgwOrderIds, $paymentId = 0) {
        if (!empty($paymentId)) {
            $query = "SELECT client_order_id,pgw_order_id from pup_payment_pgw_mapping where pgw_order_id in(?a) and payment_id = ?i order by created desc";
            $query = $this->dbQuote($query, $pgwOrderIds, $paymentId);
        } else {
            $query = "SELECT client_order_id,pgw_order_id from pup_payment_pgw_mapping where pgw_order_id in(?a) order by created desc";
            $query = $this->dbQuote($query, $pgwOrderIds);
        }
        $scOrders = $this->dbGetHashArray($query, 'pgw_order_id', array(), self::MASTER_NOCACHE);
        return $scOrders;
    }

    /**
     * This method is used to get sc order id by pgw txn id
     * @method getScOrderIdByPgwTxnIds
     * @access public 
     * @param array $pgwTxnIds
     * @return array
     */
    public function getScOrderIdByPgwTxnIds(array $pgwTxnIds) {
        $sql = "SELECT 
                        d.order_id as client_order_id, dd.`value` as pgw_tnx_id, IFNULL(pg.`value`, 'NA') as pgw_name 
                    from
                        pup_prepayment_details as d
                            inner join
                        pup_prepayment_details_data as dd ON d.id = dd.prepayment_id
                            left join
                        pup_prepayment_details_data as pg ON 
                        (d.id = pg.prepayment_id AND pg.`key` = 'pgw_name' AND pg.`status` = 1)
                    where
                        dd.`key` = 'pgw_txn_id'
                            and dd.`status` = 1
                            and dd.`value` in (?a)
                    group by d.order_id;";
        $query = $this->dbQuote($sql, $pgwTxnIds);
        $scOrders = $this->dbGetHashArray($query, 'pgw_tnx_id', array(), self::MASTER_NOCACHE);
        return $scOrders;
    }

    /**
     * This method is used to get sc order id by direcpayreferenceid
     * @method getScOrderIdByDirecPayReferenceId
     * @access public 
     * @param array $pgwTxnIds
     * @return array
     */
    public function getScOrderIdByDirecPayReferenceId(array $pgwTxnIds) {
        $sql = "SELECT 
                        d.order_id as client_order_id, d.`direcpayreferenceid` as pgw_tnx_id
                    from
                        pup_prepayment_details as d
                    where
                        d.`direcpayreferenceid` in (?a)
                    group by d.order_id;";
        $query = $this->dbQuote($sql, $pgwTxnIds);
        $scOrders = $this->dbGetHashArray($query, 'pgw_tnx_id', array(), self::MASTER_NOCACHE);
        return $scOrders;
    }

    /**
     * This method is used to check shopclue order id in prepayment details
     * @method getScOrderIdByPrePayDetails
     * @access public 
     * @param array $pgwTxnIds
     * @return array
     */
    public function getScOrderIdByPrePayDetails(array $pgwTxnIds) {
        $sql = 'SELECT 
                        d.order_id as client_order_id, d.`order_id` as pgw_tnx_id
                    from
                        pup_prepayment_details as d
                    where
                        d.`order_id` in (?a)
                    group by d.order_id;';
        $query = $this->dbQuote($sql, $pgwTxnIds);
        $scOrders = $this->dbGetHashArray($query, 'pgw_tnx_id', array(), self::MASTER_NOCACHE);
        return $scOrders;
    }

    /**
     * This method is used for getting mobile number linked to user id
     * @method getLinkedMobileNumber
     * @access public
     * @param integer $userId
     * @param integer $paymentOptionId
     * @param array $context e.g array('context' => 'CUSTOM_CONTEXT', 'reference_id' => 'UNIQUE_REF_ID')
     * @return array
     */
    public function getLinkedMobileNumber($userId, $paymentOptionId, array $context = array()) {
        $sql = "SELECT payment_option_id, phone FROM clues_payments_token_data WHERE user_id=?i AND payment_option_id=?i AND user_type IN ('L','G') AND token_status IN (0,1) ORDER BY user_type DESC, token_status DESC LIMIT ?i";
        $query = $this->dbQuote($sql, $userId, $paymentOptionId, 1);
        $res = $this->dbGetRow($query, array(), self::MASTER_NOCACHE);
        if (isset($res['phone']) && !empty($res['phone']) && !is_numeric($res['phone']) && !empty($context)) {
            $contextArray = array('context_value' => $context['context'], 'user_id' => $userId, 'reference_id' => $context['reference_id']);
            $res['phone'] = StringOperations::decryptIntegerString($res['phone'], $contextArray);
        }
        return $res;
    }

    /**
     * This method is used to get linked payment option ids for a given user id
     * @method getLinkedPaymentOptionIds
     * @access public
     * @param $userId
     * @return array
     */
    public function getLinkedPaymentOptionIds($userId) {
        $sql = 'select distinct payment_option_id from clues_payments_token_data where user_id=?i';
        $query = $this->dbQuote($sql, $userId);
        return $this->dbGetArray($query, array(), self::MASTER_CACHE);
    }

    /**
     * method used to get the token ,phone ,email for a  user ID
     * @method checkToken
     * @access public
     * @param integer $userId
     * @param integer $paymentOptionId
     * @param string $userType guest user or logged in
     * @param array $context e.g array('context' => 'CUSTOM_CONTEXT', 'reference_id' => 'UNIQUE_REF_ID')
     * @return array
     */
    public function checkToken($userId, $paymentOptionId, $userType, array $context = array()) {
        $sqlQuery = 'SELECT token,phone,email,token_status,user_type,user_id,payment_option_id  FROM clues_payments_token_data  WHERE  user_id=?i AND token_status=?i AND  payment_option_id=?i AND user_type=?s AND (token_expiry >= current_timestamp)';
        $sql = $this->dbQuote($sqlQuery, $userId, 1, $paymentOptionId, $userType);
        $res = $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
        if (isset($res['phone']) && !empty($res['phone']) && !is_numeric($res['phone']) && !empty($context)) {
            $contextArray = array('context_value' => $context['context'], 'user_id' => $userId, 'reference_id' => $context['reference_id']);
            $res['phone'] = StringOperations::decryptIntegerString($res['phone'], $contextArray);
        }
        return $res;
    }

    /**
     * Get Payment Params
     * @method getPaymentParams
     * @access public
     * @param integer $paymentId
     * @param string $clientID
     * @return string
     */
    public function getPaymentParams($paymentId, $clientID = PaymentConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $query = 'SELECT params FROM pup_payments WHERE payment_id = ?i AND client_id = ?s AND status != ?s';
        $sql = $this->dbQuote($query, $paymentId, $clientID, 'R');
        return $this->dbGetField($sql);
    }

    /**
     * This method is used to payment params by pgw order id and payment id
     * @method getPaymentInfoByPgwOrderIdAndPaymentId
     * @param string $pgwOrderId
     * @param integer $paymentId
     * @return array
     */
    public function getPaymentInfoByPgwOrderIdAndPaymentId($pgwOrderId, $paymentId, $clientID = PaymentConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $cond = '';
        if (in_array($paymentId, array(PaymentConstants::MOMOE_PAYMENT_ID, PaymentConstants::MOMOE_NB_PAYMENT_ID))) {
            $cond = $this->dbQuote('map.client_order_id=?s', $pgwOrderId);
        } else {
            $cond = $this->dbQuote('map.pgw_order_id=?s', $pgwOrderId);
        }
        $query = 'SELECT map.client_id,cop.merchant_params,map.client_order_id,map.pgw_order_id FROM '
                . '`pup_payment_pgw_mapping` AS map INNER JOIN pup_order_pgw AS cop ON '
                . 'map.order_pgw_auto_increment_id=cop.id WHERE map.payment_id=?i AND '
                . '?p AND map.client_id=?s ORDER By map.id DESC LIMIT 1';
        $sql = $this->dbQuote($query, $paymentId, $cond, $clientID);
        return $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * provide order status in cancellation bucket
     * @method getOrderSTatusInCancellationBucket
     * @access public
     * @return array all cancellation order statuses
     */
    public function getOrderSTatusInCancellationBucket() {
        $sql = 'SELECT distinct status FROM clues_status_types WHERE status_group in (?a) and object_type=?s';
        $sql = $this->dbQuote($sql, array(PaymentConstants::CANCELLATION_BUCKET_ID_E, PaymentConstants::CANCELLATION_BUCKET_ID_N), PaymentConstants::ORDER_OBJECT_TYPE);
        return $this->dbGetFields($sql);
    }

    /**
     * provide category ids of product ids
     * @method getCategoryIds
     * @access public
     * @param array $productId product id 
     * @return array category ids
     */
    public function getCategoryIds($productId) {
        $sql = 'SELECT category_id from cscart_products_categories where product_id in (?n)';
        $sql = $this->dbQuote($sql, $productId);
        return $this->dbGetArray($sql);
    }

    /**
     * This method is used to emi details
     * @method getEMIDetails
     * @access public
     * @param integer $paymentTypeId payment type id
     * @param integer $total
     * @param string $clientId
     * @return array
     */
    public function getEMIDetails($paymentTypeId, $total, $clientId = PaymentConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientId)) {
            $clientId = PaymentConstants::DEFAULT_CLIENT_ID;
        }
        $paymentTypeId = (int) $paymentTypeId;
        $sql = 'select 
                cpo.payment_option_id,cpo.payment_type_id,cpo.name as bank_name,cpo.description,cpo.image_url,cpo.m_image_url,cpo.position,cpo.status,cpo.flow_type,cpo.date_created,cpo.bank_mapping_id,cpo.updated,cpo.created_by,cpo.updated_by,cpo.payment_option_name,cpoep.id,cpoep.payment_option_id,cpoep.name,cpoep.period,cpoep.fee,cpoep.promo_fee,cpoep.promo_end_date,cpoep.payment_gateway_id,cpoep.position,cpoep.status,cpoep.failed_status,cpoep.interest_rate,cpoep.emi_code,cpoep.bank_code,cpoep.min_amount,cpoep.created,cpoep.updated,cpoep.created_by,cpoep.updated_by
            from
                pup_payment_options_emi_pgw cpoep
                    inner join
                pup_payment_options cpo ON (cpoep.payment_option_id = cpo.payment_option_id)
                    inner join
                pup_payment_types cpt ON (cpo.payment_type_id = cpt.payment_type_id)
                    inner join
                pup_payments cp ON (cpoep.payment_gateway_id = cp.payment_id)
            where
                cpoep.status = ?s and cp.status = ?s
                    and cpo.status = ?s
                    and cpo.payment_type_id = ?i and cpoep.min_amount<=?i and cp.client_id = ?s
                    group by cpo.payment_option_id,cpoep.id';
        $sql = $this->dbQuote($sql, 'A', 'A', 'A', $paymentTypeId, $total, $clientId);
        return $this->dbGetArray($sql);
    }

    /**
     * Fetches payment rules for given client id and module name
     * @method getCodPaymentRulesForClientId
     * @access public
     * @param string $clientId client id (alpha-numeric)
     * @param integer $moduleId module id
     * @return array
     */
    public function getPaymentRulesForClientId($clientId = CluesConstants::DEFAULT_CLIENT_ID, $moduleId = 0) {
        $clientId = empty($clientId) ? CluesConstants::DEFAULT_CLIENT_ID : $clientId;
        $sql = 'SELECT 
                    r.name, m.rule_status, m.rule_configurations
                FROM
                    pup_client_payment_rules r
                        INNER JOIN
                    pup_client_payment_rules_mapping m ON r.id = m.rule_id AND r.status = m.status
                WHERE
                    r.module_id = ?i AND r.status = ?i
                        AND m.client_id = ?s
                ORDER BY m.priority';
        $query = $this->dbQuote($sql, $moduleId, 1, $clientId);
        return $this->dbGetHashArray($query, 'name');
    }

    /**
     * provide pgw response(pre payment details) corresponding to pgw order id
     * @method getPrepaymentDetailsByPGWOrderId
     * @access public
     * @param string $pgwOrderId 
     * @return array $result prepayment details
     */
    public function getPrepaymentDetailsByPGWOrderId($pgwOrderId) {
        $result = array();
        if (!empty($pgwOrderId)) {
            $sql = 'SELECT 
                        cpd.other_details AS pgw_response,
                        cpd.amount,
                        cpd.date_created,
                        IF(REGEXP_INSTR(cpd.payment_gateway,"momoe"),"MOMOE","SHOPCLUES") AS website,
                        (CASE cpdd1.value when 1 then ?s when 2 then ?s when 3 then ?s ELSE ?s END) as payment_status,
                        IFNULL(cpdd.value,cpd.direcpayreferenceid) AS bank_txn_id,
                        cpd.direcpayreferenceid as direct_pay_ref_id,
                        IFNULL(cpdd1.value,cpd.direcpayreferenceid) AS merchant_ref_id,
                        cpdd3.value as pg_unique_ref_number
                    FROM
                        pup_prepayment_details cpd
                            INNER JOIN
                        pup_payment_pgw_mapping cppm ON cpd.id = cppm.prepayment_details_auto_increment_id
                            LEFT JOIN
                        pup_prepayment_details_data cpdd ON cpd.id = cpdd.prepayment_id and cpdd.key=?s
                            INNER JOIN
                        pup_prepayment_details_data cpdd1 ON cpd.id = cpdd1.prepayment_id and cpdd1.key=?s
                            LEFT JOIN 
                        pup_prepayment_details_data cpdd3 ON cpd.id = cpdd3.prepayment_id and cpdd3.key=?s
                    WHERE
                        cppm.pgw_order_id = ?s';
            $query = $this->dbQuote($sql, strtoupper(PaymentConstants::PAYMENT_SUCCESS), strtoupper(PaymentConstants::PAYMENT_MANUALLY_REFUNDED), strtoupper(PaymentConstants::PAYMENT_PARTIALLY_REFUNDED), strtoupper(PaymentConstants::PAYMENT_FAILED), 'pgw_txn_id', 'payment_status', 'pg_bank_ref_id', $pgwOrderId);
            $result = $this->dbGetHashArray($query, 'bank_txn_id', array(), self::MASTER_NOCACHE);
        }
        return $result;
    }

    /**
     * provide pgw response(pre payment details) corresponding to provided key and data
     * @method getPrepaymentDetails
     * @access public
     * @param string $key      
     * @param string $value
     * @param integer $txnAmt
     * @param string $txnStartTime  
     * @param string $txnEndTime  
     * @return array $result prepayment details
     */
    public function getPrepaymentDetails($key, $value, $txnAmt = 0, $txnStartTime = null, $txnEndTime = null) {
        $result = array();
        if (!empty($key) && !empty($value)) {
            switch ($key) {
                case 'card_number':
                    if (!empty($txnAmt) && !empty($txnStartTime)) {
                        if (!empty($txnEndTime)) {
                            $appendSql = $this->dbQuote(' cpdd.key = ?s and cpdd.value=?s and cpd.amount >= ?d and cpd.date_created >= ?s and cpd.date_created <= ?s', $key, $value, $txnAmt, $txnStartTime, $txnEndTime);
                        } else {
                            $appendSql = $this->dbQuote(' cpdd.key = ?s and cpdd.value=?s and cpd.amount >= ?d and cpd.date_created = ?s', $key, $value, $txnAmt, $txnStartTime);
                        }
                    }
                    $joinSql = ' INNER JOIN
                        pup_prepayment_details_data cpdd ON cpd.id = cpdd.prepayment_id';
                    break;
                case 'order_id':
                    $appendSql = $this->dbQuote(' cpd.order_id=?i', $value);
                    $joinSql = '';
                    break;
                case 'prepayment_id':
                    $appendSql = $this->dbQuote(' cpd.id in (?a)', $value);
                    break;
                default:
                    break;
            }
            $sql = 'SELECT 
                        cpd.other_details AS pgw_response,
                        cpd.amount,
			cpd.date_created,
                        IF(REGEXP_INSTR(cpd.payment_gateway, "momoe"),
                            "MOMOE",
                            "SHOPCLUES") AS website,
                        (CASE cpdd2.value when 1 then ?s when 2 then ?s when 3 then ?s ELSE ?s END) AS payment_status,
                        IFNULL(cpdd1.value, cpd.direcpayreferenceid) AS bank_txn_id,
			cpd.direcpayreferenceid AS direct_pay_ref_id,
                        IFNULL(cpdd1.value, cpd.direcpayreferenceid) AS merchant_ref_id,                        
                        cpdd3.value AS pg_unique_ref_number                        
                    FROM
                        pup_prepayment_details cpd    
                            ?p
                            LEFT JOIN 
                        pup_prepayment_details_data cpdd1 ON cpd.id = cpdd1.prepayment_id and cpdd1.key=?s
                            LEFT JOIN 
                        pup_prepayment_details_data cpdd3 ON cpd.id = cpdd3.prepayment_id and cpdd3.key=?s
                            INNER JOIN 
                        pup_prepayment_details_data cpdd2 ON cpd.id = cpdd2.prepayment_id and cpdd2.key=?s
                    WHERE
                        ?p';
            $query = $this->dbQuote($sql, strtoupper(PaymentConstants::PAYMENT_SUCCESS), strtoupper(PaymentConstants::PAYMENT_MANUALLY_REFUNDED), strtoupper(PaymentConstants::PAYMENT_PARTIALLY_REFUNDED), strtoupper(PaymentConstants::PAYMENT_FAILED), $joinSql, 'pgw_txn_id', 'pg_bank_ref_id', 'payment_status', $appendSql);
            $result = $this->dbGetHashArray($query, 'bank_txn_id', array(), self::MASTER_NOCACHE);
        }
        return $result;
    }

    /**
     * provide prepayment ids corresponding to provided bank txn id
     * @method getPrepaymentIdByBankTxnId
     * @access public
     * @param string $bankTxnId      
     * @return array prepayment_ids
     */
    public function getPrepaymentIdByBankTxnId($bankTxnId) {
        $sql = 'select prepayment_id from pup_prepayment_details_data where `key`=?s and value=?s';
        $sql = $this->dbQuote($sql, 'pgw_txn_id', $bankTxnId);
        return $this->dbGetFields($sql);
    }

    /**
     * provide prepayment ids corresponding to provided bank txn id/direcpayreferenceid
     * @method getPrepaymentIdByDirectPayRefId
     * @access public
     * @param string $bankTxnId      
     * @return array prepayment_ids
     */
    public function getPrepaymentIdByDirectPayRefId($bankTxnId) {
        $sql = 'select id from pup_prepayment_details where direcpayreferenceid=?s';
        $sql = $this->dbQuote($sql, $bankTxnId);
        return $this->dbGetFields($sql);
    }

    /**
     * This method is used to get payment last success payment details by shopclues order id
     * @method getLastSuccessPaymentDetailsByOrderId
     * @access public
     * @param integer $orderId order id
     * @param array $extraInfoArray extra info array
     * @return array
     */
    public function getLastSuccessPaymentDetailsByOrderId($orderId, array $extraInfoArray = array()) {
        $pgExpCol = $pgExpCond = '';
        //for fetching pg expense 
        if (!empty($extraInfoArray) && (in_array(PaymentConstants::PG_EXPENSE, $extraInfoArray) || (isset($extraInfoArray[PaymentConstants::PG_EXPENSE]) && !empty($extraInfoArray[PaymentConstants::PG_EXPENSE])))) {
            $pgExpCol = ", IF(IFNULL(md4.`value`, '') != '', md4.`value`, '') AS pg_expense";
            $pgExpCond = $this->dbQuote(' left join pup_prepayment_details_data as md4 ON (md4.prepayment_id = m.prepayment_details_auto_increment_id and md4.`key` = ?s)', PaymentConstants::APPX_PG_EXP);
        }
        $sql = "SELECT 
                m.client_order_id as order_id,
                m.payment_id,
                m.payment_option_id,
                m.emi_id,
                e.`name` AS emi_name,
                t.payment_type_id,
                o.image_url as pay_option_image,
                o.m_image_url as pay_option_m_image,
                t.image_url as pay_type_image,
                t.m_image_url as pay_type_m_image,
                IF(t.payment_type_id > 0,
                    t.`name`,
                    'CB/CB+/GC/Credit') AS pgw_type,
                IF(m.payment_id > 0,
                    s.payment,
                    'CB/CB+/GC/Credit') AS pgw,
                IF(m.payment_option_id > 0,
                    o.`name`,
                    'CB/CB+/GC/Credit') AS pgw_option_name,
                IF(IFNULL(md.id, 0) > 0, 1, 0) AS payment_status,
                IF(m.payment_option_id=?i,IFNULL(md3.value,'Normal'),'') as udf_payment_type,
                IF(IFNULL(md1.`value`, '') != '',
                    md1.`value`,
                    'NA') AS pgw_txn_id,
                IF(IFNULL(md2.`value`, '') != '',
                    md2.`value`,
                    '') AS card_number,
                m.pgw_order_id,
                m.client_order_id,
                m.prepayment_details_auto_increment_id,
                m.`status`,
                m.created AS created_timestamp,
                DATE_FORMAT(FROM_UNIXTIME(m.created),
                        '%d %M %Y %H:%i:%s') AS created,
                m.updated,
                m.updated_by,
                m.created_by,
                dd.amount AS payment_received_amt,
                dd.direcpayreferenceid,
                dd.flag ?p
            FROM
                `pup_payment_pgw_mapping` AS m
                    LEFT JOIN
                pup_payment_descriptions AS s ON m.payment_id = s.payment_id
                    LEFT JOIN
                    pup_payment_options as o ON o.payment_option_id = m.payment_option_id
                        left join
                    pup_payment_types t ON t.payment_type_id = o.payment_type_id
                        left join
                    pup_payment_options_emi_pgw e ON (e.id = m.emi_id
                        and t.payment_type_id = 6)
                        left join
                    pup_prepayment_details as dd ON (dd.id = m.prepayment_details_auto_increment_id)
                        left join
                    pup_prepayment_details_data as md ON (md.prepayment_id = m.prepayment_details_auto_increment_id
                        and md.`key` = 'payment_status'
                        and md.`value` = 1)
                        left join
                    pup_prepayment_details_data as md1 ON (md1.prepayment_id = m.prepayment_details_auto_increment_id
                        and md1.`key` = 'pgw_txn_id')
                        left join
                    pup_prepayment_details_data as md2 ON (md2.prepayment_id = m.prepayment_details_auto_increment_id
                        and md2.`key` = 'card_number')
                        left join 
                    pup_prepayment_details_data as md3 ON (md3.prepayment_id = m.prepayment_details_auto_increment_id
                        and md3.`key` = 'udf_payment_type') ?p

                where
                    m.client_order_id = ?s and m.payment_option_id not in (?a) and md.id > 0 order by m.id desc limit 1";
        $sanitizeSql = $this->dbQuote($sql, PaymentConstants::TB_PAYMENT_OPTION_ID, $pgExpCol, $pgExpCond, $orderId, array(PaymentConstants::COD_PAYMENT_OPTION_ID));
        return $this->dbGetRow($sanitizeSql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get payment id from pgw mapping table
     * @method getPaymentIdFromPaymentOptionId
     * @access public
     * @param integer orderId
     * @param integer paymentOptionId
     * @param integer emiId
     * @return integer payment id
     */
    public function getPaymentIdFromPaymentOptionId($orderId, $paymentOptionId, $emiId) {
        $sql = 'SELECT payment_id from pup_payment_pgw_mapping where client_order_id = ?s and payment_option_id = ?i and emi_id = ?i and status != ?i order by id desc limit ?i';
        $sanitizeSql = $this->dbQuote($sql, $orderId, $paymentOptionId, $emiId, 0, 1);
        return $this->dbGetField($sanitizeSql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get all the payment try for order
     * @method getAllPaymentTryForOrder
     * @access public
     * @param array $orderIds
     * @return array
     */
    public function getAllPaymentTryForOrder($orderIds = array(), $filters = array()) {
        $sql = 'SELECT 
                    cpdd.id AS prepaymentdetailsdata_primary_id,
                    co.total,
                    cpdd.updated_by,
                    cppm.client_order_id,
                    cpd.direcpayreferenceid AS ref_id,
                    cppm.pgw_order_id,
                    cpdd1.value AS pgw_txn_id,
                    cppm.payment_option_id,
                    cppm.payment_id,
                    FROM_UNIXTIME(cppm.updated) AS updated,
                    FROM_UNIXTIME(cppm.created) AS created,
                    cpd.payment_gateway,
                    co.order_id,
                    co.payment_id AS order_payment_id,
                    co.payment_option_id AS order_payment_option_id,
                    cppm.prepayment_details_auto_increment_id AS prepayment_id,
                    cppm.id AS pgw_mapping_primary_key,
                    cppm.status,
                    cpdd.key,
                    cpdd.value,
                    IF(cppm.client_order_id IS NULL, 1, 0) AS is_null_txn,
                    FROM_UNIXTIME(co.timestamp) AS order_creation_time
                FROM
                    cscart_orders co
                        LEFT JOIN
                    pup_payment_pgw_mapping cppm ON co.order_id = cppm.client_order_id
                        LEFT JOIN
                    pup_prepayment_details_data cpdd ON cppm.prepayment_details_auto_increment_id = cpdd.prepayment_id
                        AND cpdd.key = ?s
                        LEFT JOIN
                    pup_prepayment_details_data cpdd1 ON cpdd1.prepayment_id = cppm.prepayment_details_auto_increment_id
                        AND cpdd1.key = ?s
                        LEFT JOIN
                    pup_prepayment_details cpd ON cppm.prepayment_details_auto_increment_id = cpd.id
                WHERE
                    co.order_id IN (?n) AND co.payment_option_id != ?i';
        if (isset($filters['pgwOrderIds']) && !empty($filters['pgwOrderIds'])) {
            $sql .= $this->dbQuote(' AND cppm.pgw_order_id in (?a)', $filters['pgwOrderIds']);
        }
        if (isset($filters['pgwTxnIds']) && !empty($filters['pgwTxnIds'])) {
            $sql .= $this->dbQuote(' AND cpdd1.key=?s and cpdd1.value in (?a)', 'pgw_txn_id', $filters['pgwTxnIds']);
        }
        $sql = $this->dbQuote($sql, 'payment_status', 'pgw_txn_id', $orderIds, 0);
        return $this->dbGetArray($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to update value of given row id in pup_prepayment_details_data
     * @method updateStatusToManuallyProcessed
     * @access public
     * @param array $prepaymentDetailsDataPrimaryId
     * @param array $status
     * @param array $updatedBy
     * @return int
     */
    public function updateStatusToManuallyProcessed($prepaymentDetailsDataPrimaryId, $status, $updatedBy) {
        $sql = 'Update pup_prepayment_details_data set value=?i, updated=?i, updated_by=?i where id = ?i';
        $sql = $this->dbQuote($sql, $status, time(), $updatedBy, $prepaymentDetailsDataPrimaryId);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to create payment details data entry if not found 
     * @method createPrepaymentDetailDataEntry
     * @access public
     * @param int $prepayment_id
     * @param string $key
     * @param int $value
     * @param string $updatedBy
     * @return int
     */
    public function createPrepaymentDetailDataEntry($prepayment_id, $key, $value, $updatedBy) {
        $sql = 'INSERT into pup_prepayment_details_data (prepayment_id, `key`, value, status, created, updated, created_by, updated_by) values(?i, ?s, ?s, ?i, ?i, ?i, ?i, ?i)';
        $sql = $this->dbQuote($sql, $prepayment_id, $key, $value, 1, time(), time(), $updatedBy, $updatedBy);
        return $this->dbQueryInsert($sql);
    }

    /**
     * This method is used check if key exist or not for prepayment id
     * @method getPrepaymentDetailsDataStatus
     * @access public
     * @param array $prepayment_id
     * @param array $key
     * @return integer
     */
    public function getPrepaymentDetailsDataStatus($prepayment_id, $key) {
        $sql = 'SELECT id FROM pup_prepayment_details_data WHERE prepayment_id=?i AND `key`=?s';
        $sql = $this->dbQuote($sql, $prepayment_id, $key);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get pgw commission rate from pup_order_pgw_data table
     * @method getPgwCommissionData
     * @access public
     * @param  integer $orderPgwId
     * @return float|integer
     */
    public function getPgwCommissionData($orderPgwId) {
        $sql = 'SELECT object_value FROM pup_order_pgw_data WHERE order_pgw_id = ?i AND object_key = ?s AND status = ?i';
        $query = $this->dbQuote($sql, $orderPgwId, PaymentConstants::APPX_PG_COMM_RATE, 1);
        return $this->dbGetField($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * get pgw commission rate
     * @method getPgwCommissionRateData
     * @access public
     * @param array $ids [payment_id, payment_option_id, emi_id, amount]
     * @return float|integer
     */
    public function getPgwCommissionRateData(array $ids) {
        if ((isset($ids['payment_option_id']) && !empty($ids['payment_option_id'])) && (isset($ids['emi_id']) && !empty($ids['emi_id']))) {
            $queryCond = $this->dbQuote(' payment_option_id = ?i AND emi_id = ?i', $ids['payment_option_id'], $ids['emi_id']);
        } else if ((isset($ids['payment_option_id']) && !empty($ids['payment_option_id'])) && (!isset($ids['emi_id']) || empty($ids['emi_id']))) {
            $queryCond = $this->dbQuote(' payment_option_id = ?i AND emi_id = ?i', $ids['payment_option_id'], 0);
        } else {
            $queryCond = $this->dbQuote(' payment_option_id = ?i AND emi_id = ?i', 0, 0);
        }
        $sql = 'SELECT commission_rate FROM pup_payment_option_pgw_commission_rate WHERE payment_id = ?i AND ?p AND status=?i AND from_amount <= ?i AND to_amount >= ?i';
        $query = $this->dbQuote($sql, $ids['payment_id'], $queryCond, 1, $ids['amount'], $ids['amount']);
        return $this->dbGetField($query);
    }

    /**
     * get amount from pup_prepayment_details table
     * @method getAmountFromPrepaymentDetails
     * @access public
     * @param integer id
     * @return integer 
     */
    public function getAmountFromPrepaymentDetails($id) {
        $sql = 'SELECT amount FROM pup_prepayment_details WHERE id = ?i limit ?i';
        $query = $this->dbQuote($sql, $id, 1);
        return $this->dbGetField($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This function gets all the users by usergroup id for admin uses
     * @method getUserByUserGroupId
     * @access public
     * @param array $userGroupId
     * @return array
     */
    public function getUserByUserGroupId($userGroupId) {
        $sql = 'SELECT user_id FROM  cscart_usergroup_links where usergroup_id in (?n) and status=?s';
        $sql = $this->dbQuote($sql, $userGroupId, 'A');
        return $this->dbGetHashArray($sql, 'user_id');
    }

    /**
     * This method is used to create payment details entry if not found 
     * @method insertPrepaymentDetails
     * @access public
     * @param array $insertArray
     * @return int
     */
    public function insertPrepaymentDetails($insertArray) {
        return $this->dbQueryInsert("INSERT INTO pup_prepayment_details (" . implode(",", array_keys($insertArray)) . ") VALUES ('" . implode("','", $insertArray) . "')");
    }

    /**
     * This method is used to update prepayment_details_id in pup_payment_pgw_mapping table 
     * @method updatePrepaymentPrimaryKey
     * @access public
     * @param int $prepaymentId
     * @param int $pgwMappingPrimaryKey
     * @return int
     */
    public function updatePrepaymentPrimaryKey($prepaymentId, $pgwMappingPrimaryKey, $userId) {
        $sql = 'update pup_payment_pgw_mapping set prepayment_details_auto_increment_id = ?i, updated = ?i, updated_by = ?i where id = ?i limit ?i';
        $sql = $this->dbQuote($sql, $prepaymentId, time(), $userId, $pgwMappingPrimaryKey, 1);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to get order pgw auto increment id, payment id, payment option id and emi id from clues payment pgw mapping table
     * @method getDetailsFromPaymentPgwMapping
     * @access public
     * @param integer $pgwMappingPrimaryKey
     * @return array
     */
    public function getDetailsFromPaymentPgwMapping($pgwMappingPrimaryKey) {
        $sql = 'SELECT order_pgw_auto_increment_id, payment_id, payment_option_id, emi_id FROM pup_payment_pgw_mapping WHERE id = ?i limit ?i';
        $query = $this->dbQuote($sql, $pgwMappingPrimaryKey, 1);
        return $this->dbGetRow($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is getPaymentGateway
     * @method getPaymentGateway
     * @access public
     * @param array $pgwMappingPrimaryKey
     * @return array
     */
    public function getPaymentGateway($pgwMappingPrimaryKey) {
        $sql = 'select cp.payment_gateway_name,cpo.payment_option_name from pup_payment_pgw_mapping cpm join pup_payments cp on cpm.payment_id=cp.payment_id join pup_payment_options cpo on cpo.payment_option_id=cpm.payment_option_id where cpm.id=?i';
        $sql = $this->dbQuote($sql, $pgwMappingPrimaryKey);
        return $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * this function create insertInPgwMappingTable
     * @method insertInPgwMappingTable
     * @access public
     * @param array $insertArray
     * @return integer
     */
    public function insertInPgwMappingTable($insertArray) {
        return $this->dbQueryInsert("INSERT INTO pup_payment_pgw_mapping (" . implode(",", array_keys($insertArray)) . ") VALUES ('" . implode("','", $insertArray) . "')");
    }

    /**
     * get custom data from pup_prepayment_details_data
     * @method getCustomPrepaymentDetailsData
     * @access public
     * @param integer $prepaymentId
     * @param array $keys | OPTIONAL
     * @return array $keyValuePairs
     */
    public function getCustomPrepaymentDetailsData($prepaymentId, array $keys = array()) {
        $cond = '';
        if (!empty($keys)) {
            $cond = $this->dbQuote(' AND `key` IN (?a)', $keys);
        }
        $sql = 'SELECT `key`, `value` FROM pup_prepayment_details_data WHERE prepayment_id = ?i AND status = ?i ?p';
        $sanitizedSql = $this->dbQuote($sql, $prepaymentId, 1, $cond);
        return $this->dbGetHashArray($sanitizedSql, 'key');
    }

    /**
     * get payment mode details from payment id and payment option id
     * @method getPaymentOptionName
     * @access public
     * @param integer $paymentId
     * @param integer $paymentOptionId
     * @return array $paymentModeDetails
     */
    public function getPaymentModeDetails($paymentId, $paymentOptionId) {
        $sql = "SELECT 
                    t.payment_type_id,
                    IF(t.payment_type_id > 0,
                        t.`name`,
                        'CB/CB+/GC/Credit') AS pgw_type,
                    IF(d.payment_id > 0,
                        d.payment,
                        'CB/CB+/GC/Credit') AS pgw,
                    IF(o.payment_option_id > 0,
                        o.`name`,
                        'CB/CB+/GC/Credit') AS pgw_option_name
                FROM
                    pup_payment_options o
                        LEFT JOIN
                    pup_payment_types t ON t.payment_type_id = o.payment_type_id
                        LEFT JOIN
                    pup_payment_option_pgw p ON p.payment_option_id = o.payment_option_id
                        LEFT JOIN
                    pup_payment_descriptions d ON d.payment_id = p.payment_gateway_id
                WHERE
                    o.payment_option_id = ?i
                        AND p.payment_gateway_id = ?i";
        $sanitizedSql = $this->dbQuote($sql, $paymentOptionId, $paymentId);
        $paymentModeDetails = $this->dbGetRow($sanitizedSql);
        return $paymentModeDetails;
    }

    /**
     * fetching momoe_ref_id by txn id for momoe 
     * @method getMomoeOrderIdByTxnid
     * @access public
     * @param array $txnIds
     * @return array 
     */
    public function getMomoeOrderIdByTxnid($txnIds = array()) {
        $sql = "SELECT 
                    cpd.order_id as shopclues_order_id,
                    cpdd1.value as momoe_ref_id,
                    cpdd2.value as payment_txn_id,
                    cpdd3.value as pg_bank_ref_id
                FROM
                    pup_prepayment_details cpd
                        JOIN
                    pup_prepayment_details_data cpdd ON cpd.id = cpdd.prepayment_id
                        LEFT JOIN
                    pup_prepayment_details_data cpdd1 ON cpd.id = cpdd1.prepayment_id
                        AND cpdd1.`key` = 'momoe_ref_id'
                        LEFT JOIN
                    pup_prepayment_details_data cpdd2 ON cpd.id = cpdd2.prepayment_id
                        AND cpdd2.`key` = 'pgw_txn_id'
                        LEFT JOIN
                    pup_prepayment_details_data cpdd3 ON cpd.id = cpdd3.prepayment_id
                        AND cpdd3.`key` = 'pg_bank_ref_id'
                WHERE
                    ((cpdd.`key` = 'pg_bank_ref_id'
                        AND cpdd.value IN (?a)
                        OR (cpdd.`key` = 'pgw_txn_id'
                        AND cpdd.value IN (?a)))
                        OR (cpdd.`key` = 'momoe_ref_id'
                        AND cpdd.value IN (?a)))
                        group by cpdd1.value,cpdd2.value,cpdd3.value;";
        $query = $this->dbQuote($sql, $txnIds, $txnIds, $txnIds);
        $momoeRefIdsMappingData = $this->dbGetArray($query, array(), self::MASTER_NOCACHE);
        return $momoeRefIdsMappingData;
    }

    /**
     * This method is used to check cod availability on Pincode
     * @method validPincodeForCod
     * @access public
     * @return integer
     */
    public function validPincodeForCod($pincode) {
        $sql = 'SELECT pincode from clues_cod_disabled_pincodes where id=?i and pincode=?s and type=?s and status=?s';
        $query = $this->dbQuote($sql, 0, $pincode, 'E', 'A');
        return $this->dbGetField($query);
    }

    /**
     * This method is used to fetch offer applied on order ids
     * @method checkIfOfferAlreadyConsumed
     * @access public
     * @param $orderIds array
     * @param $status array
     * @return array
     */
    public function checkIfOfferAlreadyConsumed($orderIds = array(), $status = array()) {
        $queryCond = '';
        if (!empty($status)) {
            $queryCond = $this->dbQuote(' and status in (?a)', $status);
        }
        $sql = 'SELECT order_id, user_id, offer_type, offer_value, expiry_date as offer_expiry, status, order_credit_status, offer_desc from clues_user_offers_redeem where order_id in (?a) ?p';
        $query = $this->dbQuote($sql, $orderIds, $queryCond);
        return $this->dbGetArray($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to insert offer applied on order id
     * @method insertOrderOfferData
     * @access public
     * @param $orderId integer
     * @param $userId integer
     * @param $valuePerOrder integer
     * @param $offerType integer
     * @param $status integer
     * @param $creditStatus string
     * @param $offerExpire string
     * @param $offerDesc string
     * @return integer
     */
    public function insertOrderOfferData($orderId, $userId, $valuePerOrder, $offerType, $status, $creditStatus, $offerExpire, $offerDesc) {
        $sql = 'INSERT INTO clues_user_offers_redeem (order_id, user_id, offer_type, offer_value, expiry_date, status, order_credit_status, offer_desc, cd, md) VALUES (?i, ?i, ?i, ?i, ?s, ?i, ?s, ?s, ?s, ?s)';
        $query = $this->dbQuote($sql, $orderId, $userId, $offerType, $valuePerOrder, $offerExpire, $status, $creditStatus, $offerDesc, date('Y-m-d H:i:s', time()), date('Y-m-d H:i:s', time()));
        return $this->dbQueryInsert($query);
    }

    /**
     * This method is used to update scratch card offer status
     * @method updateOfferStatus
     * @access public
     * @param array $orderIds
     * @param integer $fromStatus
     * @param integer $toStatus
     * @return integer
     */
    public function updateOfferStatus(array $orderIds, $fromStatus, $toStatus) {
        $query = 'UPDATE clues_user_offers_redeem SET status = ?i WHERE order_id in (?a) and status = ?i';
        $sql = $this->dbQuote($query, $toStatus, $orderIds, $fromStatus);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to check bucket by order status
     * @method getBucketByOrderStatus
     * @param string $status status
     * @return integer
     */
    public function getBucketByOrderStatus($status) {
        $query = 'SELECT status_group FROM clues_status_types WHERE status = ?s and object_type = ?s';
        $sql = $this->dbQuote($query, $status, 'O');
        return $this->dbGetField($sql);
    }

    /**
     * This method is used to add card details in pup_user_card_info table
     * @method addCardDetails
     * @param array $cardDetails
     * @return integer
     */
    public function addCardDetails($cardDetails) {
        $sql = 'INSERT INTO pup_user_card_info (user_id, name, default_card, card_token, pup_tk, hash, card_hash_2c2p, card_obf, status, expiryMonth, expiryYear, created, updated, created_by, updated_by) VALUES (?i, ?s, ?s, ?s, ?s, ?s, ?s, ?s, ?i, ?s, ?s, ?s, ?s, ?i, ?i)';
        $query = $this->dbQuote($sql, $cardDetails['user_id'], $cardDetails['card_nickname'], $cardDetails['default_card'], $cardDetails['card_token'], $cardDetails['pup_tk'], $cardDetails['card_hash'], $cardDetails['card_hash_2c2p'], $cardDetails['masked_card_no'], PaymentConstants::INITIAL_STATUS_FOR_USER_CARD, $cardDetails['expiry_month'], $cardDetails['expiry_year'], date('Y:m:d H:i:s'), date('Y:m:d H:i:s'), $cardDetails['user_id'], $cardDetails['user_id']);
        return $this->dbQueryInsert($query);
    }

    /**
     * This method is used to check pup token from pup_user_card_info table
     * @method checkPupToken
     * @param string $pupToken
     * @return integer
     */
    public function checkPupToken($pupToken) {
        $query = 'SELECT count(1) AS cnt FROM pup_user_card_info WHERE pup_tk = ?s';
        $sql = $this->dbQuote($query, $pupToken);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get user card info from pup_user_card_info table
     * @method getUserCardInfo
     * @param integer $userId
     * @param array $defaultCards
     * @return array
     */
    public function getUserCardInfo($userId, array $defaultCards = array(0, 1)) {
        $sql = 'select name, pup_tk as pup_card_token, card_obf, expiryMonth, expiryYear, default_card from pup_user_card_info where user_id = ?i and status = ?i and default_card in (?n) order by default_card desc';
        $query = $this->dbQuote($sql, $userId, PaymentConstants::INITIAL_STATUS_FOR_USER_CARD, $defaultCards);
        return $this->dbGetArray($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to check whether same hash of a particular user exists or not
     * @method checkCardHashExists
     * @param integer $userId
     * @param string $cardHash
     * @return array
     */
    public function checkCardHashExists($userId, $cardHash) {
        $sql = 'select name, card_token, pup_tk, card_obf, expiryMonth, expiryYear from pup_user_card_info where user_id = ?i and hash = ?s and status = ?i';
        $query = $this->dbQuote($sql, $userId, $cardHash, 1);

        return $this->dbGetArray($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get data of user card from pup_user_card_info table
     * @method getDataForDeleteCard
     * @param integer $userId
     * @param string $pupToken
     * @return integer
     */
    public function getDataForDeleteCard($userId, $pupToken) {
        $sql = 'select id, card_token, card_hash_2c2p from pup_user_card_info where user_id = ?i and pup_tk = ?s and status = ?i';
        $query = $this->dbQuote($sql, $userId, $pupToken, 1);
        return $this->dbGetRow($query, self::MASTER_NOCACHE);
    }

    /**
     * This method is used to delete user card from pup_user_card_info table
     * @method deleteUserCard
     * @param integer $cardId
     * @param string $status
     * @param string $hashMsg
     * @return integer
     */
    public function deleteUserCard($cardId, $status, $hashMsg) {
        $sql = 'update pup_user_card_info set status = ?i, hash = ?s where id = ?i';
        $query = $this->dbQuote($sql, $status, $hashMsg, $cardId);
        return $this->dbQueryUpdate($query);
    }

    /**
     * This method is used to update default card of user in pup_user_card_info table
     * @method updateDefaultCardsOfUser
     * @param integer $userId
     * @param integer $defaultValue
     * @param integer $cardId
     * @return integer
     */
    public function updateDefaultCardsOfUser($userId, $defaultValue, $cardId) {
        $sql = 'update pup_user_card_info set default_card = ?i, updated_by = ?i where user_id = ?i and status = ?i and id != ?i';
        $query = $this->dbQuote($sql, $defaultValue, $userId, $userId, PaymentConstants::INITIAL_STATUS_FOR_USER_CARD, $cardId);
        return $this->dbQueryUpdate($query);
    }

    /**
     * This method is used to check pup token from pup_user_card_info table
     * @method checkPupToken
     * @param string $cardHash2c2p
     * @param integer $status
     * @return integer
     */
    public function countHashValue2c2p($cardHash2c2p, $status) {
        if (empty($cardHash2c2p)) {
            return 0;
        }
        $query = 'SELECT count(*) AS cnt FROM pup_user_card_info WHERE card_hash_2c2p = ?s AND status = ?i';
        $sql = $this->dbQuote($query, $cardHash2c2p, $status);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

   /**
     * This method is used to check count of 2c2p token
     * @method checkPupToken
     * @param string $cardHash2c2p
     * @param integer $status
     * @return integer
     */
    public function count2c2pCardToken($cardToken, $status) {
        if (empty($cardHash2c2p)) {
            return 0;
        }
        $query = 'SELECT count(1) AS cnt FROM pup_user_card_info WHERE card_token = ?s AND status = ?i';
        $sql = $this->dbQuote($query, $cardToken, $status);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get card info from token
     * @method getCardInfoFromToken
     * @param integer $userId
     * @param string $token
     * @return array
     */
    public function getCardInfoFromToken($userId, $token) {
        $currentYear = (int) date("y");
        $currentMonth = (int) date("m");
        $sql = 'select name, pup_tk as pup_card_token, card_obf, expiryMonth, expiryYear, default_card from pup_user_card_info where pup_tk = ?s and status = ?i and user_id = ?i and expiryYear >= ?i and if(expiryYear > ?i, 1, expiryMonth >= ?i)';
        $query = $this->dbQuote($sql, $token, 1, $userId, $currentYear, $currentYear, $currentMonth);
        return $this->dbGetRow($query, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get card info from token
     * @method getDataForCurrentToken
     * @param integer $userId
     * @param string $pupToken
     * @return array
     */
    public function getDataForCurrentToken($userId, $pupToken) {
        $sql = 'select id, name from pup_user_card_info where user_id = ?i and pup_tk = ?s and status = ?i';
        $query = $this->dbQuote($sql, $userId, $pupToken, 1);
        return $this->dbGetRow($query, self::MASTER_NOCACHE);
    }

    /**
     * This method is used to update name of user in pup_user_card_info table
     * @method setDefaultNickName
     * @param integer $cardId
     * @param integer $userId
     * @param string $nickName
     * @return integer
     */
    public function setDefaultNickName($cardId, $userId, $nickName) {
        $sql = 'update pup_user_card_info set name = ?s, updated_by = ?i, updated = ?s where status = ?i and id = ?i';
        $query = $this->dbQuote($sql, $nickName, $userId, date("Y/m/d h:m:s"), 1, $cardId);
        return $this->dbQueryUpdate($query);
    }

    /**
     * This method is used to get card info if the card is set as default or not
     * @method getInfoIfDefaultCard
     * @param integer $userId
     * @param string $pupToken
     * @return array
     */
    public function getInfoIfDefaultCard($userId, $pupToken) {
        $sql = 'select id, default_card from pup_user_card_info where user_id = ?i and pup_tk = ?s and status = ?i';
        $query = $this->dbQuote($sql, $userId, $pupToken, 1);
        return $this->dbGetRow($query, self::MASTER_NOCACHE);
    }

    /**
     * This method is used to update default card of user in pup_user_card_info table
     * @method setDefaultCard
     * @param integer $cardId
     * @param integer $userId
     * @param integer $defaultValue
     * @return integer
     */
    public function setDefaultCard($cardId, $userId, $defaultValue) {
        $sql = 'update pup_user_card_info set default_card = ?i, updated_by = ?i, updated = ?s where status = ?i and id = ?i';
        $query = $this->dbQuote($sql, $defaultValue, $userId, date("Y-m-d H:i:s"), 1, $cardId);
        return $this->dbQueryUpdate($query);
    }

    /**
     * This method is used to get default card of user in pup_user_card_info table
     * @method getDefaultCardsOfUser
     * @param integer $userId
     * @param integer $defaultValue
     * @param integer $cardId
     * @return array
     */
    public function getDefaultCardsOfUser($userId, $defaultValue, $cardId) {
        $sql = 'select id from pup_user_card_info where user_id = ?i and status = ?i and default_card = ?i and id != ?i';
        $query = $this->dbQuote($sql, $userId, PaymentConstants::INITIAL_STATUS_FOR_USER_CARD, $defaultValue, $cardId);
        return $this->dbGetArray($query, self::MASTER_NOCACHE);
    }

}
