<?php

namespace clues\model\pgw;

use clues\system\BasePlatform;
use clues\system\Logger;
use clues\system\DI;
use clues\model\pgw\PgwModelConstants;
use clues\system\CluesConstants;
use clues\system\util\StringOperations;
use clues\system\CluesRegistry;

/**
 * This class is used to perform all common operations related to payment
 * @class PGWDriver
 * @access public
 * @category pgw
 * @package model/pgw
 */
abstract class PGWDriver extends BasePlatform {

    /**
     * used to hold input params
     * @var array
     */
    protected $inputParams = array();

    /**
     * client id
     * @var string
     */
    protected $clientID;

    /**
     * Payment service object
     * @access protected
     * @var object 
     */
    protected $paymentService;

    /**
     * $PaymentPgwMappingObj
     * @var object 
     */
    protected $PaymentPgwMappingObj;

    /**
     * order info object
     * @access protected
     * @var object 
     */
    protected $orderInfo;

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @param mixed $data
     */
    public function __construct($data = array()) {
        parent::__construct($data);
        DI::Map('PaymentService', 'clues\\model\\pgw\\PaymentService');
        $this->paymentService = DI::Singleton('PaymentService');
        DI::Map('PaymentPgwMappingObj', 'clues\\model\\pgw\\PaymentPgwMapping');
        $this->PaymentPgwMappingObj = DI::Singleton('PaymentPgwMappingObj');
        $this->setAppClientIdDetails();
    }

    /**
     * This method is used to make registry key for get order info
     * @method getOrderInfoRegistryKey
     * @access public
     * @param string $orderId
     * @return string
     */
    public function getOrderInfoRegistryKey($orderId) {
        return $orderId . PgwModelConstants::REGISTRY_KEY_GET_ORDER_INFO;
    }

    /**
     * Get pgw order info
     * @method getPgwOrderInfo
     * @access public
     * @param string $orderId can be client order id or pgw order id
     * @param string $cache cache data in $orderInfo
     * @return array
     * @throws \Exception
     */
    public function getPgwOrderInfo($orderId, $cache = true) {
        if (isset($this->orderInfo) && !empty($this->orderInfo) && $cache) {
            return $this->orderInfo;
        }
        try {
            $pgwOrderId = $orderId;
            if (!empty($orderId)) {
                $_tempOrderId = $this->PaymentPgwMappingObj->getScOrderIdByPgwOrderId($orderId);
                if (empty($_tempOrderId)) {
                    $orderId = $this->paymentService->getOrderIdByPgwOrderId($orderId);
                } else {
                    $orderId = $_tempOrderId;
                }
                unset($_tempOrderId);
            }
            if (!empty($orderId)) {
                $orderInfo = $this->getOrderInfo($orderId, PgwModelConstants::ORDER_INFO_EXTENDED, array(), PgwModelConstants::CONTEXT_PGW_REQUEST, $pgwOrderId);
            } else {
                $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Empty Order Id');
                throw new PgwException(PgwStatus::INVALID_OR_BLANK_ORDER_ID, $exceptionData);
            }
        } catch (\Exception $ex) {
            throw $ex;
        }
        if ($cache) {
            $this->orderInfo = $orderInfo;
        }
        return $orderInfo;
    }

    /**
     * Get Order Info 
     * @method getOrderInfo
     * @access protected
     * @param integer $orderId
     * @param string $type
     * @param array $orderFields order custom fields
     * @param string $context for decrypt mobile no
     * @param string $pgwOrderId
     * @throws Exception
     * @return mixed
     */
    protected function getOrderInfo($orderId, $type = PgwModelConstants::ORDER_INFO_EXTENDED, array $orderFields = array(), $context = PgwModelConstants::CONTEXT_PGW_REQUEST, $pgwOrderId = null) {
        $orderData = array();
        $registryKey = $this->getOrderInfoRegistryKey($orderId);
        $registryFlag = false;
        if (CluesRegistry::isKeySet($registryKey)) {
            $orderData = CluesRegistry::getObject($registryKey);
            $registryFlag = true;
        } else if (!empty($pgwOrderId)) {
            $paymentInfo = $this->paymentService->getPaymentDetailsFromPgwMapping($pgwOrderId);
            if (isset($paymentInfo['order_pgw_auto_increment_id']) && !empty($paymentInfo['order_pgw_auto_increment_id']) && $paymentInfo['order_pgw_auto_increment_id'] > 0) {
                $orderData = $this->paymentService->getOrderInfoFromOrderPgwById($paymentInfo['order_pgw_auto_increment_id']);
            }
        }
        if (empty($orderData)) {
            $orderData = $this->paymentService->getOrderInfoFromOrderPgw($orderId);
        }
        if (empty($orderId)) {
            $orderId = $orderData['order_id'];
        }
        if ($registryFlag === false) {
            $registryKey = $this->getOrderInfoRegistryKey($orderId);
            CluesRegistry::addObject($registryKey, $orderData);
        }
        $orderData = (object) $orderData;
        $paymentId = 0;
        $paymentOptionId = (isset($orderData->payment_option_id) && !empty($orderData->payment_option_id)) ? $orderData->payment_option_id : 0;
        $paymentEmiId = (isset($orderData->emi_id) && !empty($orderData->emi_id)) ? $orderData->emi_id : 0;
        if (!empty($paymentOptionId)) {
            $paymentId = $this->getPaymentIdByPaymentOption($paymentOptionId, $paymentEmiId);
        }
        $paymentInfo = array();
        if ($paymentId) {
            $paymentInfo = $this->paymentService->getPaymentInfo($paymentId);
        }
        $orderData->payment = (object) $paymentInfo;
        $securityLogRefId = StringOperations::getMd5UniqueId($context);
        $logData = array();
        $logData['field1'] = PgwModelConstants::CONTEXT_PGW_REQUEST;
        $logData['field2'] = 'order_id:' . $orderId;
        $logData['field5'] = 'refid:' . $securityLogRefId;
        $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
        $logData['error_name'] = 'get_order_info_in_pgw_driver';
        $additionalInfo = array();
        $additionalInfo['file'] = __FILE__;
        $additionalInfo['line'] = __LINE__;
        $additionalInfo['module'] = CluesConstants::DOMAIN_SECURITY_NAME;
        $additionalInfo['domain'] = CluesConstants::MODULE_SECURITY_NAME;
        $additionalInfo['level'] = Logger::INFO;
        $additionalInfo['method'] = __METHOD__;
        $logData['additional_data'] = $additionalInfo;
        $this->LoggerObj->log(CluesConstants::DOMAIN_SECURITY_NAME, CluesConstants::MODULE_SECURITY_NAME, $logData, Logger::INFO);
        return $orderData;
    }

    /**
     * provide payment id for payment option id
     * @method getPaymentIdByPaymentOption
     * @access public
     * @param integer $paymentOptionId payment option id
     * @param integer $emiId emi id
     * @param string $clientID client id
     * @return integer payment id
     */
    public function getPaymentIdByPaymentOption($paymentOptionId, $emiId = 0, $clientID = PgwModelConstants::DEFAULT_CLIENT_ID) {
        $data = $this->paymentService->getPaymentId($paymentOptionId, $emiId, $clientID);
        if (empty($data) && $clientID != PgwModelConstants::DEFAULT_CLIENT_ID) {
            $data = $this->paymentService->getPaymentId($paymentOptionId, $emiId, PgwModelConstants::DEFAULT_CLIENT_ID);
        }
        return $data;
    }

    /**
     * this method is used to set client ID
     * @access protected
     * @method setClientID
     * @param string $clientID
     * @return void
     */
    protected function setClientID($clientID) {
        if (!empty($clientID)) {
            $this->inputParams['client_id'] = $this->clientID = $clientID;
        }
    }

    /**
     * This method is used to set app_client_id details
     * @method setAppClientIdDetails
     * @param string $clientId client id
     * @param boolean $updateClientId
     * @access protected
     * @return array
     */
    protected function setAppClientIdDetails($clientId = null, $updateClientId = true) {
        if (empty($clientId)) {
            $clientId = $this->RequestObj->getHeader('app_client_id');
        }
        if (empty($clientId)) {
            $clientId = PgwModelConstants::DEFAULT_CLIENT_ID;
        }
        if (!empty($clientId)) {
            $details = $this->paymentService->getAppClientIdDetails($clientId);
            if (!empty($details)) {
                $details['client_configurations'] = StringOperations::jsonDecodeOrUnserialize($details['client_configurations']);
            }
            CluesRegistry::addObject('app_client_id_info', $details);
            if ($updateClientId) {
                $this->setClientID($clientId);
            }
        }
    }

    /**
     * get payment info by payment id
     * @method getPaymentInfo
     * @access public
     * @param integer $paymentId payment id
     * @param string $clientID order client id
     * @return array payment info
     */
    public function getPaymentInfo($paymentId, $clientID = PgwModelConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientID)) {
            $clientID = PgwModelConstants::DEFAULT_CLIENT_ID;
        }
        $paramData = $this->paymentService->getPaymentInfo($paymentId, $clientID);
        if ((empty($paramData) || !is_array($paramData)) && ($clientID != PgwModelConstants::DEFAULT_CLIENT_ID)) {
            $paramData = $this->paymentService->getPaymentInfo($paymentId, PgwModelConstants::DEFAULT_CLIENT_ID);
        }
        return $paramData;
    }

    /**
     * Get exception data
     * @method _getExceptionData
     * @access protected
     * @param integer $line
     * @param string $method
     * @param string $file
     * @param string $errorName
     * @param string $field1
     * @param string $field2
     * @param string $field3
     * @param string $field4
     * @return array
     */
    protected function _getExceptionData($line, $method, $file, $errorName = '', $field1 = '', $field2 = '', $field3 = '', $field4 = '') {
        $exception_data = array();
        $exception_data['field1'] = $field1;
        $exception_data['field2'] = $field2;
        $exception_data['field1'] = $field3;
        $exception_data['field2'] = $field4;
        $exception_data['error_name'] = $errorName;
        $additionalInfo = array();
        $additionalInfo['file'] = $file;
        $additionalInfo['line'] = $line;
        $additionalInfo['module'] = PgwModelConstants::MODULE_NAME;
        $additionalInfo['domain'] = PgwModelConstants::DOMAIN_NAME;
        $additionalInfo['level'] = Logger::ERROR;
        $additionalInfo['method'] = $method;
        $exception_data['additional_data'] = $additionalInfo;
        return $exception_data;
    }

    /**
     * decrypt initiate payment request Data
     * @method decryptInitiatePaymentRequest
     * @access protected
     * @return array
     */
    protected function decryptInitiatePaymentRequest() {
        $requestBodydata = $this->RequestObj->getALLRequestData();
        if (isset($requestBodydata['initiate_payment_payload']) && !empty($requestBodydata['initiate_payment_payload'])) {
            $_tempKeyData = CluesRegistry::getObject('app_client_id_info');
            if (isset($_tempKeyData['client_configurations']['initiate_payment_key']) &&
                    !empty($_tempKeyData['client_configurations']['initiate_payment_key']) &&
                    isset($_tempKeyData['client_configurations']['initiate_des_method']) &&
                    !empty($_tempKeyData['client_configurations']['initiate_des_method'])) {
                DI::Map('EncryptDecrypt', 'clues\\model\\pgwVerify\\util\\EncryptDecrypt');
                $EncryptDecryptObj = DI::Singleton('EncryptDecrypt');
                $EncryptDecryptObj->setKey($_tempKeyData['client_configurations']['initiate_payment_key']);
                $finalJson = $EncryptDecryptObj->decrypt($requestBodydata['initiate_payment_payload'], $_tempKeyData['client_configurations']['initiate_des_method']);
                unset($EncryptDecryptObj);
                $finalArray = json_decode($finalJson, true);
                if (!empty($finalArray)) {
                    $requestBodydata = array_merge($requestBodydata, $finalArray);
                }
            }
        }
        return $requestBodydata;
    }

}
