<?php

namespace clues\model\pgw;

use clues\model\pgw\PGWDriver;
use clues\system\DI;
use clues\model\pgw\PgwModelConstants;
use clues\system\CluesConstants;
use clues\model\pgw\exception\PgwException;
use clues\model\pgw\exception\PgwStatus;
use clues\system\Logger;
use clues\system\CluesRegistry;

/**
 * This class is responsible for payment gateways operations
 * @class PGWManager
 * @access public
 * @category Model class
 * @package model/pgw
 */
class PGWManager extends PGWDriver {

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @return void
     */
    public function __construct() {
        parent::__construct();
        DI::Map('PGWModel', 'clues\\model\\pgw\\PGWModel');
    }

    /**
     * Response from PGW
     * @method responseFromPgw
     * @access public
     * @return array
     * @throws PgwException
     */
    public function responseFromPgw() {
        $pgwResponse = $this->RequestObj->postRequestParams();
        $requestData = $this->RequestObj->getALLRequestData();
        $pgwResponse = empty($pgwResponse) ? $requestData : $pgwResponse;
        $return = isset($requestData['sc_return_response']) && !empty($requestData['sc_return_response']) ? true : false;
        $redirect = !$return;
        try {
            $pgwData = $this->_decryptPgwData($requestData);
            $orderId = (isset($requestData['order_id']) && !empty($requestData['order_id'])) ? $requestData['order_id'] : (isset($pgwData['order_id']) && !empty($pgwData['order_id']) ? $pgwData['order_id'] : 0);
            if (empty($orderId)) {
                $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Empty Order Id');
                throw new PgwException(PgwStatus::EMPTY_ORDER_ID, $exceptionData);
            }
            $pgwModelObj = DI::Singleton('PGWModel', $pgwData);
            $orderInfo = $pgwModelObj->getPgwOrderInfo($orderId, false);
            $paymentId = isset($orderInfo->payment->payment_id) && !empty($orderInfo->payment->payment_id) ? $orderInfo->payment->payment_id : 0;
            return $pgwModelObj->saveResponseFromPGW($orderId, $pgwResponse, $redirect, $return);
        } catch (\Exception $ex) {
            $logData = array('field1' => 'clues_udf:' . $requestData['clues_pgw'], 'field2' => 'error_code: ' . $ex->getCode(), 'error_name' => 'Home page redirection in responseFromPGW', 'data' => $ex->getMessage());
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::MODULE_NAME, $logData, Logger::ERROR);
            $this->_homePageRedirection($requestData, $redirect);
        }
    }

    /**
     * Failed Page redirection
     * @method _homePageRedirection
     * @access private
     * @param array $requestData
     * @param boolean $redirect
     * @return array
     */
    private function _homePageRedirection($requestData, $redirect) {
        $platform = isset($requestData['platform']) && !empty($requestData['platform']) ? $requestData['platform'] : CluesConstants::PARAM_DEFAULT_PLATFORM;
        if ($platform == CluesConstants::PARAM_DEFAULT_PLATFORM) {
            $redirectUrl = $this->CluesConfigObj->getKeyValue('home_desk_dir', PgwModelConstants::PAYMENT_DB_CONFIG_MODULE_NAME);
        } else {
            $redirectUrl = $this->CluesConfigObj->getKeyValue('home_mobile_dir', PgwModelConstants::PAYMENT_DB_CONFIG_MODULE_NAME);
        }
        if (!$redirect) {
            return array('redirect_url' => $redirectUrl);
        }
        header('location:' . $redirectUrl);
        exit;
    }

    /**
     * Decrypt pgw data
     * @method _decryptPgwData
     * @access private
     * @param array $requestData
     * @return array
     * @throws PgwException
     */
    private function _decryptPgwData($requestData) {
        if (!isset($requestData['clues_pgw']) || empty($requestData['clues_pgw'])) {
            return array();
        }
        DI::Map('Crypt', 'clues\\system\\util\\Crypt');
        $cryptObj = DI::Singleton('Crypt');
        $encryptedPgwData = str_replace(' ', '+', $requestData['clues_pgw']);
        $decryptedPgwString = $cryptObj->decrypt($encryptedPgwData);
        if (empty($decryptedPgwString) || !is_string($decryptedPgwString)) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Empty Clues PGW');
            throw new PgwException(PgwStatus::EMPTY_CLUES_PGW, $exceptionData);
        }
        $pgwParams = explode('|', $decryptedPgwString);
        return array('payment_id' => $pgwParams[0], 'payment_option_id' => $pgwParams[1], 'emi_id' => $pgwParams[2], 'order_id' => $pgwParams[3], 'soaUrl' => $pgwParams[4], 'platform' => $pgwParams[5]);
    }

    /**
     * this method is used to intiate payment process
     * @method initiatePayment
     * @access public
     * @return array if response_in_json key is true
     */
    public function initiatePayment() {
        $requestBodydata = $this->decryptInitiatePaymentRequest();
        if (!isset($requestBodydata['initiate_payment_payload']) || empty($requestBodydata['initiate_payment_payload'])) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'initiate payment payload missing');
            throw new PgwException(PgwStatus::PGW_PARAMTER_MISSING, $exceptionData);
        }
        if (!isset($requestBodydata['order_id']) || empty($requestBodydata['order_id'])) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Order Id Missing');
            throw new PgwException(PgwStatus::ORDER_ID_MISSING, $exceptionData);
        }
        if (!isset($requestBodydata['payment_option_id']) || empty($requestBodydata['payment_option_id'])) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Payment Option Id Missing');
            throw new PgwException(PgwStatus::PAYMENT_OPTION_ID_MISSING, $exceptionData);
        }
        if (!isset($requestBodydata['total']) || empty($requestBodydata['total'])) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Total Key Missing');
            throw new PgwException(PgwStatus::TOTAL_KEY_MISSING, $exceptionData);
        }
        if (!isset($requestBodydata['user']) || empty($requestBodydata['user'])) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'User Details Missing');
            throw new PgwException(PgwStatus::USER_DETAILS_MISSING, $exceptionData);
        }
        if (!isset($requestBodydata['user']['user_id']) || empty($requestBodydata['user']['user_id'])) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'User Id Missing');
            throw new PgwException(PgwStatus::USER_ID_MISSING, $exceptionData);
        }
        if (!isset($requestBodydata['user']['name']) || empty($requestBodydata['user']['name'])) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'User Name Missing');
            throw new PgwException(PgwStatus::USER_NAME_MISSING, $exceptionData);
        }
        if (!isset($requestBodydata['user']['mobile']) || empty($requestBodydata['user']['mobile'])) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'User Mobile Missing');
            throw new PgwException(PgwStatus::USER_MOBILE_MISSING, $exceptionData);
        }
        /*if (!isset($requestBodydata['card_info']) || empty($requestBodydata['card_info']) || (!isset($requestBodydata['card_info']['client_token']) && !isset($requestBodydata['card_info']['secure_pay_token']))) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Card Details Missing');
            throw new PgwException(PgwStatus::CARD_DETAILS_MISSING, $exceptionData);
        }*/
        if (!isset($requestBodydata['card_info']) || empty($requestBodydata['card_info']) || (!isset($requestBodydata['card_info']['pup_card_token']))) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Card Details Missing');
            throw new PgwException(PgwStatus::CARD_DETAILS_MISSING, $exceptionData);
        }
        $requestBodydata['card_info']['client_token'] = $requestBodydata['card_info']['pup_card_token'];
        $registryKey = $this->getOrderInfoRegistryKey($requestBodydata['order_id']);
        $requestBodydata['clientID'] = $this->clientID;
        CluesRegistry::addObject($registryKey, $requestBodydata);
        $responseInJson = false;
        if(isset($requestBodydata['response_in_json']) && (strtolower($requestBodydata['response_in_json']) === "true" || (is_bool($requestBodydata['response_in_json']) && $requestBodydata['response_in_json']))) {
            $responseInJson = true;
        }
        $pgwModel = DI::Singleton('PGWModel', $requestBodydata);
        $initiateResponseArray = $pgwModel->initiatePayment();
        if($responseInJson && !empty($initiateResponseArray) && is_array($initiateResponseArray)) {
            return $initiateResponseArray;
        }
    }
}
