<?php

namespace clues\model\pgw;

use clues\model\pgw\PGWDriver;
use clues\system\Logger;
use clues\system\DI;
use clues\model\pgw\PgwModelConstants;
use clues\system\util\ObjectOperations;
use clues\system\CluesRegistry;
use clues\model\pgw\exception\PgwException;
use clues\model\pgw\exception\PgwStatus;

/**
 * This class is responsible for payment gateways operations
 * @class PGWModel
 * @access public
 * @category Model class
 * @package model/pgw
 */
class PGWModel extends PGWDriver {

    /**
     * Platform
     * @var string 
     */
    public $platform;

    /**
     * Order Id
     * @var integer 
     */
    protected $orderId;

    /**
     * Soa api flag
     * @var String 
     */
    private $_isSoaApi = false;

    /**
     * Current Api version
     * @access protected
     * @var integer 
     */
    protected $currentApiVersion = 0;

    /**
     * Payment Id
     * @access public
     * @var integer 
     */
    public $paymentId;

    /**
     * request data array
     * @access protected
     * @var array 
     */
    protected $requestData;

    /**
     * payment option id
     * @access public
     * @var integer 
     */
    public $payment_option_id;

    /**
     * Ui part in intitiate/sendtopgw payment api
     * @access private
     * @var boolean
     */
    private $_showUISendToPgw = true;

    /**
     * Atom mode start time
     * @var float
     * @access protected
     */
    protected $atomModelStartTime = 0;

    /**
     * mobile dir config
     * @var string
     * @access protected
     */
    protected $mobileDirConfig;

    /**
     * desk dir config
     * @var string
     * @access protected
     */
    protected $deskDirConfig;

    /**
     * root dir config
     * @var string
     * @access protected
     */
    protected $rootDirConfig;

    /**
     * Default page msg for initiate payment
     * @var string html
     */
    protected $defaultInitiatePayMsg = '<center><h1>Placing Order</h1></center>';

    /**
     * used to identify whether to return response or not
     * @access protected
     * @var boolean
     */
    protected $initiatePaymentReturnResponseArray = false;

    /**
     * Used to hold pgw mapping script name and atom pgw file
     * @var array
     */
    private static $_pgwMappingArray = array(
        1 => 'Pgw2C2P'
    );

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @param mixed $data
     */
    public function __construct($data = array()) {
        parent::__construct($this->inputParams);
        $this->init($data);
    }

    /**
     * Method used to get payment processor id 
     * @method getPaymentProcessor
     * @access public
     * @param integer $paymentId 
     * @param string $clientId
     * @return integer
     */
    public function getPaymentProcessor($paymentId, $clientId) {
        $processorId = $this->paymentService->getPaymentProcessor($paymentId, $clientId);
        if (empty($processorId) && $clientId != PgwModelConstants::DEFAULT_CLIENT_ID) {
            $processorId = $this->paymentService->getPaymentProcessor($paymentId, PgwModelConstants::DEFAULT_CLIENT_ID);
        }
        return $processorId;
    }

    /**
     * This method is used to get the pgw object
     * @method getPGW
     * @access public
     * @param integer $paymentId id payment id
     * @param integer $orderId order id
     * @param string $platform platform
     * @param integer $guestUser (guestUser = 1 / logged in = 0')
     * @return mixed
     */
    public function getPGW($paymentId, $orderId, $platform = 'M', $guestUser = 1) {
        $pgwobj = $this->mapPgwObject($paymentId);
        $response = array();
        $response['pgwObj'] = $pgwobj;
        $response['return_url'] = $this->getReturnUrl($paymentId, $orderId, $platform, $guestUser);
        return $response;
    }

    /**
     * This method is used to get the pgw object
     * @method mapPgwObject
     * @access protected
     * @param integer $paymentId
     * @return mixed
     */
    protected function mapPgwObject($paymentId) {
        $pgwObj = null;
        if (!empty($paymentId) && isset(self::$_pgwMappingArray[$paymentId]) && !empty(self::$_pgwMappingArray[$paymentId])) {
            $mapStr = 'clues\\model\\pgw\\' . self::$_pgwMappingArray[$paymentId];
            DI::Map(self::$_pgwMappingArray[$paymentId], $mapStr);
            $pgwObj = DI::Singleton(self::$_pgwMappingArray[$paymentId], $this->inputParams);
        }
        return $pgwObj;
    }

    /**
     * Get Return url
     * @method getReturnUrl
     * @access protected
     * @param string $pgw pgw script
     * @param integer $orderId order id
     * @param string $platform platform
     * @param integer $guestUser (guestUser = 1 / logged in = 0')
     * @return string
     */
    protected function getReturnUrl($paymentId, $orderId, $platform, $guestUser = 1) {
        $isSoaApi = ($this->_isSoaApi) ? 1 : 0;
        DI::Map('Crypt', 'clues\\system\\util\\Crypt');
        $cryptObj = DI::Singleton('Crypt');
        $udfString = $this->paymentId . '|' . $this->payment_option_id . '|' . $this->emiId . '|' . $orderId . '|' . $isSoaApi . '|' . $platform;
        $encryptedUdfString = $cryptObj->encrypt($udfString);
        unset($udfString);
        $root_dir = $this->rootDirConfig;
        $desk_dir = $this->deskDirConfig;
        $key_array = $this->CluesConfigObj->getKeyValue('api_keys', 'auth');
        $key = array_search($platform, $key_array);
        $returnUrl = '';
        $pgw = self::$_pgwMappingArray[$paymentId];
        switch ($paymentId) {
            case 1:
                $returnUrl = $root_dir . '/atom/DoPayment/ResponseFromPgw?key=' . $key . '&platform=' . $platform . '&pgw=' . $pgw . '&clues_pgw=' . $encryptedUdfString;
                break;
            default :
               $returnUrl = $root_dir . '/atom/DoPayment/ResponseFromPgw?key=' . $key . '&platform=' . $platform . '&pgw=' . $pgw . '&clues_pgw=' . $encryptedUdfString;
                break;
        }
        return $returnUrl;
    }

    /**
     * this method is used to update payment in order
     * @method _checkChangedPayment
     * @access private    
     * @return integer
     */
    private function _checkChangedPayment() {
        return 0;
    }

    /**
     * This method is used to process payment for changed payment
     * @method _changedPaymentProcess
     * @access private
     * @return array if $initiateResponseArray is set and not empty
     */
    private function _changedPaymentProcess() {
        $extraData = array(
            'order_id' => $this->orderId,
            'platform' => $this->platform
        );
        $this->orderInfo = $this->getOrderInfo($this->orderId, PgwModelConstants::ORDER_INFO_EXTENDED, array(), PgwModelConstants::CONTEXT_PGW_REQUEST);
        $initiateResponseArray = $this->sendToPgw(false, $extraData);
        if ($this->initiatePaymentReturnResponseArray && !empty($initiateResponseArray) && is_array($initiateResponseArray)) {
            return $initiateResponseArray;
        }
    }

    /**
     * this method is used to place order
     * @method initiatePayment
     * @access public
     * @return array if $initiateResponseArray is set and not empty
     */
    public function initiatePayment() {
        try {
            $this->orderInfo = $this->getOrderInfo($this->orderId, PgwModelConstants::ORDER_INFO_EXTENDED, array(), PgwModelConstants::CONTEXT_PGW_REQUEST);
            $initiateResponseArray = $this->_processPayment();
            if ($this->initiatePaymentReturnResponseArray && !empty($initiateResponseArray) && is_array($initiateResponseArray)) {
                return $initiateResponseArray;
            }
        } catch (\Exception $e) {
            $logData = array('field1' => 'orderId:' . $this->orderId, 'field2' => 'order info paymentOptionId' . $this->orderInfo->payment_option_id, 'field3' => 'platform:' . $this->platform, 'field4' => 'requestData' . json_encode($this->requestData), 'error_name' => 'intiatePaymentException', 'data' => $e->getMessage());
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::INITIATE_PAYMENT_MODULE_NAME, json_encode($logData), Logger::ERROR);
            unset($logData);
            if(in_array($e->getCode(), PgwModelConstants::INITIATE_PAYMENT_THROW_EXCEPTION_CODES)) {
                throw $e;
            }
            $response = $this->orderPageRedirection($this->orderId, $this->platform, PgwModelConstants::PAYMENT_FAILED_STATUS, !$this->initiatePaymentReturnResponseArray);
            if ($this->initiatePaymentReturnResponseArray) {
                $response['error'] = $e->getMessage();
                return $response;
            }
        }
    }

    /**
     * this method is used to provide ui based on platform type
     * @method placingOrderUI
     * @access protected
     * @param boolean $showDefaultMsg
     * @return string
     */
    protected function placingOrderUI($showDefaultMsg = false) {
        if ($showDefaultMsg) {
            return $this->defaultInitiatePayMsg;
        }
        if ($this->_showUISendToPgw) {
            $this->_showUISendToPgw = false;
            $returnStr = $this->defaultInitiatePayMsg;
            switch (true) {
                case ($this->_isSoaApi) :
                    $returnStr = $this->defaultInitiatePayMsg;
                    break;
                default :
                    $htmlJson = $this->CluesConfigObj->getKeyValue('html_for_placing_order_ui', 'payment');
                    $htmlArray = json_decode($htmlJson, true);
                    if (empty($htmlArray)) {
                        $returnStr = $this->defaultInitiatePayMsg;
                    } else {
                        $platform = $this->platform;
                        if (isset($htmlArray[$platform]) && is_array($htmlArray[$platform]) &&
                                isset($htmlArray[$platform]['html']) && !empty($htmlArray[$platform]['html']) && isset($htmlArray[$platform]['enable']) && $htmlArray[$platform]['enable'] == 1) {
                            $returnStr = $htmlArray[$platform]['html'];
                        } elseif (isset($htmlArray['DEF']) && is_array($htmlArray['DEF']) && !empty($htmlArray['DEF']['html'])) {
                            $returnStr = $htmlArray['DEF']['html'];
                        } else {
                            $returnStr = $this->defaultInitiatePayMsg;
                        }
                    }
                    unset($htmlJson, $htmlArray, $platform);
                    break;
            }
            return $returnStr;
        }
    }

    /**
     * this method is used to process payment
     * @method _processPayment
     * @access private
     * @return array if $initiateResponseArray is set and not empty
     */
    private function _processPayment() {
        $isPaymentChanged = $this->_checkChangedPayment();
        if ($isPaymentChanged) {
            $initiateResponseArray = $this->_changedPaymentProcess();
            if ($this->initiatePaymentReturnResponseArray && !empty($initiateResponseArray) && is_array($initiateResponseArray)) {
                return $initiateResponseArray;
            }
        } else {
            $extraData = array(
                'order_id' => $this->orderId,
                'platform' => $this->platform
            );
            $initiateResponseArray = $this->sendToPgw(false, $extraData);
            if ($this->initiatePaymentReturnResponseArray && !empty($initiateResponseArray) && is_array($initiateResponseArray)) {
                return $initiateResponseArray;
            }
        }
    }

    /**
     * this method is used to redirect to payment page in case any error
     * @method _paymentPageRedirectionError
     * @access private
     * @param boolean $logFlag flag for logging
     * @return void 
     */
    private function _paymentPageRedirectionError($logFlag = TRUE) {
        if ($logFlag) {
            $logData = array('field1' => 'orderId:' . $this->orderId, 'field2' => 'paymentOptionId' . $this->orderInfo->payment_option_id, 'field3' => 'orderTotal:' . $this->orderInfo->total, 'error_name' => 'zeroTotalError');
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::INITIATE_PAYMENT_MODULE_NAME, json_encode($logData), Logger::ERROR);
        }
        $platformTemp = strtoupper($this->platform);
        $platformWiseLandingPageUrl = $this->CluesConfigObj->getKeyValue('platform_wise_landing_page_url', 'payment', 'api');
        $paymentURL = $platformWiseLandingPageUrl[$platformTemp]['returnurl_fail'];
        unset($logData, $platformTemp);
        header("Location:" . $paymentURL);
        exit();
    }

    /**
     * Get Payment deails by order id from clues_payment_pgw_mapping
     * @method getPaymentInfoFromPgwMapping
     * @access public
     * @param integer $orderId
     * @return mixed array('payment_id', 'payment_option_id');
     */
    public function getPaymentInfoFromPgwMapping($orderId) {
        $response = array('payment_id' => 0, 'payment_option_id' => 0);
        $orderData = $this->paymentService->getPaymentDetailsFromPgwMapping($orderId);
        if (!empty($orderData)) {
            if (isset($orderData['payment_id']) && !empty($orderData['payment_id']) && isset($orderData['payment_option_id']) && !empty($orderData['payment_option_id'])) {
                $response['payment_id'] = (int) $orderData['payment_id'];
                $response['payment_option_id'] = (int) $orderData['payment_option_id'];
                $response['order_id'] = (int) $orderData['client_order_id'];
            }
        }
        return $response;
    }

    /**
     * Insert in clues_payment_service_data for soa api
     * @method saveUrlsApi
     * @access public
     * @param integer $orderId string $successUrl string $failedUrl 
     * @return integer
     */
    public function saveUrlsApi($orderId, $successUrl, $failedUrl) {
        return $this->paymentService->insertSoaUrls($orderId, $successUrl, $failedUrl);
    }

    /**
     * Get from  clues_payment_service_data for soa api
     * @method getSoaUrlsForRedirection
     * @access public
     * @param integer $orderId order id
     * @return mixed 
     */
    public function getUrlsForRedirection($orderId) {
        $data = $this->paymentService->getSoaUrls($orderId);
        return $data;
    }

    /**
     * Used to send order to payment gateway
     * @method sendToPgw
     * @access public
     * @param boolean $returnFlag return form
     * @param array $extraData array('convert_to_cod'=><convert_to_cod>, 'to_status' => <to_status>, 'otp_verified'=><otp_verified>)
     * @return array if $initiateResponseArray is set and not empty
     */
    public function sendToPgw($returnFlag = false, array $extraData = array()) {
        $orderInfoPropCount = ObjectOperations::getCount($this->orderInfo);
        if (empty($this->orderInfo) || $orderInfoPropCount == 0) {
            $this->requestData = $this->decryptInitiatePaymentRequest();
            if (((isset($this->requestData['order_id']) && !empty($this->requestData['order_id'])) || (isset($this->requestData['orderId']) && !empty($this->requestData['orderId']))) && isset($this->requestData['platform']) && !empty($this->requestData['platform'])) {
                $this->orderId = (isset($this->requestData['orderId']) && !empty($this->requestData['orderId'])) ? $this->requestData['orderId'] : $this->requestData['order_id'];
                $this->platform = $this->requestData['platform'];
            } else {
                $logData = array('field1' => 'orderId:' . $this->orderId, 'field2' => 'platform:' . $this->platform, 'field3' => 'sendToPgw', 'field4' => '', 'error_name' => 'orderIdOrPlatformEmpty', 'data' => json_encode($this->requestData));
                $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, 'sendToPgwOrderException', json_encode($logData), Logger::ERROR);
            }
            try {
                $this->orderInfo = $this->getOrderInfo($this->orderId, PgwModelConstants::ORDER_INFO_EXTENDED, array(), PgwModelConstants::CONTEXT_PGW_REQUEST);
            } catch (\Exception $e) {
                $logData = array('field1' => 'orderId:' . $this->orderId, 'field2' => 'platform:' . $this->platform, 'field3' => 'requestData', 'field4' => json_encode($this->requestData), 'error_name' => 'sendToPgwOrderException', 'data' => $e->getMessage());
                $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, 'sendToPgwOrderException', json_encode($logData), Logger::ERROR);
            }
        }
        $data = $this->requestData;
        $paymentId = isset($this->orderInfo->payment->payment_id) && !empty($this->orderInfo->payment->payment_id) ? $this->orderInfo->payment->payment_id : $this->orderInfo->payment_id;
        $this->setIpInRegistry($data);
        $data['processor'] = $this->orderInfo->payment->processor;
        $data['failed_url'] = $this->getOrderFailedUrl($this->orderId, $this->platform);
        $data['success_url'] = $this->getOrderSuccessUrl($this->orderId, $this->platform);
        $data['payment_id'] = $paymentId = isset($this->orderInfo->payment->payment_id) ? $this->orderInfo->payment->payment_id : 0;
        $data['emi_id'] = isset($this->orderInfo->emi_id) ? $this->orderInfo->emi_id : 0;
        $data['payment_option_id'] = isset($this->orderInfo->payment_option_id) ? $this->orderInfo->payment_option_id : 0;
        $this->init($data);
        $response = $this->getPGW($paymentId, $this->orderId, $this->platform, 0);
        $pgwObj = $response['pgwObj'];
        if (empty($pgwObj)) {
            exit("please contact pup admin");
        }
        $this->setPaymentTypeId($this->orderInfo);
        if(isset($this->requestData['card_info']['client_token']) && !empty($this->requestData['card_info']['client_token']) && !in_array($this->orderInfo->payment_type_id, PgwModelConstants::CARD_TOKEN_REQUIRED_PAYMENT_TYPE_IDS)) {
            $exceptionData = $this->_getExceptionData(__LINE__, __METHOD__, __FILE__, 'Invalid payment option id');
            throw new PgwException(PgwStatus::INVALID_PAYMENT_OPTION_ID, $exceptionData);
        }
        $pgwObj->initialize($this->orderInfo, $response['return_url']);
        $form = $pgwObj->getDataForRedirection();
        if ($returnFlag) {
            return $form;
        }
        if (isset($form['payment_id']) && $form['payment_id'] == PgwModelConstants::Pgw2C2P_PAYMENT_ID && isset($form['form_array'][PgwModelConstants::PGW_DIRECT_VERIFY]) && !empty($form['form_array'][PgwModelConstants::PGW_DIRECT_VERIFY]) && $form['form_array'][PgwModelConstants::PGW_DIRECT_VERIFY]) {
            $extraData['skip_place_order_display'] = 1;
        }
        if (!isset($extraData['skip_place_order_display']) && !$this->initiatePaymentReturnResponseArray) {
            echo $this->placingOrderUI();
        }
        $initiateResponseArray = $this->postFormPgw($form, $pgwObj, $extraData);
        if ($this->initiatePaymentReturnResponseArray && !empty($initiateResponseArray) && is_array($initiateResponseArray)) {
            return $initiateResponseArray;
        }
    }

    /**
     * This method is uesd to get order failed url
     * @method getOrderFailedUrl
     * @access protected
     * @param integer $orderId order id
     * @param integer $platform platform
     * @return string
     */
    protected function getOrderFailedUrl($orderId, $platform, $momoeUserId = 0) {
        if ($this->_isSoaApi) {
            $soaData = $this->getUrlsForRedirection($orderId);
            if (isset($soaData) && is_array($soaData) && isset($soaData[0]) &&
                    is_array($soaData[0]) && isset($soaData[0]['returnurl_fail']) &&
                    !empty($soaData[0]['returnurl_fail'])) {
                $failedURL = $soaData[0]['returnurl_fail'];
                return $failedURL;
            }
        }
        $platformTemp = strtoupper($platform);
        $platformWiseLandingPageUrl = $this->CluesConfigObj->getKeyValue('platform_wise_landing_page_url', 'payment', 'api');
        $failedURL = $platformWiseLandingPageUrl[$platformTemp]['returnurl_fail'];
        unset($platformTemp);
        return $failedURL;
    }

    /**
     * This method is uesd to get order failed url
     * @method getOrderSuccessUrl
     * @access protected
     * @param integer $orderId order id
     * @param integer $platform platform
     * @return string
     */
    protected function getOrderSuccessUrl($orderId, $platform) {
        if ($this->_isSoaApi) {
            $soaData = $this->getUrlsForRedirection($orderId);
            if (isset($soaData) && is_array($soaData) && isset($soaData[0]) &&
                    is_array($soaData[0]) && isset($soaData[0]['returnurl_success']) &&
                    !empty($soaData[0]['returnurl_success'])) {
                $successURL = $soaData[0]['returnurl_success'];
                return $successURL;
            }
        }
        $platformTemp = strtoupper($platform);
        $platformWiseLandingPageUrl = $this->CluesConfigObj->getKeyValue('platform_wise_landing_page_url', 'payment', 'api');
        $successUrl = $platformWiseLandingPageUrl[$platformTemp]['returnurl_success'];
        unset($platformTemp);
        return $successURL;
    }

    /**
     * Get form to be posted on pgw
     * @method postFormPgw
     * @access public
     * @param string $form 
     * @param object  $pgwObj
     * @param array $extraData array('convert_to_cod'=><convert_to_cod>, 'to_status' => <to_status>, 'otp_verified'=><otp_verified>)
     * @return String $form_assign or array depending upon $this->initiatePaymentReturnResponseArray
     */
    public function postFormPgw($form, $pgwObj = null, array $extraData = array()) {
        if ($form) {
            if (isset($form['payment_id']) && $form['payment_id'] == PgwModelConstants::Pgw2C2P_PAYMENT_ID && isset($form['form_array'][PgwModelConstants::PGW_DIRECT_VERIFY]) && !empty($form['form_array'][PgwModelConstants::PGW_DIRECT_VERIFY]) && $form['form_array'][PgwModelConstants::PGW_DIRECT_VERIFY]) {
                $this->logProcessTimeTaken(__CLASS__, PgwModelConstants::INITIATE_PAYMENT_METHOD_NAME, __FILE__);
                $this->atomModelStartTime = microtime(true);
                $extraData = $form;
                $extraData['order_id'] = $form['form_fields']['pgw_order_id'];
                if($this->initiatePaymentReturnResponseArray) {
                    $response = $this->saveResponseFromPGW($form['form_fields']['pgw_order_id'], $extraData, false, true, 0);
                    return $response;
                }
                $this->saveResponseFromPGW($form['form_fields']['pgw_order_id'], $extraData, true, false, 0);
            } else {
                if ($this->initiatePaymentReturnResponseArray) {
                    return $form['form_fields'];
                } else {
                    $form_assign = "<form";
                    foreach ($form["form_array"] as $key => $value) {
                        $form_assign .= " " . $key . "='" . $value . "'";
                    }
                    $form_assign .= ">";
                    if (!empty($form['form_fields'])) {
                        foreach ($form["form_fields"] as $key => $value) {
                            $form_assign .= "<input type='hidden' name='" . $key . "' value='" . $value . "'>";
                        }
                    }
                    $form_assign .= "<script type='text/javascript'> document." . $form["form_array"]["name"] . ".submit()</script></form>";
                    $this->logProcessTimeTaken(__CLASS__, PgwModelConstants::INITIATE_PAYMENT_METHOD_NAME, __FILE__);
                    echo $form_assign;
                    die;
                }
            }
        }
    }

    /**
     * This method is uesd to redirect user to order failed/success page
     * @method orderPageRedirection
     * @access protected
     * @param integer $orderId order id
     * @param integer $platform platform
     * @param mixed $status status
     * @param boolean $flag flag false returning response ,true for redirect
     * @param integer $momoeUserId
     * @return array
     */
    protected function orderPageRedirection($orderId, $platform, $status, $flag = true, $momoeUserId = 0) {
        if (isset($status) && !empty($status) && in_array($status, array('P'))) {
            $url = $this->getOrderSuccessUrl($orderId, $platform);
        } else {
            $url = $this->getOrderFailedUrl($orderId, $platform, $momoeUserId);
        }
        if ($flag) {
            $this->logProcessTimeTaken(__CLASS__, PgwModelConstants::SAVE_RESPONSE_FROM_PGW_METHOD_NAME, __FILE__, $this->atomModelStartTime);
            header('location:' . $url);
            exit();
        }
        $response = array('url' => $url, 'status' => $status);
        return $response;
    }

    /**
     * This method is used to initialise the pgw data
     * @method init
     * @access public
     * @param array $data
     * @return void
     */
    public function init($data = array()) {
        $this->payment_option_id = (isset($data['payment_option_id']) && is_numeric($data['payment_option_id']) && !empty($data['payment_option_id'])) ? $data['payment_option_id'] : 0;
        $this->platform = isset($data['platform']) ? $data['platform'] : '';
        $this->profileId = isset($data['profileId']) ? $data['profileId'] : 0;
        $this->paymentId = isset($data['payment_id']) ? $data['payment_id'] : 0;
        $this->emiId = isset($data['emi_id']) ? $data['emi_id'] : 0;
        $this->currentApiVersion = (isset($data['api_version']) && !empty($data['api_version'])) ? $data['api_version'] : 0;
        if (isset($data['response_in_json']) && (strtolower($data['response_in_json']) === "true" || (is_bool($data['response_in_json']) && $data['response_in_json']))) {
            $this->initiatePaymentReturnResponseArray = true;
        }
        $this->requestData = $this->decryptInitiatePaymentRequest();
        if(isset($this->requestData['card_info']['pup_card_token']) && !empty($this->requestData['card_info']['pup_card_token'])) {
            $this->requestData['card_info']['client_token'] = $this->requestData['card_info']['pup_card_token'];
        }
        $this->requestData = array_merge($data, $this->requestData);
        if (!empty($data)) {
            $this->inputParams = $data;
        } else {
            $this->inputParams = $this->requestData;
        }
        $this->requestConsInputParams = $this->inputParams;
        if (isset($this->requestData['order_id']) && !empty($this->requestData['order_id'])) {
            $this->orderId = $orderId = $this->requestData['order_id'];
        }
        $successUrl = isset($this->requestData['returnurl_success']) ? urldecode($this->requestData['returnurl_success']) : '';
        $failedUrl = isset($this->requestData['returnurl_fail']) ? urldecode($this->requestData['returnurl_fail']) : '';
        if (!empty($successUrl) && !empty($failedUrl) && ($this->_isSoaApi === false)) {
            $tempStatus = $this->saveUrlsApi($orderId, $successUrl, $failedUrl);
            if (!empty($tempStatus)) {
                $this->_isSoaApi = true;
            }
        }
        //common config
        $this->mobileDirConfig = $this->CluesConfigObj->getKeyValue('mobile_dir', 'payment');
        $this->deskDirConfig = $this->CluesConfigObj->getKeyValue('desk_dir', 'payment');
        $this->rootDirConfig = $this->CluesConfigObj->getKeyValue('ROOT_DIR', 'payment');
    }

    /**
     * To further process the response from payment gateway
     * @method saveResponseFromPGW
     * @access public
     * @param integer $orderId 
     * @param string $pgwResponse
     * @param boolean $redirect
     * @param boolean $retrnResponse
     * @return void
     */
    public function saveResponseFromPGW($orderId, $pgwResponse, $redirect = true, $retrnResponse = false) {
        if(empty($this->_isSoaApi)) {
            $this->_isSoaApi = (isset($this->requestData['soaUrl']) && !empty($this->requestData['soaUrl'])) ? true : false;
        }
        $logData = array('field1' => 'order_id: ' . $orderId, 'field2' => 'pgwResponse', 'error_name' => 'response_from_pgw', 'data' => $pgwResponse, 'method' => __METHOD__, 'line' => __LINE__);
        $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, 'clues_pre_payment', $logData, Logger::INFO);
        try {
            $paymentResponse = false;
            $orderInfo = $this->getPgwOrderInfo($orderId);
            $platformOrder = $orderInfo->platform;
            $this->orderInfo = $orderInfo;
            $paymentOptionId = isset($orderInfo->payment_option_id) ? $orderInfo->payment_option_id : 0;
            $paymentId = isset($orderInfo->payment->payment_id) ? $orderInfo->payment->payment_id : 0;
            $logData = array_merge($logData, array('field2' => 'payment_id', 'data' => $paymentId, 'line' => __LINE__));
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, 'clues_pre_payment', $logData, Logger::INFO);
            $clientId = !empty($orderInfo->clientID) ? $orderInfo->clientID :PgwModelConstants::DEFAULT_CLIENT_ID;
            $this->setAppClientIdDetails($clientId);
            $response = $this->getPGW($paymentId, $orderId);
            $pgwObj = $response['pgwObj'];
            if (empty($pgwObj) && $redirect) {
                $this->orderPageRedirection($this->orderInfo->order_id, $platformOrder, PgwModelConstants::FAILED_STATUS);
            }
            $pgwNotifyJson = $this->CluesConfigObj->getKeyValue('pgw_notify_api_config', 'payment', 'api');
            $notifyPgwConfigArray = json_decode($pgwNotifyJson, true);
            if (in_array($paymentId, $notifyPgwConfigArray['pgw'])) {
                $formattedResponse = $pgwObj->getFormattedResponse($pgwResponse, $orderInfo);
                $responseFromNotify = $this->checkNotifyStatusAndSavePostPaymentData($orderId, $formattedResponse, $paymentId, $paymentOptionId);
            }
            if (!isset($responseFromNotify) || $responseFromNotify['notify_status'] != 1) {
                $responseData = $pgwObj->saveResponse($orderInfo, $pgwResponse);
                $responseData['notify_status'] = isset($responseFromNotify['notify_status']) ? $responseFromNotify['notify_status'] : -1;
            } else if (isset($responseFromNotify) && isset($responseFromNotify['notify_status']) && $responseFromNotify['notify_status'] == 1) {
                $responseData = $responseFromNotify;
                $paymentResponse = true;
            }
            $isPaymentMarkSuccess = CluesRegistry::getObject(PgwModelConstants::PAYMENT_MARKED_SUCCESSFUL);
            if (isset($responseData['status']) && ($responseData['status'] == PgwModelConstants::PAID_STATUS) && empty($isPaymentMarkSuccess)) {
                $responseData['status'] = PgwModelConstants::FAILED_STATUS;
            }
            $toStatus = $responseData['status'];
            $logData = array_merge($logData, array('field2' => 'responseData', 'data' => $responseData, 'line' => __LINE__));
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, 'clues_pre_payment', $logData, Logger::INFO);
            if (!isset($responseData['notify_status']) || ($responseData['notify_status'] == 0 && $responseData['status'] == 'P') || $responseData['notify_status'] == -1) {
                $paymentResponse = true;// no need to change order status
            }
        } catch (\Exception $e) {
            $logData = array_merge($logData, array('field2' => 'errorMessage', 'data' => $e->getMessage(), 'line' => __LINE__));
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, 'clues_pre_payment', $logData, Logger::ERROR);
        }
        $status = ($toStatus == 'P' && $paymentResponse == true) ? 'P' : 'F';
        if ($this->_isSoaApi && isset($this->requestData['sc_return_response']) && !empty($this->requestData['sc_return_response'])) {
            $retrnResponse = $this->requestData['sc_return_response'];
            $redirect = !$retrnResponse;
        }
        if(!isset($pgwResponse['payment_id']) || empty($pgwResponse['payment_id'])) {
            $pgwResponse['payment_id'] = $this->paymentId;
        }
        $orderDetails = $this->paymentService->getPrepaymentData($pgwResponse['order_id'], $pgwResponse['payment_id']);
        $orderDetails['toStatus'] = $toStatus;
        $responseNotifyArray = $this->prepareAndInsertNotifyData($orderDetails);
        if(!empty($responseNotifyArray)) {
            DI::Map('ClientNotify', 'clues\\model\\pgwVerify\\ClientNotify');
            $ClientNotifyObj = DI::Singleton('ClientNotify');
            $notifyPendingData = $ClientNotifyObj->notifyPendingTransactions($orderDetails['client_order_id'], 0);
            unset($ClientNotifyObj);
        }
        if ($redirect) {
            $this->orderPageRedirection($this->orderInfo->order_id, $platformOrder, $status);
        } elseif ($retrnResponse) {
            $responseData = $this->orderPageRedirection($this->orderInfo->order_id, $platformOrder, $status, false);
            return $responseData;
        }
    }

    /**
     * this method is used to check if order already paid with notify api if yes then save payment response
     * @method checkNotifyStatusAndSavePostPaymentData
     * @access protected
     * @param integer $orderId
     * @param array $formattedResponseFromPgw
     * @param integer $paymentId
     * @param integer $paymentOptionId
     * @return array 
     */
    protected function checkNotifyStatusAndSavePostPaymentData($orderId, $formattedResponseFromPgw, $paymentId, $paymentOptionId) {
        $response = array("success" => 1, "order_id" => $orderId, "status" => "F");
        $existingNotifyResponse = $this->_checkExistingResponse($formattedResponseFromPgw['pgw_order_id'], $paymentId, $paymentOptionId);
        $response['notify_status'] = $existingNotifyResponse['status'];
        if ($response['notify_status'] == 1) {
            $notifyMetaData['prepayment_id'] = $existingNotifyResponse['prepayment_id'];
            $notifyMetaData['key'] = PgwModelConstants::PAYMENT_RESPONSE_KEY;
            $notifyMetaData['value'] = json_encode($formattedResponseFromPgw['prepayment_data']['other_details']);
            $notifyMetaData['status'] = 1;
            $notifyMetaData['created'] = $notifyMetaData['updated'] = time();
            $notifyMetaData['created_by'] = $notifyMetaData['updated_by'] = $this->orderInfo->user['user_id'];
            $this->paymentService->savePostPaymentMetaData($notifyMetaData);
            $response['status'] = 'P';
            CluesRegistry::addObject(PgwModelConstants::PAYMENT_MARKED_SUCCESSFUL, PgwModelConstants::PAYMENT_SUCCESS_STATUS);
        }
        return $response;
    }

    /**
     * this method is used to check notify response
     * @method _checkExistingResponse
     * @access private
     * @param string $pgwOrderId 
     * @param integer $paymentId 
     * @return array payment status
     */
    private function _checkExistingResponse($pgwOrderId, $paymentId) {
        $prepaymentData = $this->paymentService->getPrepaymentData($pgwOrderId, $paymentId);
        $response = array('status' => -1, 'prepayment_id' => 0);
        if (!empty($prepaymentData) && isset($prepaymentData['prepayment_details_auto_increment_id']) && !empty($prepaymentData['prepayment_details_auto_increment_id'])) {
            $paymentStatus = $this->paymentService->getPaymentStatusFromPrepaymentData($prepaymentData['prepayment_details_auto_increment_id']);
            $response['prepayment_id'] = $prepaymentData['prepayment_details_auto_increment_id'];
            $response['status'] = $paymentStatus;
        }
        return $response;
    }

    /**
     * This method is to set payment type id in given object
     * @method setPaymentTypeId
     * @access protected
     * @param object $orderInfo order info object
     * @return object updated object
     */
    protected function setPaymentTypeId(&$orderInfo) {
        if (!empty($orderInfo) && is_object($orderInfo) &&
                isset($orderInfo->payment_option_id) && !empty($orderInfo->payment_option_id)) {
            $tempPayOptionId = $orderInfo->payment_option_id;
            $tempArray = $this->paymentService->getAllPaymentOptionAndPaymentTypeHashArray();
            if (!empty($tempArray) && is_array($tempArray) && isset($tempArray[$tempPayOptionId]) &&
                    !empty($tempArray[$tempPayOptionId]) && isset($tempArray[$tempPayOptionId]['payment_type_id']) &&
                    !empty($tempArray[$tempPayOptionId]['payment_type_id'])) {
                $orderInfo->payment_type_id = $tempArray[$tempPayOptionId]['payment_type_id'];
            }
            unset($tempPayOptionId, $tempArray);
        }
        return $orderInfo;
    }

    /**
     * Log process time taken
     * @method logProcessTimeTaken
     * @access protected
     * @param string $class
     * @param string $method
     * @param string $file
     * @param float $customStartTime
     * @return void
     */
    protected function logProcessTimeTaken($class, $method, $file, $customStartTime = 0) {
        if (!empty($customStartTime) || defined('START_TIME')) {
            $explodedArray = explode('\\', get_class());
            $controllerName = end($explodedArray);
            $controller = !empty($controllerName) ? ucfirst($controllerName) : $class;
            $additionalInfo = array();
            $additionalInfo['field1'] = $this->RequestObj->getServerData('SERVER_NAME') . $this->RequestObj->getServerData('REQUEST_URI');
            $additionalInfo['field2'] = "Status Code:" . PgwModelConstants::PROCESS_TIME_OUTPUT_STATUS;
            $additionalInfo['field4'] = $controller . ':' . strtoupper($method);
            $additionalInfo['time_taken'] = $this->getTimeTaken($customStartTime);
            $additionalInfo['error_name'] = 'ResponseTime';
            $additionalInfo['additional_data']['file'] = $file;
            $additionalInfo['additional_data']['line'] = __LINE__;
            $additionalInfo['additional_data']['class'] = $class;
            $additionalInfo['additional_data']['method'] = $method;
            $additionalInfo['additional_data']['input_params'] = $this->RequestObj->getALLRequestData();
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME_API, PgwModelConstants::MODULE_NAME_API_DEBUGGING, $additionalInfo, Logger::INFO);
        }
    }

    /**
     * this method is used to prepare and insert notify data
     * @method prepareAndInsertNotifyData
     * @access protected
     * @param array $orderDetails ['orderId' => <oid>, 'toStatus' => <status>, 'paymentOptionId' => <opt_id>, 'paymentId' => <payment_id>]
     * @return string
     */
    protected function prepareAndInsertNotifyData(array $orderDetails) {
        $formDataJson = '';
        $notifyId = 0;
        DI::Map('Payment', 'clues\\model\\payment\\Payment');
        $PaymentObj = DI::Singleton('Payment');
        $paymentDetails = $PaymentObj->getPaymentDetailsByOrderId($orderDetails['client_order_id']);
        if (!empty($paymentDetails) && is_array($paymentDetails)) {
            $keys = PgwModelConstants::NOTIFY_DATA_KEYS;
            $paymentDetails = array_pop($paymentDetails);
            $tempDetails = [];
            foreach ($keys as $key) {
                if (!isset($paymentDetails[$key])) {
                    $tempDetails[$key] = '';
                } else {
                    $tempDetails[$key] = $paymentDetails[$key];
                    unset($paymentDetails[$key]);
                }
            }
            if (strtoupper($orderDetails['toStatus']) == 'F') {
                $tempDetails['payment_status'] = 0;
                $tempDetails['flag'] = 'failure';
                $tempDetails['udf_payment_type'] = $tempDetails['pgw_txn_id'] = $tempDetails['card_number'] = $tempDetails['pgw_order_id'] = $tempDetails['prepayment_details_auto_increment_id'] = $tempDetails['direcpayreferenceid'] = '';
            }
            $paymentDetails = $tempDetails;
        } else {
            $paymentTypeDetails = $this->paymentService->getPaymentTypeDetails($orderDetails['payment_option_id']);
            $pgwName = $this->paymentService->getPgwNameFromPaymentId($orderDetails['payment_id']);
            $paymentDetails = array(
                'order_id' => $orderDetails['client_order_id'],
                'client_order_id' => $orderDetails['client_order_id'],
                'payment_id' => $orderDetails['payment_id'],
                'payment_option_id' => $orderDetails['payment_option_id'],
                'payment_status' => 0,
                'payment_status_info' => 'Payment Not Received',
                'flag' => 'failure',
                'emi_id' => 0,
                'emi_name' => NULL,
                'status' => 1,
                'pgw_type' => $paymentTypeDetails['pgw_type'],
                'pgw_option_name' => $paymentTypeDetails['pgw_option_name'],
                'payment_type_id' => $paymentTypeDetails['payment_type_id'],
                'pgw' => $pgwName['pgw'],
            );
            $paymentDetails['created_by'] = $paymentDetails['updated_by'] = PgwModelConstants::ADMIN_USER_ID;
            $paymentDetails['created_timestamp'] = $paymentDetails['updated'] = time();
            $paymentDetails['created'] = date('d M Y H:i:s', time());
            foreach ($keys as $key) {
                if (!isset($paymentDetails[$key])) {
                    $paymentDetails[$key] = '';
                }
            }
        }
        $ipDetails = $this->_getIpDetailsForOrderId($paymentDetails['order_id'], $paymentDetails['payment_option_id']);
        $paymentDetails['request_ip'] = $ipDetails['request_ip'];
        $paymentDetails['response_ip'] = $ipDetails['response_ip'];
        $formDataJson = json_encode($paymentDetails);
        $notifyId = $this->paymentService->insertPupNotifyMapping($orderDetails, $formDataJson, PgwModelConstants::ADMIN_USER_ID, PgwModelConstants::ATOM_REDIRECT_SOURCE);
        return ['notifyData' => $formDataJson, 'notifyId' => $notifyId];
    }

    /**
     * get ip details for order id
     * @method _getIpDetailsForOrderId
     * @access private
     * @param integer $orderId
     * @param integer $paymentOptionId
     * @return array $ip ['request_ip' => <ip_addr>, 'response_ip' => <ip_addr>]
     */
    private function _getIpDetailsForOrderId($orderId, $paymentOptionId) {
        $ip = ['request_ip' => '', 'response_ip' => ''];
        $prepaymentId = 0;
        if (CluesRegistry::isKeySet(PgwModelConstants::REGISTRY_PREPAYMENT_ID)) {
            $prepaymentId = CluesRegistry::getObject(PgwModelConstants::REGISTRY_PREPAYMENT_ID);
        }
        $orderPgwId = $this->paymentService->getMappingData($orderId, $paymentOptionId, $prepaymentId);
        if (!empty($orderPgwId)) {
            $ip['request_ip'] = $this->paymentService->getPrePgwMetaData($orderPgwId, 'request_ip');
        }
        if (!empty($prepaymentId)) {
            $ip['response_ip'] = $this->paymentService->getPostPgwMetaData($prepaymentId, 'response_ip');
        } else {
            if (CluesRegistry::isKeySet(PgwModelConstants::PGW_RESPONSE_IP_ADDRESS)) {
                $ip['response_ip'] = CluesRegistry::getObject(PgwModelConstants::PGW_RESPONSE_IP_ADDRESS);
            }
            if (empty($ip['response_ip'])) { // ip empty in registry
                $ip['response_ip'] = $this->RequestObj->getIp();
            }
        }
        return $ip;
    }

    /**
     * set client_ip data in registry, if not present already
     * @method setIpInRegistry
     * @access protected
     * @param array $data
     * @return void
     */
    protected function setIpInRegistry(array $data) {
        if (CluesRegistry::isKeySet(PgwModelConstants::CLIENT_IP_FOR_SEND_TO_PGW)) {
            $existingIp = CluesRegistry::getObject(PgwModelConstants::CLIENT_IP_FOR_SEND_TO_PGW);
            if ($this->_checkValidIp($existingIp)) {
                return;
            }
        }
        $ip = (!empty($data) && isset($data['client_ip']) && !empty($data['client_ip']) && $this->_checkValidIp($data['client_ip'])) ? $data['client_ip'] : $this->RequestObj->getIp();
        if (!empty($ip)) {
            CluesRegistry::addObject(PgwModelConstants::CLIENT_IP_FOR_SEND_TO_PGW, $ip);
        }
    }

    /**
     * checks if ip address is a avlid ipv4/ipv6 address
     * @method _checkValidIp
     * @access private
     * @param string $ip
     * @return boolean
     */
    private function _checkValidIp($ip) {
        if (empty($ip)) {
            return false;
        }
        DI::Map('ClientIP', 'clues\\system\\ClientIP');
        $ClientIpObj = DI::Singleton('ClientIP');
        return $ClientIpObj->isIPv4($ip) OR $ClientIpObj->isIPv6($ip);
    }
}
