<?php

namespace clues\model\pgw;

use clues\system\util\StringOperations;
use clues\model\pgw\PgwBase;
use clues\model\pgw\PgwModelConstants;
use clues\system\Logger;
use clues\system\CluesRegistry;
use clues\model\pgw\exception\PgwException;
use clues\model\pgw\exception\PgwStatus;
use clues\system\util\JWT;
use clues\system\DI;

/**
 * This class is used to perform all common operations related to 2C2P pgw
 * @class Pgw2C2P
 * @access public
 * @package model/pgw
 * @category pgw
 * @note no hashing/encryption provided by 2C2P 
 */
class Pgw2C2P extends PgwBase {

    /**
     * Used for prepayment details
     * @var array
     */
    private static $_fieldsForPrepaymentDetails = [
        'referenceNo' => PgwModelConstants::PGW_REF_NO,
        'approvalCode' => PgwModelConstants::PGW_APPROVAL_CODE,
        'cardNo' => PgwModelConstants::CARD_NUMBER,
        'cardType' => PgwModelConstants::PG_CARD_TYPE,
        'cardToken' => PgwModelConstants::PG_CARD_TOKEN
    ];

    /**
     * Used to store paytm upi seamless order failed reason
     * @var string
     */
    private $_failedReason = null;

    /**
     * pgw mapping table notify single update required
     * @var boolean
     * @access protected
     */
    protected static $cppmNotifySingleUpdateRequired = true;

    /**
     * @method __construct
     * @access public
     * @return void
     */
    public function __construct(array $data = array()) {
        parent::__construct($data);
        if (empty($this->paymentGateway)) {
            $this->paymentGateway = PgwModelConstants::Pgw2C2P;
        }
    }

    /**
     * This method is used to get the data for the payment
     * @method getDataForRedirection
     * @access public
     * @return form array
     */
    public function getDataForRedirection() {
        $this->PaymentPgwMappingObj->addPgwOrderId($this, $this->pgwMappingDataArray);
        $orderData = $this->_getFormattedOrderData(); // to get payment token
        if(isset($orderData['pgw_request_response']) && !empty($orderData['pgw_request_response'])) {
            $requestResponse = $orderData['pgw_request_response'];
            unset($orderData['pgw_request_response']);
        }
        $metaDataArray = array();
        if(isset($orderData['pgw_2c2p_payment_response_token']) && !empty($orderData['pgw_2c2p_payment_response_token'])) {
            $metaDataArray['pgw_2c2p_payment_response_token'] = $orderData['pgw_2c2p_payment_response_token'];
            unset($orderData['pgw_2c2p_payment_response_token']);
        }
        $pgw2c2pVerify = 0;
        if(isset($orderData[PgwModelConstants::PGW_DIRECT_VERIFY]) && !empty($orderData[PgwModelConstants::PGW_DIRECT_VERIFY]) && $orderData[PgwModelConstants::PGW_DIRECT_VERIFY]) {
            $pgw2c2pVerify = 1;
            unset($orderData[PgwModelConstants::PGW_DIRECT_VERIFY]);
        }
        $prepaymentData = $this->_getPaymentDataForOrder($orderData);
        $orderPgwId = $this->savePrePaymentData($prepaymentData, $metaDataArray);
        if(isset($requestResponse) && !empty($requestResponse)) {
            $this->paymentService->updatePgwRequestResponseOrderPgw(json_encode($requestResponse), $orderPgwId);
            unset($requestResponse, $orderPgwId);
        }
        if(isset($orderData['pgw_2c2p_redirect_url']) && !empty($orderData['pgw_2c2p_redirect_url'])) {
            $targetUrl = $orderData['pgw_2c2p_redirect_url'];
        }
        $formFields = $this->_getFormFields($orderData);
        $formArray = array('name' => $this->paymentGateway, 'method' => 'post', 'action' => $targetUrl);
        if($pgw2c2pVerify) {
            $formArray[PgwModelConstants::PGW_DIRECT_VERIFY] = $pgw2c2pVerify;
        }
        return array('payment_id' => $this->paymentId, 'form_array' => $formArray, 'form_fields' => $formFields);
    }

    /**
     * This method is used to make new pgw order id
     * @method getPgwOrderId
     * @access public
     * @param boolean $createNew 
     * @return integer pgwOrderId
     */
    public function getPgwOrderId($createNew = true) {
        $tempRefId = '';
        for ($retryCount = 1; $retryCount <= 10; $retryCount++) {
            $tempRefId = 'P' . $retryCount . StringOperations::getMd5UniqueId('pup');
            $tempRefId = strtoupper($tempRefId);
            $existCount = $this->paymentService->getScOrderIdByPgwOrderId($tempRefId);
            if (empty($existCount)) {
                break;
            }
        }
        if (empty($tempRefId)) {
            $tempRefId = $this->orderId . 'p' . time() . mt_rand(10000, 99999);
        }
        $this->pgwOrderId = $tempRefId;
        $this->setPgwMappingPreAdditionalData();
        return $this->pgwOrderId;
    }

    /**
     * This method is used to get payment token and data
     * @method _getFormattedOrderData
     * @access private
     * @return $orderData
     */
    private function _getFormattedOrderData() {
        $orderData = array('pgw' => PgwModelConstants::Pgw2C2P, 'order_id' => $this->orderInfo->order_id, 'pgw_order_id' => $this->pgwOrderId);
        $pgwRequestResponse = $PT_resData = $paymentChannelCodes = array();
        $paymentChannelCodes = $this->paymentService->getPaymentChannelInfo($this->orderInfo->payment_option_id, $this->orderInfo->payment->payment_id);
        if(isset($this->orderInfo->card_info) && !empty($this->orderInfo->card_info) && isset($this->orderInfo->card_info['client_token']) && !empty($this->orderInfo->card_info['client_token'])) {
            $cardTokenInfo = $this->paymentService->getCardInfoFromToken($this->orderInfo->card_info['client_token'], $this->userId);
            if(isset($cardTokenInfo['card_token']) && !empty($cardTokenInfo['card_token'])) {
                $this->orderInfo->card_info['card_token'] = $cardTokenInfo['card_token'];
            }
            /*else if(isset($cardTokenInfo['secure_pay_token']) && !empty($cardTokenInfo['secure_pay_token'])) {
                $this->orderInfo->card_info['secure_pay_token'] = $cardTokenInfo['secure_pay_token'];
            }*/
            else {
                $exceptionData = array();
                $exceptionData['error_name'] = 'Invalid token';
                $additionalInfo = array();
                $additionalInfo['file'] = __FILE__;
                $additionalInfo['line'] = __LINE__;
                $additionalInfo['module'] = PgwModelConstants::MODULE_NAME;
                $additionalInfo['domain'] = PgwModelConstants::DOMAIN_NAME;
                $additionalInfo['level'] = Logger::ERROR;
                $additionalInfo['method'] = __METHOD__;
                $exceptionData['additional_data'] = $additionalInfo;
                throw new PgwException(PgwStatus::INVALID_TOKEN_OR_CARD_EXPIRED, $exceptionData);
            }
        }
        $PT_resData = $this->_getPaymentToken();
        $paymentToken = (isset($PT_resData['respCode']) && !empty($PT_resData['respCode']) && $PT_resData['respCode'] = PgwModelConstants::Pgw2C2P_PAYMENT_TOKEN_SUCCESS_CODE && isset($PT_resData['paymentToken']) && !empty($PT_resData['paymentToken'])) ? $PT_resData['paymentToken'] : '';
        $pgwRequestResponse['paymentTokenApi'] = isset($PT_resData['paymentTokenApi']) && !empty($PT_resData['paymentTokenApi']) ? $PT_resData['paymentTokenApi'] : array();
        if(!empty($paymentToken)) {
            $orderData['pgw_2c2p_payment_response_token'] = $paymentToken;
            $paymentTokenRes = $this->_getPaymentLink($paymentToken, $paymentChannelCodes);
            $pgwRequestResponse['paymentApi'] = isset($paymentTokenRes['paymentApi']) && !empty($paymentTokenRes['paymentApi']) ? $paymentTokenRes['paymentApi'] : array();
            $orderData['pgw_2c2p_redirect_url'] = isset($paymentTokenRes['pgw_2c2p_redirect_url']) && !empty($paymentTokenRes['pgw_2c2p_redirect_url']) ? $paymentTokenRes['pgw_2c2p_redirect_url'] : '';
            $orderData[PgwModelConstants::PGW_DIRECT_VERIFY] = isset($paymentTokenRes[PgwModelConstants::PGW_DIRECT_VERIFY]) && !empty($paymentTokenRes[PgwModelConstants::PGW_DIRECT_VERIFY]) ? $paymentTokenRes[PgwModelConstants::PGW_DIRECT_VERIFY] : 0;
        } else {
            $logData = array();
            $logData['field1'] = 'order_id:' . $this->orderInfo->order_id;
            $logData['field2'] = 'user_id:' . $this->orderInfo->user['user_id'];
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
            $logData['error_name'] = 'empty_token_from_2C2P_payment_token_api';
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::ERROR;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            $pgwRequestResponse['paymentApi'] = 'empty_token_from_2C2P_payment_token_api';
            throw new PgwException(PgwStatus::EMPTY_PAYMENT_TOKEN_FROM_PGW, $logData);
            }
        $orderData['pgw_request_response'] = $pgwRequestResponse;
        return $orderData;
    }

    /**
     * This method is used to get payment token
     * @method _getPaymentToken
     * @access private
     * @return $response
     */
    private function _getPaymentToken() {
        $response = array();
        $updatedRedirectUrl = '&order_id='.  $this->pgwOrderId . '&pgwToken='; // 2c2p appends alphanumber number at the end which is payment token, so added a keyword pgwToken for this purpose
        $this->updateRedirectUrl($updatedRedirectUrl);
        try {
            $paymentTokenDataArray = array (
                'merchantID' => $this->orderInfo->payment->params['merchantID'], //Unique merchant ID that is registered with 2C2P, Type-Alphanumeric
                'invoiceNo' => $this->pgwOrderId, //Unique merchant order number, Type-Alphanumeric
                'description' => isset($this->orderInfo->products_desciption) && !empty($this->orderInfo->products_desciption) ? $this->orderInfo->products_desciption : "product desc", //Product description. HTML Encode is required if it contains special characters, Type-Characters
                'amount' => $this->orderInfo->total, //Example: 000000002500.90000, Type-Decimal
                'currencyCode' => $this->orderInfo->payment->params['paymentTokenCurrencyCode'], //Transaction currency code in 3 alphabetical values as specified in ISO 4217. If the value is empty, the system will use the merchant's base currency, Type-Alphabet
                'backendReturnUrl' => $this->orderInfo->payment->params['notifyUrl'], //Specify backend url to be notified after payment is completed via 2C2P
                'frontendReturnUrl' => $this->redirecturl, //Specify the return url to be redirected to merchant after payment is completed via 2C2P
                'request3DS' => 'N'
            );
            if(isset($this->orderInfo->card_info) && !empty($this->orderInfo->card_info) && isset($this->orderInfo->card_info['request3DS']) && !empty($this->orderInfo->card_info['request3DS'])) {
                $paymentTokenDataArray['request3DS'] = $this->orderInfo->card_info['request3DS'];
            }
            if(isset($this->orderInfo->card_info) && !empty($this->orderInfo->card_info) && isset($this->orderInfo->card_info['card_token']) && !empty($this->orderInfo->card_info['card_token'])) {
                $paymentTokenDataArray['cardTokens'] = (array) $this->orderInfo->card_info['card_token'];
            }
            $logData = array();
            $logData['field1'] = 'order_id:' . $this->orderInfo->order_id;
            $logData['field2'] = 'user_id:' . $this->orderInfo->user['user_id'];
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
            $logData['error_name'] = '2C2P_payment_token_api';
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::INFO;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::MODULE_NAME, $logData, Logger::INFO);
            $paymentTokenUrl = $this->orderInfo->payment->params['baseUrl'] . $this->orderInfo->payment->params['paymentTokenApiUrl'];
            $reqPayloadData = JWT::createJWTPayload($paymentTokenDataArray, $this->orderInfo->payment->params['secretKey']);
            $reqPayloadArray = array(
                'payload' => $reqPayloadData
            );
            $paymentTokenPayload = json_encode($reqPayloadArray, JSON_UNESCAPED_SLASHES);
            $paymentTokenResData = $this->_getCurlResponse($paymentTokenUrl, $paymentTokenPayload);
            if(isset($paymentTokenResData['payload']) && !empty($paymentTokenResData['payload'])) {
                $response = JWT::processJWTPayload($paymentTokenResData['payload'], $this->orderInfo->payment->params['secretKey']);
            }
            $response['paymentTokenApi'] = array('request' => $paymentTokenPayload, 'response' => json_encode($paymentTokenResData));
        } catch (\Exception $e) {
            $logData = array();
            $logData['field1'] = 'order_id:' . $this->orderInfo->order_id;
            $logData['field2'] = 'user_id:' . $this->orderInfo->user['user_id'];
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
            $logData['error_name'] = $e->getMessage();
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::ERROR;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            throw new PgwException(PgwStatus::PAYMENT_TOKEN_API_ERROR, $logData);
        }
        return $response;
    }
    /**
     * This method is used to get payment link
     * @method _getPaymentLink
     * @access private
     * @param string $paymentToken
     * @param array $paymentChannelCodes
     * @return $response
     */
    private function _getPaymentLink($paymentToken, $paymentChannelCodes = array()) {
        $response = array();
        if(empty($paymentToken)) {
            return $response;
        }
        try {
            $paymentDataArray = array (
                'paymentToken' => $paymentToken,
                'locale' => 'en',
                'responseReturnUrl' => $this->redirecturl,
                'payment' => array(
                    'code' => array(
                        'channelCode' => $paymentChannelCodes['product_code']
                    )
                )
            );
            if(isset($paymentChannelCodes['bank_code']) && !empty($paymentChannelCodes['bank_code'])) {
                $paymentDataArray['payment']['code']['agentCode'] = $paymentChannelCodes['bank_code'];
            }
            if(isset($paymentChannelCodes['sub_product_code']) && !empty($paymentChannelCodes['sub_product_code'])) {
                $paymentDataArray['payment']['code']['agentChannelCode'] = $paymentChannelCodes['sub_product_code'];
            }
            if(isset($this->orderInfo->card_info) && !empty($this->orderInfo->card_info)) {
                if(isset($this->orderInfo->user['name']) && !empty($this->orderInfo->user['name'])) {
                    $paymentDataArray['payment']['data']['name'] = $this->orderInfo->user['name'];
                }
                if(isset($this->orderInfo->user['email']) && !empty($this->orderInfo->user['email'])) {
                    $paymentDataArray['payment']['data']['email'] = $this->orderInfo->user['email'];
                }
                if(isset($this->orderInfo->user['mobile']) && !empty($this->orderInfo->user['mobile'])) {
                    $paymentDataArray['payment']['data']['mobileNo'] = $this->orderInfo->user['mobile'];
                }
                /*if(isset($this->orderInfo->card_info['secure_pay_token']) && !empty($this->orderInfo->card_info['secure_pay_token'])) {
                    $paymentDataArray['payment']['data']['securePayToken'] = $this->orderInfo->card_info['secure_pay_token'];
                    $paymentDataArray['payment']['data']['cardTokenize'] = true;
                } else */
                if(isset($this->orderInfo->card_info['card_token']) && !empty($this->orderInfo->card_info['card_token'])) {
                    $paymentDataArray['payment']['data']['token'] = $this->orderInfo->card_info['card_token'];
                    $paymentDataArray['payment']['data']['cardTokenize'] = false;
                }
            }
            $paymentDataArray = (object) array_filter((array) $paymentDataArray);
            $logData = array();
            $logData['field1'] = 'order_id:' . $this->orderInfo->order_id;
            $logData['field2'] = 'user_id:' . $this->orderInfo->user['user_id'];
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
            $logData['error_name'] = '2C2P_payment_api';
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::INFO;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::MODULE_NAME, $logData, Logger::INFO);
            $paymentData = json_encode($paymentDataArray, JSON_UNESCAPED_SLASHES);
            $paymentUrl = $this->orderInfo->payment->params['baseUrl'] . $this->orderInfo->payment->params['paymentApiUrl'];
            $paymentResData = $this->_getCurlResponse($paymentUrl, $paymentData);
            $response['paymentApi'] = array('request' => $paymentData, 'response' => json_encode($paymentResData));
            if(isset($paymentResData['respCode']) && !empty($paymentResData['respCode']) && in_array($paymentResData['respCode'], array('1001')) && isset($paymentResData['data']) && !empty($paymentResData['data']) && filter_var($paymentResData['data'], FILTER_VALIDATE_URL)) {
                $response['pgw_2c2p_redirect_url'] = $paymentResData['data'];
            } else {
                $response[PgwModelConstants::PGW_DIRECT_VERIFY] = 1;
            }
        } catch (\Exception $e) {
            $logData = array();
            $logData['field1'] = 'order_id:' . $this->orderInfo->order_id;
            $logData['field2'] = 'user_id:' . $this->orderInfo->user['user_id'];
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
            $logData['error_name'] = $e->getMessage();
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::ERROR;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            throw new PgwException(PgwStatus::PAYMENT_API_ERROR, $logData);
        }
        return $response;
    }

    /**
     * This method is used to get payment data the payment
     * @method _getPaymentDataForOrder
     * @access private
     * @param $orderData
     * @return array $insertArray
     * @throws ParameterMissing
     */
    private function _getPaymentDataForOrder($orderData) {
        $registryKey = $this->getOrderInfoRegistryKey($this->orderInfo->order_id);
        if (CluesRegistry::isKeySet($registryKey)) {
            $orderData = CluesRegistry::getObject($registryKey);
        }
        $insertArray = array('order_id' => $this->orderInfo->order_id, 'amount' => $this->orderInfo->total, 'payment_gateway' => $this->paymentGateway, 'order_data' => $orderData);
        return $insertArray;
    }

    /**
     * This method is used to get the form
     * @method _getFormFields
     * @access private
     * @param $orderData
     * @return array $formFields
     * @throws ParameterMissing
     */
    private function _getFormFields($orderData) {
        $formFields = array();
        if (isset($orderData['pgw']) && !empty($orderData['pgw'])) {
            $formFields['pgw'] = $orderData['pgw'];
        }
        if (isset($orderData['order_id']) && !empty($orderData['order_id'])) {
            $formFields['order_id'] = $orderData['order_id'];
        }
        if (isset($orderData['pgw_order_id']) && !empty($orderData['pgw_order_id'])) {
            $formFields['pgw_order_id'] = $orderData['pgw_order_id'];
        }
        if (isset($orderData['pgw_2c2p_redirect_url']) && !empty($orderData['pgw_2c2p_redirect_url'])) {
            $formFields['pgw_2c2p_redirect_url'] = $orderData['pgw_2c2p_redirect_url'];
        }
        return $formFields;
    }

    /**
     * This method is used to get formatted notify response
     * @method getFormattedResponse
     * @access public
     * @param array $notifyResponse
     * @return response array
     * @throws exception
     */
    public function getFormattedResponse($notifyResponse) {
        $responseArray = $orderInfo = array();
        if(isset($notifyResponse['payload']) && !empty($notifyResponse['payload'])) {
            $notifyResponse = json_decode(base64_decode(explode('.', $notifyResponse['payload'])[1]), true);
        }
        if(isset($notifyResponse['order_id']) && !empty($notifyResponse['order_id'])) {
            $orderInfo = $this->getPgwOrderInfo($notifyResponse['order_id']);
        }
        if(empty($orderInfo)) {
            return $responseArray;
        }
        $this->orderInfo = $orderInfo;
        $this->txnRs = $notifyResponse;
        $this->setPgwMappingPostAdditionalData($notifyResponse['order_id']);
        $verifyResponse = $this->txnRs = $this->_verifyResponse($orderInfo, $notifyResponse['order_id']);
        $responseArray['pgw_order_id'] = $this->pgwOrderId;
        $responseArray['prepayment_data'] = $this->_getResponseArray($notifyResponse, 'notify');
        if (in_array($verifyResponse['respCode'], array('0000')) && $verifyResponse['amount'] >= $orderInfo->total && $verifyResponse['invoiceNo'] == $this->pgwOrderId) {
            $this->_failedReason = '';
        } else if (!in_array($verifyResponse['respCode'], array('0000'))) {
            $this->_failedReason = 'Wrong status';
        } else if ($verifyResponse['amount'] < $orderInfo->total) {
            $this->_failedReason = 'Less amt received';
        } else if ($verifyResponse['invoiceNo'] != $this->pgwOrderId) {
            $this->_failedReason = 'Pgw order id mismatch';
        }
        if (!empty($this->_failedReason)) {
            $responseArray['prepayment_data']['flag'] = $this->_failedReason;
        }
        return $responseArray;
    }

    /**
     * Save response from pgw after payment
     * @method saveResponse
     * @param object $orderInfo order info object
     * @param mixed $txnRs pgw transaction response
     * @access public
     * @return array
     */
    public function saveResponse($orderInfo, $txnRs) {
        $response = array();
        try {
            $this->orderInfo = $orderInfo;
            $this->txnRs = $txnRs;
            $this->setPgwMappingPostAdditionalData($txnRs['order_id']);
            $verifyResponse = $this->txnRs = $this->_verifyResponse($orderInfo, $txnRs['order_id']);
            $insertArray = $this->_getResponseArray($txnRs, 'redirect');
            $metaDataArray = array('payment_status' => 0);
            $responseData = array('success' => 0, 'order_id' => $orderInfo->order_id, 'message' => $verifyResponse['respDesc'], 'status' => PgwModelConstants::FAILED_STATUS);
            if (in_array($verifyResponse['respCode'], array('0000')) && $verifyResponse['amount'] >= $orderInfo->total && $verifyResponse['invoiceNo'] == $this->pgwOrderId) {
                $this->_failedReason = '';
                $responseData = array('success' => 1, 'order_id' => $orderInfo->order_id, 'message' => $verifyResponse['respDesc'], 'status' => PgwModelConstants::PAID_STATUS);
                $metaDataArray['payment_status'] = 1;
            } else if (!in_array($verifyResponse['respCode'], array('0000'))) {
                $this->_failedReason = 'Wrong status';
            } else if ($verifyResponse['amount'] < $orderInfo->total) {
                $this->_failedReason = 'Less amt received';
            } else if ($verifyResponse['invoiceNo'] != $this->pgwOrderId) {
                $this->_failedReason = 'Pgw order id mismatch';
            }
            if (!empty($this->_failedReason)) {
                $insertArray['flag'] = $this->_failedReason;
            }
            $this->setPgwTxnData($insertArray['other_details'], $metaDataArray);
            $this->savePostPaymentData($insertArray, $metaDataArray);
        } catch (Exception $e) {
            $logData = array();
            $logData['field1'] = 'order_id:' . $orderInfo->order_id;
            $logData['field2'] = 'user_id:' . $orderInfo->user['user_id'];
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
            $logData['error_name'] = $e->getMessage();
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::ERROR;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::MODULE_NAME, $logData, Logger::ERROR);
        }
        return $responseData;
    }

    /**
     * This method is used to get response array
     * @method _getResponseArray
     * @access private
     * @param array $notifyResponse
     * @param string $source
     * @return insert array
     */
    private function _getResponseArray(array $notifyResponse, $source = 'notify') {
        $insertArray = array();
        $insertArray['direcpayreferenceid'] = (isset($this->txnRs['tranRef']) && !empty($this->txnRs['tranRef'])) ? $this->txnRs['tranRef'] : 0;
        $insertArray['order_id'] = $this->orderInfo->order_id;
        $insertArray['flag'] = isset($this->txnRs['respDesc']) ? $this->txnRs['respDesc'] : '';
        $insertArray['other_details'] = array($source => $notifyResponse, 'verify' => $this->txnRs);
        $insertArray['amount'] = isset($this->txnRs['amount']) ? $this->txnRs['amount'] : 0;
        $insertArray['payment_gateway'] = !empty($this->paymentGateway) ? $this->paymentGateway : PgwModelConstants::Pgw2C2P;
        $insertArray['status_code'] = isset($this->txnRs['respCode']) ? $this->txnRs['respCode'] : NULL;
        return $insertArray;
    }

    /**
     * This method is used to verify Response of notify
     * @method _verifyResponse
     * @access private
     * @param object $orderInfo
     * @param string $invoiceNo
     * @return array $responsePayloadData
     * @throws exception
     */
    private function _verifyResponse($orderInfo, $invoiceNo) {
        $responsePayloadData = array();
        try {
            $paymentInquiryDataArray = array (
                'merchantID' => $orderInfo->payment->params['merchantID'], //Unique merchant ID that is registered with 2C2P, Type-Alphanumeric
                'invoiceNo' => $invoiceNo, //Unique merchant order number, Type-Alphanumeric
                'locale' => 'en'
            );
            $logData = array();
            $logData['field1'] = 'order_id:' . $orderInfo->order_id;
            $logData['field2'] = 'user_id:' . $orderInfo->user['user_id'];
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
            $logData['error_name'] = '2C2P_payment_inquiry_api';
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::INFO;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::MODULE_NAME, $logData, Logger::INFO);
            $paymentInquiryUrl = $orderInfo->payment->params['baseUrl'] . $orderInfo->payment->params['paymentInquiryApiUrl'];
            $paymentInquiryPayload = JWT::createJWTPayload($paymentInquiryDataArray, $orderInfo->payment->params['secretKey']);
            $paymentInquiryPayloadArray = array(
                'payload' => $paymentInquiryPayload
            );
            $paymentInquiryPayload = json_encode($paymentInquiryPayloadArray, JSON_UNESCAPED_SLASHES);
            $paymentInquiryResData = $this->_getCurlResponse($paymentInquiryUrl, $paymentInquiryPayload);
            if(isset($paymentInquiryResData['payload']) && !empty($paymentInquiryResData['payload'])) {
                $responsePayloadData = JWT::processJWTPayload($paymentInquiryResData['payload'], $orderInfo->payment->params['secretKey']);
                if(isset($responsePayloadData['cardNo']) && !empty($responsePayloadData['cardNo'])) {
                    $responsePayloadData['cardNo'] = 'XXXXXXXXXXXX' . substr($responsePayloadData['cardNo'], -4);
                }
                if(isset($responsePayloadData['cardToken']) && !empty($responsePayloadData['cardToken']) && isset($orderInfo->card_info) && !empty($orderInfo->card_info)) {
                    $clientToken = (isset($orderInfo->card_info['client_token']) && !empty($orderInfo->card_info['client_token'])) ? $orderInfo->card_info['client_token'] : '';
                    // $securePayToken = (isset($orderInfo->card_info['secure_pay_token']) && !empty($orderInfo->card_info['secure_pay_token'])) ? $orderInfo->card_info['secure_pay_token'] : '';
                    if($clientToken) {
                        $currentTokenIncId = $this->paymentService->getDataForCurrentToken($clientToken);
                        $updatedCardToken = $this->paymentService->updateCardToken($responsePayloadData['cardToken'], $clientToken, $orderInfo->user['user_id']);
                        if($updatedCardToken) {
                            DI::Map('CluesHistory', 'clues\\system\\CluesHistory');
                            $cluesHistoryObj = DI::Singleton('CluesHistory');
                            $cluesHistoryObj->insertIntoHistoryTable(PgwModelConstants::USER_CARD_INFO_TABLE, PgwModelConstants::USER_CARD_INFO_HISTORY_TABLE, 'id', $currentTokenIncId);
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            $logData = array();
            $logData['field1'] = 'order_id:' . $orderInfo->order_id;
            $logData['field2'] = 'user_id:' . $orderInfo->user['user_id'];
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwModelConstants::DOMAIN_NAME;
            $logData['error_name'] = $e->getMessage();
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwModelConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwModelConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::ERROR;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::MODULE_NAME, $logData, Logger::ERROR);
        }
        return $responsePayloadData;
    }

    /**
     * This method is used to get response data
     * @method _getCurlResponse
     * @param string $url 
     * @param array $params 
     * @param array $header
     * @access private
     * @return array $responseArray
     */
    private function _getCurlResponse($url, $params, array $header = array()) {
        $headerArray = array_merge(array('Content-Type: application/*+json'), $header);
        if (is_array($params)) {
            $paramsString = json_encode($params);
        } else {
            $paramsString = $params;
        }
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headerArray);
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $paramsString);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        $responseArray = isset($response) && !empty($response) ? json_decode($response, true) : array();
        return $responseArray;
    }

    /**
     * This method is used to get notify response status
     * @method getNotifyResponseStatus
     * @access public
     * @param integer $orderTotal
     * @return integer
     */
    public function getNotifyResponseStatus($orderTotal) {
        if (empty($this->_failedReason) && isset($this->txnRs['respCode']) && !empty($this->txnRs['respCode']) && in_array($this->txnRs['respCode'], array('0000')) && $this->txnRs['amount'] >= $orderTotal) {
            return 1;
        }
        return 0;
    }

    /**
     * This method is used to set pgw txn data
     * @method setPgwTxnData
     * @access public
     * @param mixed $response pgw response
     * @param array $setArray reference array
     * @return void
     */
    public function setPgwTxnData($response, array &$setArray) {
        if (!isset($response['verify']) || empty($response['verify']) || !is_array($response['verify'])) {
            return;
        }
        $response = $response['verify'];
        foreach (self::$_fieldsForPrepaymentDetails as $prepaymentKey => $value) {
            if (isset($response[$prepaymentKey]) && !empty($response[$prepaymentKey])) {
                $setArray[$value] = $response[$prepaymentKey];
            }
        }
    }
}
