<?php

namespace clues\model\pgw;

use clues\model\pgw\PGWDriver;
use clues\system\DI;
use clues\model\pgw\PgwModelConstants;
use clues\system\Logger;
use clues\system\exception\Status;
use clues\system\util\StringOperations;
use clues\system\CluesRegistry;

/**
 * This class is used to perform all common operations related to payment
 * @class PgwBase
 * @access public
 * @package model/pgw
 * @category pgw
 */
abstract class PgwBase extends PGWDriver {

    /**
     * orderId
     * @var array 
     */
    public $orderId;

    /**
     * pgwOrderId
     * @var integer 
     */
    public $pgwOrderId;

    /**
     * userId
     * @var integer 
     */
    public $userId;

    /**
     * paymentId
     * @var integer 
     */
    public $paymentId;

    /**
     * paymentOptionId
     * @var integer 
     */
    public $paymentOptionId;

    /**
     * orderPgwPrimaryId
     * @var integer
     */
    public $orderPgwPrimaryId;

    /**
     * prepaymentPgwPrimaryId
     * @var integer 
     */
    public $prepaymentPgwPrimaryId;

    /**
     * pgwMappingDataArray
     * @var array 
     */
    public $pgwMappingDataArray;

    /**
     * redirecturl
     * @var srtring 
     */
    protected $redirecturl;

    /**
     * status
     * @var integer 
     */
    protected $status;

    /**
     * checksum
     * @var string 
     */
    protected $checksum;

    /**
     * txnresponse
     * @var array 
     */
    protected $txnresponse;

    /**
     * return_checksum
     * @var string 
     */
    protected $returnChecksum;

    /**
     * request_message
     * @var array 
     */
    protected $requestMessage;

    /**
     * txnRs
     * @var array 
     */
    protected $txnRs;

    /**
     * cod_url
     * @var array 
     */
    protected $codUrl;

    /**
     * emiId
     * @var integer 
     */
    protected $emiId;

    /**
     * Amount
     * @var integer 
     */
    protected $amount;

    /**
     * Payment Gateway
     * @var string 
     */
    protected $paymentGateway;

    /**
     * Payment Gateway constructor request data
     * @var array 
     */
    protected $pgwReqData;

    /**
     * Payment gateway merchant params
     * @var array
     */
    protected $pgwMerchantParams;

    /**
     * payment object
     * @var object
     */
    protected $PaymentObj;

    /**
     * Set pgw names array
     * @access private
     * @var array
     */
    private $_paymentIdPaymentOptionIdPgwName = array();

    /**
     * Set Pgw Error info 
     * @access protected
     * @var mixed:array|boolean
     */
    protected $errorInfo;

    /**
     *
     * @var type Override Fields for pre payment details
     * @var static array
     * @access protected
     */
    protected static $overridePPDFields = array(
        'direcpayreferenceid',
        'flag',
        'amount',
        'status_code'
    );

    /**
     * PGWHashManager Object
     * @var object 
     */
    public $PGWHashManager;

    /**
     * pgw mapping table notify single update required
     * @var boolean
     * @access protected
     */
    protected static $cppmNotifySingleUpdateRequired = false;

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @return void
     */
    public function __construct(array $data = array()) {
        parent::__construct($data);
        $this->pgwReqData = $data;
        DI::Map('PaymentObj', 'clues\\model\\payment\\Payment');
        $this->PaymentObj = DI::Singleton('PaymentObj');
        $this->pgwOrderId = $this->orderId;
        $this->setPgwMappingPreAdditionalData();
        DI::Map('PGWHashManager', 'clues\\model\\pgw\\PGWHashManager\\PGWHashManager');
        $this->PGWHashManager = DI::Singleton('PGWHashManager');
    }

    /**
     * This method is used to make new pgw order id
     * @method getPgwOrderId
     * @access public
     * @param boolean $createNew 
     * @return integer pgworderid
     */
    public function getPgwOrderId($createNew = true) {
        if ($createNew) {
            $time = time();
            $randomNumber = mt_rand(10000, 99999);
            $this->pgwOrderId = $this->orderId . 'sc' . $time . $randomNumber;
        } else {
            $this->pgwOrderId = $this->orderId;
        }
        $this->setPgwMappingPreAdditionalData();
        return $this->pgwOrderId;
    }

    /**
     * This method is used to set parameters for pgw
     * @method setPgwMappingPreAdditionalData
     * @access public
     * @return void
     */
    public function setPgwMappingPreAdditionalData() {
        if(!empty($this->orderInfo)) {
            $this->orderId = isset($this->orderInfo->order_id) ? $this->orderInfo->order_id : 0;
            $this->paymentId = isset($this->orderInfo->payment->payment_id) ? $this->orderInfo->payment->payment_id : 0;
            $this->paymentOptionId = isset($this->orderInfo->payment_option_id) ? $this->orderInfo->payment_option_id : 0;
            if (empty($this->userId)) {
                if (isset($this->orderInfo->user) && isset($this->orderInfo->user['user_id'])) {
                    $this->userId = $this->orderInfo->user['user_id'];
                } else {
                    $this->userId = 0;
                }
            }
            $this->paymentGateway = $this->getPgwName(array('payment_option_id' => $this->paymentOptionId, 'payment_id' => $this->paymentId));
            $this->emiId = isset($this->orderInfo->emi_id) ? $this->orderInfo->emi_id : 0;
            $this->pgwMappingDataArray['orderId'] = $this->orderId;
            $this->pgwMappingDataArray['paymentId'] = $this->paymentId;
            $this->pgwMappingDataArray['userId'] = $this->userId;
            $this->pgwMappingDataArray['paymentOptionId'] = $this->paymentOptionId;
            $this->pgwMappingDataArray['pgwOrderId'] = $this->pgwOrderId;
            $this->pgwMappingDataArray['orderPgwPrimaryId'] = $this->orderPgwPrimaryId;
            $this->pgwMappingDataArray['prepaymentPgwPrimaryId'] = $this->prepaymentPgwPrimaryId;
            $this->pgwMappingDataArray['emiId'] = $this->emiId;
            $this->pgwMappingDataArray['clientID'] = $this->clientID;
            $this->pgwMappingDataArray['amount'] = $this->orderInfo->total;
            if(!empty($this->paymentId)) {
                $paymentParams = $this->getPaymentParams($this->paymentId);
                if(!empty($paymentParams)) {
                    $this->orderInfo->payment->params = $this->pgwMerchantParams = $paymentParams;
                }
            }
        }
    }

    /**
     * This method is used to set parameters for pgw
     * @method setPgwMappingPostAdditionalData
     * @access public
     * @params integer $pgwOrderId
     * @return void
     */
    public function setPgwMappingPostAdditionalData($pgwOrderId) {
        if (empty($this->userId)) {
            if (isset($this->orderInfo->user) && isset($this->orderInfo->user['user_id'])) {
                $this->userId = $this->orderInfo->user['user_id'];
            } else {
                $this->userId = 0;
            }
        }
        $this->pgwOrderId = $pgwOrderId;
        $this->orderId = $this->getScOrderIdByPgwOrderId($this->pgwOrderId, $this->paymentId);
        $this->setPgwMappingPreAdditionalData();
    }

    /**
     * This method is used to set parameters for pgw
     * @method getPgwMappingPreAdditionalData
     * @access public
     * @return array
     */
    public function getPgwMappingPreAdditionalData() {
        return $this->pgwMappingDataArray;
    }

    /**
     * This method is used to save post payment response data
     * @method savePostPaymentData
     * @access public
     * @param array $postpaymentDataArray
     * @param array $metaDataArray
     * @return interger prepayment auto increment id
     */
    public function savePostPaymentData(array $postpaymentDataArray, array $metaDataArray = array(), $updateMappingData = true) {
        if (!empty($postpaymentDataArray)) {
            $postpaymentDataArray['status_code'] = (isset($postpaymentDataArray['status_code'])) ? $postpaymentDataArray['status_code'] : null;
            $postpaymentDataArray['other_details'] = isset($postpaymentDataArray['other_details']) && !empty($postpaymentDataArray['other_details']) ? json_encode($postpaymentDataArray['other_details']) : '';
            $this->prepaymentPgwPrimaryId = $this->paymentService->saveAfterPaymentData($postpaymentDataArray);
        }
        if (!empty($this->prepaymentPgwPrimaryId)) {
            $prepaymentIdAdded = CluesRegistry::addObject(PgwModelConstants::REGISTRY_PREPAYMENT_ID, $this->prepaymentPgwPrimaryId);
            if ($prepaymentIdAdded === false) {
                CluesRegistry::updateObject(PgwModelConstants::REGISTRY_PREPAYMENT_ID, $this->prepaymentPgwPrimaryId);
            }
        }
        //handling for multiple redirect response from pg
        //check old prepayment id exist or not
        //if not then simply create new entry in prepayment table and map id mapping table irrespective of status
        //if prepayment id exists then check payment status if old payment status 0 and new payment status 1 then update entry in mapping table with new prepaymnt id
        //for rest cases just map old payment id with new prepayment id
        if ($updateMappingData) {
            $prepaymentData = $this->paymentService->getPrepaymentData($this->pgwMappingDataArray['pgwOrderId'], $this->pgwMappingDataArray['paymentId']);
            $oldPrepaymentId = !empty($prepaymentData['prepayment_details_auto_increment_id']) ? $prepaymentData['prepayment_details_auto_increment_id'] : 0;
            $oldPaymentStatus = empty($oldPrepaymentId) ? 0 : $this->paymentService->getPaymentStatusFromPrepaymentData($oldPrepaymentId);
            $oldPaymentStatus = !empty($oldPaymentStatus) ? $oldPaymentStatus : 0;
            $currentPaymentStatusResponse = ($metaDataArray[PgwModelConstants::PAYMENT_STATUS_KEY] == PgwModelConstants::PAYMENT_SUCCESS_STATUS) ? 1 : 0;
            if (empty($oldPrepaymentId) || (!empty($oldPrepaymentId) && $oldPaymentStatus <= 0 && $currentPaymentStatusResponse == 1)) {
                $this->paymentService->updatePrepaymentIdInMapping(array('pgw_order_id' => $this->pgwMappingDataArray['pgwOrderId'], 'payment_id' => $this->pgwMappingDataArray['paymentId']), $this->prepaymentPgwPrimaryId, $this->userId);
                if (!empty($oldPrepaymentId)) {
                    $metaDataArray[PgwModelConstants::OLD_PREPAYMENT_ID_KEY] = $oldPrepaymentId;
                }
            } else {
                $metaDataArray[PgwModelConstants::OLD_PREPAYMENT_ID_KEY] = $oldPrepaymentId;
            }
        }
        // save ip address
        if (!isset($metaDataArray['response_ip']) || empty($metaDataArray['response_ip'])) {
            $ip = $this->RequestObj->getIp();
            $metaDataArray['response_ip'] = $ip;
            if (CluesRegistry::isKeySet(PgwModelConstants::PGW_RESPONSE_IP_ADDRESS) == false) {
                CluesRegistry::addObject(PgwModelConstants::PGW_RESPONSE_IP_ADDRESS, $ip);
            }
        }
        if (!empty($metaDataArray) && is_array($metaDataArray) && !empty($this->prepaymentPgwPrimaryId)) {
            $tempMetaDataArray = array();
            $tempMetaDataArray['created'] = $tempMetaDataArray['updated'] = time();
            $tempMetaDataArray['created_by'] = $tempMetaDataArray['updated_by'] = $this->userId;
            $tempMetaDataArray['status'] = 1;
            $tempMetaDataArray['prepayment_id'] = $this->prepaymentPgwPrimaryId;
            foreach ($metaDataArray as $key => $value) {
                if (is_string($key)) {
                    $tempMetaDataArray['key'] = $key;
                    $tempMetaDataArray['value'] = $value;
                    $metaInsertId = $this->paymentService->savePostPaymentMetaData($tempMetaDataArray);
                    if ($key == PgwModelConstants::PAYMENT_STATUS_KEY && $value == PgwModelConstants::PAYMENT_SUCCESS_STATUS && !empty($metaInsertId)) {
                        $_tempPayStatusAdded = CluesRegistry::addObject(PgwModelConstants::PAYMENT_MARKED_SUCCESSFUL, PgwModelConstants::PAYMENT_SUCCESS_STATUS);
                        if ($_tempPayStatusAdded === false) {
                            CluesRegistry::updateObject(PgwModelConstants::PAYMENT_MARKED_SUCCESSFUL, PgwModelConstants::PAYMENT_SUCCESS_STATUS);
                        }
                    }
                }
            }
            unset($tempMetaDataArray);
        }
        return $this->prepaymentPgwPrimaryId;
    }

    /**
     * This method is used to save pre payment request data
     * @method savePrePaymentData
     * @access public
     * @param array $prepaymentDataArray
     * @param array $metaDataArray
     * @return interger order pgw primary key
     */
    public function savePrePaymentData(array $prepaymentDataArray, array $metaDataArray = array()) {
        if (!isset($prepaymentDataArray['merchant_params']) || empty($prepaymentDataArray['merchant_params'])) {
            $prepaymentDataArray['merchant_params'] = json_encode($this->pgwMerchantParams);
        }
        if (isset($prepaymentDataArray['order_data']) && !empty($prepaymentDataArray['order_data'])) {
            if(!isset($prepaymentDataArray['order_data']['clientID']) || empty($prepaymentDataArray['order_data']['clientID'])) {
                $prepaymentDataArray['order_data']['clientID'] = $this->clientID;
            }
            $prepaymentDataArray['order_data'] = json_encode($prepaymentDataArray['order_data']);
        }
        $this->orderPgwPrimaryId = $this->paymentService->savePrePaymentData($prepaymentDataArray);
        $this->PaymentPgwMappingObj->updateOrderPgwPrimaryKeyByPgwOrderId($this->orderPgwPrimaryId, $this->pgwOrderId, $this->userId, $this->paymentId);
        // if ip not in $metaDataArray -> check in Registry, if not in registry OR empty in registry -> set manually
        if (!isset($metaDataArray['request_ip']) || empty($metaDataArray['request_ip'])) {
            if (CluesRegistry::isKeySet(PgwModelConstants::CLIENT_IP_FOR_SEND_TO_PGW)) {
                $metaDataArray['request_ip'] = CluesRegistry::getObject(PgwModelConstants::CLIENT_IP_FOR_SEND_TO_PGW);
            }
            if (empty($metaDataArray['request_ip'])) {
                $metaDataArray['request_ip'] = $this->RequestObj->getIp();
            }
        }
        if (isset($this->pgwReqData) && !empty($this->pgwReqData)) {
            foreach (PgwModelConstants::CLIENT_UDF_KEYS as $key) {
                if (isset($this->pgwReqData[$key]) && !is_null($this->pgwReqData[$key])) {
                    $metaDataArray[$key] = $this->pgwReqData[$key];
                }
            }
        }
        if (!empty($metaDataArray) && is_array($metaDataArray) && !empty($this->orderPgwPrimaryId)) {
            $tempMetaDataArray = array();
            $tempMetaDataArray['created'] = $tempMetaDataArray['updated'] = time();
            $tempMetaDataArray['created_by'] = $tempMetaDataArray['updated_by'] = $this->userId;
            $tempMetaDataArray['status'] = 1;
            $tempMetaDataArray['order_pgw_id'] = $this->orderPgwPrimaryId;
            foreach ($metaDataArray as $key => $value) {
                if (is_string($key)) {
                    $tempMetaDataArray['key'] = $key;
                    $tempMetaDataArray['value'] = $value;
                    $this->paymentService->addPrePgwMetaData($tempMetaDataArray);
                }
            }
            unset($tempMetaDataArray);
        }
        return $this->orderPgwPrimaryId;
    }

    /**
     * Save response from pgw after payment
     * @method saveResponse
     * @param object $orderInfo order info object
     * @param mixed $txnResponse pgw transaction response
     * @access public
     * @return array
     */
    abstract public function saveResponse($orderInfo, $txnResponse);

    /**
     * Get Data for redirection
     * @method getDataForRedirection
     * @access public
     * @return array
     */
    abstract public function getDataForRedirection();

    /**
     * This method is used to initialise the payment
     * @method initialize
     * @access public
     * @param array $orderInfo info of the order
     * @param string $redirectUrl Url of redirection
     * @return data function
     */
    public function initialize($orderInfo, $redirectUrl, $codUrl = '') {
        $this->orderInfo = $orderInfo;
        $this->redirecturl = $redirectUrl;
        $this->codUrl = $codUrl;
        $this->setPgwMappingPreAdditionalData();
    }

    /**
     * This method is used to update redirect url
     * @method updateRedirectUrl
     * @access protected
     * @param string $redirectUrl Url of redirection
     * @return void
     */
    protected function updateRedirectUrl($redirectUrl) {
        $this->redirecturl = $this->redirecturl . $redirectUrl;
    }

    /**
     * This method is used for magic method call
     * @method __call
     * @access public
     * @param string $name
     * @param mixed $arguments
     * @return type
     */
    public function __call($name, $arguments) {
        $additionalInfo = $additionalInfo['additional_data'] = array();
        $additionalInfo['field1'] = $name;
        $additionalInfo['field2'] = 'Exception code:' . Status::METHOD_NOT_EXIST;
        $additionalInfo['additional_data']['file'] = __FILE__;
        $additionalInfo['additional_data']['line'] = __LINE__;
        $additionalInfo['additional_data']['domain'] = PgwModelConstants::DOMAIN_NAME;
        $additionalInfo['additional_data']['module'] = PgwModelConstants::MODULE_NAME;
        $additionalInfo['additional_data']['level'] = Logger::CRITICAL;
        $additionalInfo['additional_data']['method'] = __METHOD__;
        $additionalInfo['additional_data']['input_params'] = array_merge($this->pgwReqData, $arguments);
        $additionalInfo['additional_data']['placeholders'] = array($name, get_called_class());
        $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::MODULE_NAME, $additionalInfo, Logger::CRITICAL);
        return null;
    }

    /**
     * This method is used to get sc order id
     * @method getScOrderIdByPgwOrderId
     * @access public 
     * @param integer $pgwOrderId
     * @return integer order id
     */
    public function getScOrderIdByPgwOrderId($pgwOrderId, $paymentId = 0) {
        return $this->PaymentPgwMappingObj->getScOrderIdByPgwOrderId($pgwOrderId, $paymentId);
    }

    /**
     * Get Payment Params
     * @method getPaymentParams
     * @access protected
     * @param integer $paymentId
     * @return array
     */
    protected function getPaymentParams($paymentId) {
        return $this->PaymentObj->getPaymentInfoByPaymentIdAndCID($paymentId, $this->clientID);
    }

    /**
     * this method is used change order status
     * @method changeOrderStatus
     * @access protected
     * @param string $toStatus
     * @return void
     */
    protected function changeOrderStatus($toStatus) {
        try {
            DI::Map('OrderTransistion', 'clues\\model\\order\\OrderTransistion');
            $orderTransitionObj = DI::Singleton('OrderTransistion', array('order_id' => $this->orderId, 'to_status' => $toStatus));
            $paymentResponse = $orderTransitionObj->ChangeOrderStatus();
            $logData = array('field1' => 'orderId:' . $this->orderId, 'field2' => 'paymentOptionId' . $this->orderInfo->payment_option_id, 'field3' => 'toStatus:' . $toStatus, 'error_name' => 'processPaymentOrderStatusChange', 'data' => 'orderResponse' . $paymentResponse['status']);
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::INITIATE_PAYMENT_MODULE_NAME, $logData, Logger::INFO);
        } catch (\Exception $e) {
            $paymentResponse = array(
                'error_code' => $e->getCode(),
                'error_msg' => $e->getMessage()
            );
            $logData = array(
                'field1' => 'orderId:' . $this->orderId,
                'field2' => 'paymentOptionId' . $this->orderInfo->payment_option_id,
                'field3' => 'toStatus:' . $toStatus,
                'field4' => 'error_code:' . $e->getCode(),
                'error_name' => 'processPaymentOrderException',
                'additional_info' => array(
                    'error_msg' => $e->getMessage(),
                    'method' => __METHOD__,
                    'line' => __LINE__
                )
            );
            $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::INITIATE_PAYMENT_MODULE_NAME, $logData, Logger::ERROR);
        }
        return $paymentResponse;
    }

    /**
     * This method is used to get user phone by order info
     * @method getUserPhoneByOrderInfo
     * @access protected
     * @return integer
     */
    protected function getUserPhoneByOrderInfo() {
        $phone = 0;
        if (isset($this->orderInfo->user->s_phone) && !empty($this->orderInfo->user->s_phone)) {
            $phone = $this->orderInfo->user->s_phone;
        } elseif (isset($this->orderInfo->user->b_phone) && !empty($this->orderInfo->user->b_phone)) {
            $phone = $this->orderInfo->user->b_phone;
        }
        return $phone;
    }

    /**
     * Remove special character from email
     * @method removeEmailSpecialChar
     * @access protected
     * @param string $email
     * @return string
     */
    protected function removeEmailSpecialChar($email = null) {
        $updateOrderEmail = false;
        if (empty($email)) {
            $email = $this->orderInfo->user->email;
            $updateOrderEmail = true;
        }
        if (isset($email) && !empty($email)) {
            //only a-zA-Z0-9-_.@ allowed for some pgws
            $email = preg_replace("/[^\w@\-.]/i", '', $email);
        }
        if ($updateOrderEmail && !empty($email)) {
            $this->orderInfo->user->email = $email;
        }
        return $email;
    }

    /**
     * save momoe txn details
     * @method saveMomoeTxnDetails
     * @access protected
     * @param array $paymentDataArray
     * @param array $metaDataArray
     * @return int $momoeOrderId
     */
    protected function saveMomoeTxnDetails(array $paymentDataArray, array $metaDataArray = array()) {
        $momoeOrderId = 0;
        if (!empty($paymentDataArray)) {
            $momoeOrderId = $this->insertMomoeTxnDetails($paymentDataArray);
        }
        if (!isset($metaDataArray[PgwModelConstants::CART_ID]) || empty($metaDataArray[PgwModelConstants::CART_ID]) && $momoeOrderId) {
            $cartDataArray = json_decode($paymentDataArray['cat_data'], true);
            if (isset($cartDataArray['_cart_id']) && !empty($cartDataArray['_cart_id'])) {
                $metaDataArray[PgwModelConstants::CART_ID] = $cartDataArray['_cart_id'];
            }
            unset($cartDataArray);
        }
        if (!empty($metaDataArray) && is_array($metaDataArray) && !empty($momoeOrderId)) {
            $tempMetaDataArray = array();
            $tempMetaDataArray['momoe_order_id'] = $momoeOrderId;
            $tempMetaDataArray['created'] = $tempMetaDataArray['updated'] = time();
            $tempMetaDataArray['created_by'] = $tempMetaDataArray['updated_by'] = $paymentDataArray['user_id'];
            $tempMetaDataArray['status'] = $paymentDataArray['status'];
            foreach ($metaDataArray as $key => $value) {
                if (is_string($key)) {
                    $tempMetaDataArray['object_key'] = $key;
                    $tempMetaDataArray['object_value'] = $value;
                    $this->paymentService->saveMomoeOrderMetaData($tempMetaDataArray);
                }
            }
            unset($tempMetaDataArray);
        }

        return $momoeOrderId;
    }

    /**
     * save momoe txn details in clues_momoe_order_details
     * @method insertMomoeTxnDetails
     * @access protected
     * @param array $paymentDataArray
     * @return int $insertRes momoe order id
     */
    protected function insertMomoeTxnDetails(array $paymentDataArray) {
        $insertRes = 0;
        try {
            $insertRes = $this->paymentService->insertMomoeTxnDetails($paymentDataArray['user_id'], $paymentDataArray['order_id'], $paymentDataArray['momoe_user_id'], $paymentDataArray['ref_id'], $paymentDataArray['status'], $paymentDataArray['cart_order'], $paymentDataArray['cat_data'], $paymentDataArray['request_param'], $paymentDataArray['pgw_request_param'], $paymentDataArray['iframe_url'], $paymentDataArray['total'], $paymentDataArray['re_order'], $paymentDataArray['pay_details_updated'], $paymentDataArray['pgw_response'], $paymentDataArray['type']);
        } catch (\Exception $e) {//first insert data without cart_data/cart_order then update cart_data
            $insertRes = $this->paymentService->insertMomoeTxnDetails($paymentDataArray['user_id'], $paymentDataArray['order_id'], $paymentDataArray['momoe_user_id'], $paymentDataArray['ref_id'], $paymentDataArray['status'], '', '', $paymentDataArray['request_param'], $paymentDataArray['pgw_request_param'], $paymentDataArray['iframe_url'], $paymentDataArray['total'], $paymentDataArray['re_order'], $paymentDataArray['pay_details_updated'], $paymentDataArray['pgw_response'], $paymentDataArray['type']);
            $this->paymentService->updateMomoeOrderDetails($insertRes, 'cart_order', $paymentDataArray['cart_order']);
            $this->paymentService->updateMomoeOrderDetails($insertRes, 'cart_data', $paymentDataArray['cat_data']);
        }
        return $insertRes;
    }

    /**
     * This method is used to get pgw data of all payment options
     * @method _getPgwData
     * @access private
     * @return array
     */
    private function _getPgwData() {
        $returnData = array();
        if (\Registry::get('config.memcache') && $GLOBALS['memcache_status']) {
            $memcache = $GLOBALS['memcache'];
        }
        if ($memcache) {
            $key = md5(__METHOD__);
            $returnData = $memcache->get($key);
            if (!empty($returnData) && is_array($returnData)) {
                return $returnData;
            }
        }
        $returnData['payment_pgw_names'] = $this->paymentService->getPgwName();
        $returnData['payment_option_pgw_names'] = $this->paymentService->getAllPaymentOptionsName();
        if ($memcache) {
            $memcache->set($key, $returnData, MEMCACHE_COMPRESSED, 86400);
        }
        return $returnData;
    }

    /**
     * This method is used to get pgwName data
     * @method getPgwName
     * @param array $pgwIds 
     * @access protected
     * @return string
     */
    protected function getPgwName(array $pgwIds = array()) {
        $pgwName = '';
        if (isset($pgwIds['payment_id']) && isset($pgwIds['payment_option_id'])) {
            if ($pgwIds['payment_id'] == 0 && $pgwIds['payment_option_id'] == 0) {
                $pgwName = PgwModelConstants::COLLECT_ZERO_PAYMENT_PGW;
            } elseif (!empty($pgwIds['payment_id']) && !empty($pgwIds['payment_option_id'])) {
                $nameKey = $pgwIds['payment_id'] . '_' . $pgwIds['payment_option_id'];
                if (isset($this->_paymentIdPaymentOptionIdPgwName[$nameKey]) && !empty($this->_paymentIdPaymentOptionIdPgwName[$nameKey])) {
                    $pgwName = $this->_paymentIdPaymentOptionIdPgwName[$nameKey];
                } else {
                    $paymentData = $this->_getPgwData();
                    if (!empty($paymentData['payment_pgw_names'][$pgwIds['payment_id']]['name']) && !empty($paymentData['payment_option_pgw_names'][$pgwIds['payment_option_id']]['name'])) {
                        $pgwName = (strtolower($paymentData['payment_pgw_names'][$pgwIds['payment_id']]['name']) == strtolower($paymentData['payment_option_pgw_names'][$pgwIds['payment_option_id']]['name'])) ? $paymentData['payment_pgw_names'][$pgwIds['payment_id']]['name'] : $paymentData['payment_pgw_names'][$pgwIds['payment_id']]['name'] . '_' . $paymentData['payment_option_pgw_names'][$pgwIds['payment_option_id']]['name'];
                    }
                }
            }
        }
        return $pgwName;
    }

    /**
     * create a unique refid with appending given salt
     * @method createRefId
     * @access public
     * @param string $salt salt to be append in the end of ref_id
     * @return string unique random refference id used by momoe for transaction id
     */
    public function createRefId($salt = '') {
        //check momoe ref id 10 times in table: clues_momoe_order_details
        for ($retryCount = 1; $retryCount <= 10; $retryCount++) {
            $tempRefId = 'A' . $retryCount . StringOperations::getMd5UniqueId($salt);
            $existCount = $this->paymentService->checkMomoeRefId($tempRefId);
            if (empty($existCount)) {
                return $tempRefId;
            }
        }
        return 0;
    }

    /**
     * This method is used to delink the wallet
     * @method delinkWallet
     * @param integer $userId
     * @param integer $paymentOptionId
     * @param string $userType
     * @param  integer $status
     */
    public function delinkWallet($userId, $paymentOptionId, $userType, $status = PgwModelConstants::TOKEN_EXCEPTION_STATUS) {
        $this->paymentService->delinkWallet($userId, $paymentOptionId, $userType, $status);
    }

    /**
     * This method is used to validate pgw ordrr id with sc order id
     * @method isPgwOrderIdAndScOrderIdMapped
     * @access protected
     * @param string $pgwOrderId
     * @param boolean $errorResponse
     * @param integer $scOrderId
     * @return mixed
     */
    protected function isPgwOrderIdAndScOrderIdMapped($pgwOrderId, $errorResponse = false, $scOrderId = 0) {
        $valid = false;
        if (empty($scOrderId) || !is_numeric($scOrderId)) {
            $scOrderId = $this->orderId;
        }
        if (!empty($scOrderId) && !empty($pgwOrderId)) {
            $_tempOrderIds = $this->paymentService->getScOrderIdsByPgwOrderId($pgwOrderId);
            if (in_array($scOrderId, $_tempOrderIds)) {
                $valid = true;
            }
        }
        if ($errorResponse && $valid === false) {
            $valid = array(
                'success' => 0,
                'order_id' => $scOrderId, //plz don't change key name
                'status' => 'F',
                'details' => 'pgw order id and sc order id mismatch',
                'pgw_order_id' => $pgwOrderId,
                'client_order_ids' => $_tempOrderIds,
                'direcpayreferenceid' => 'NA',
                'flag' => 'diff:pgoid-oid',
                'amount' => 0,
                'status_code' => 'ErMismatch'
            );
        }
        return $valid;
    }

    /**
     * This method is used to override fields in ppd data array
     * @method overridePPDFields
     * @access protected
     * @param array $input
     * @param array $callStatusInfo
     * @return array
     */
    protected function overridePPDFields(array &$input, array $callStatusInfo = array()) {
        if (!empty($callStatusInfo)) {
            //late static binding for picking up child class value if overrided by last child class
            foreach (static::$overridePPDFields as $value) {
                if (isset($callStatusInfo[$value])) {
                    $input[$value] = $callStatusInfo[$value];
                }
            }
        }
        return $input;
    }

    /**
     * This method is used to get hash string of request data
     * @method getRequestHash
     * @access protected
     * @param array $inputArray
     * @return string Hash String
     */
    protected function getRequestHash(array $inputArray) {
        return $this->PGWHashManager->getRequestHash($inputArray, static::$PGWClassName, $this->orderInfo);
    }

    /**
     * This method is used to get hash string of response data
     * @method getResponseHash
     * @access protected
     * @param array $inputArray
     * @return string Hash String
     */
    protected function getResponseHash(array $inputArray) {
        return $this->PGWHashManager->getResponseHash($inputArray, static::$PGWClassName, $this->orderInfo);
    }

    /**
     * This method is used to get hash string of notify response data
     * @method getNotifyResponseHash
     * @access protected
     * @param array $inputArray
     * @return string Hash String
     */
    protected function getNotifyResponseHash(array $inputArray) {
        return $this->PGWHashManager->getNotifyResponseHash($inputArray, static::$PGWClassName, $this->orderInfo);
    }

    /**
     * This method is used to match hash string of response data
     * @method matchResponseHash
     * @access protected
     * @param array $inputArray
     * @return string Hash String
     */
    protected function matchResponseHash(array $inputArray) {
        return $this->PGWHashManager->checkResponseHash($inputArray, static::$PGWClassName, $this->orderInfo);
    }

    /**
     * This method tells whether pgw mapping table requires single update or not
     * @method cppmNotifySingleUpdateRequired
     * @access public
     * @return boolean
     */
    public function cppmNotifySingleUpdateRequired() {
        //late static binding for picking up child class value
        return static::$cppmNotifySingleUpdateRequired;
    }
}