<?php

namespace clues\model\pgw;

use clues\model\pgw\PGWDriver;
use clues\system\Logger;
use clues\system\DI;
use clues\model\pgw\PgwModelConstants;
use clues\system\CluesRegistry;

/**
 * This class is responsible for payment notify operations
 * @class PgwNotify
 * @access public
 * @category Model class
 * @package model/pgw
 */
class PgwNotify extends PGWDriver {

    /**
     * Used to hold pgw file name for notify api
     * @var array
     */
    private static $_notifyPgwConfigArray;

    /**
     * sc order id
     * @access private 
     * @var string
     */
    private $_orderId;

    /**
     * request data
     * @access private
     * @var array
     */
    private $_requestData;

    /**
     * pgw object
     * @access private
     * @var object
     */
    private $_pgwObj;

    /**
     * logger field array
     * @access private
     * @var array
     */
    private $_loggerFieldArray = array();

    /**
     * notify response array
     * @access private
     * @var array 
     */
    private $_formattedNotifyResponse = array();

    /**
     * prepayment id
     * @access private
     */
    private $_prepaymentId;

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @return void
     */
    public function __construct(array $data = array()) {
        parent::__construct();
        $this->_requestData = $data;
        $this->_addLog(__LINE__, 'pgw_response', Logger::INFO);
        $pgwNotifyJson = $this->CluesConfigObj->getKeyValue('pgw_notify_api_config', 'payment', 'api');
        self::$_notifyPgwConfigArray = json_decode($pgwNotifyJson, true);
        if (!isset($this->_requestData['pgw']) || empty($this->_requestData['pgw']) || !array_key_exists($this->_requestData['pgw'], self::$_notifyPgwConfigArray['pgw'])) {
            throw new \Exception('pgw not found in notify');
        }
        DI::Map($this->_requestData['pgw'], 'clues\\model\\pgw\\' . $this->_requestData['pgw']);
        $this->_pgwObj = DI::Singleton($this->_requestData['pgw']);
        $this->_formattedNotifyResponse = $this->_pgwObj->getFormattedResponse($this->_requestData);
        $this->_formattedNotifyResponse['prepayment_data']['other_details_meta_data'] = $this->_formattedNotifyResponse['prepayment_data']['other_details'];
        $this->_formattedNotifyResponse['prepayment_data']['other_details'] = isset($this->_formattedNotifyResponse['prepayment_data']['other_details']) && !empty($this->_formattedNotifyResponse['prepayment_data']['other_details']) ? json_encode($this->_formattedNotifyResponse['prepayment_data']['other_details']) : '';
        if (!isset($this->_formattedNotifyResponse['pgw_order_id']) || empty($this->_formattedNotifyResponse['pgw_order_id'])) {
            throw new \Exception('blank pgw_order_id');
        }
        $this->_loggerFieldArray['field1'] = 'pgw-' . $this->_requestData['pgw'];
        $this->_loggerFieldArray['field2'] = 'pgw_order_id-' . $this->_formattedNotifyResponse['pgw_order_id'];
        $this->_loggerFieldArray['field3'] = 'payment_id-' . self::$_notifyPgwConfigArray['pgw'][$this->_requestData['pgw']];
    }

    /**
     * this method is used to verify and save notify response
     * @method verifyAndSaveNotifyResponse
     * @access public
     * @return void
     */
    public function verifyAndSaveNotifyResponse() {
        $prepaymentData = $this->paymentService->getPrepaymentData($this->_formattedNotifyResponse['pgw_order_id'], self::$_notifyPgwConfigArray['pgw'][$this->_requestData['pgw']]);
        if (!empty($prepaymentData) && isset($prepaymentData['client_order_id']) && !empty($prepaymentData['client_order_id'])) {
            $this->paymentService->updateTransactionStatus($prepaymentData['id'], PgwModelConstants::NOTIFY_PICKED_STATUS);
            $this->_orderId = $prepaymentData['client_order_id'];
            $this->_formattedNotifyResponse['prepayment_data']['order_id'] = $this->_orderId;
            $currentPaymentStatusData = $this->_getCurrentPaymentStatusData();
            $notifyPaymentStatus = $this->_pgwObj->getNotifyResponseStatus($currentPaymentStatusData['total']);
            $this->_saveNotifyData($prepaymentData['prepayment_details_auto_increment_id'], $currentPaymentStatusData['status'], $notifyPaymentStatus, $prepaymentData['order_pgw_auto_increment_id']);
            $isPaymentMarkSuccess = CluesRegistry::getObject(PgwModelConstants::PAYMENT_MARKED_SUCCESSFUL);
            if ($notifyPaymentStatus == 1 && empty($isPaymentMarkSuccess)) {
                $notifyPaymentStatus = 0;
            }
            $this->paymentService->updateTransactionStatus($prepaymentData['id'], PgwModelConstants::NOTIFY_UPDATED_STATUS);
            // BEGIN: PAYMENT CLIENT REDIRECTION
            $prepaymentData['toStatus'] = !empty($notifyPaymentStatus) ? 'P' : 'F';
            $responseNotifyArray = $this->prepareAndInsertNotifyData($prepaymentData);
        } else {
            $this->_addLog(__LINE__, 'no_data_in_pgw_mapping', Logger::ERROR);
            throw new \Exception('order Id not found in notify');
        }
    }

    /**
     * this method used to _save notify response and status
     * @method _saveNotifyResponseAndStatus
     * @access private
     * @param integer $prepaymentId prepayment id
     * @param integer $notifyPaymentStatus notify response status
     * @return void
     */
    private function _saveNotifyResponseAndStatus($prepaymentId, $notifyPaymentStatus) {
        $notifyMetaData = $this->_getPostPaymentMetaData($prepaymentId, PgwModelConstants::PGW_NOTIFY_RESPONSE_KEY, $this->_formattedNotifyResponse['prepayment_data']['other_details']);
        $this->paymentService->savePostPaymentMetaData($notifyMetaData);
        $notifyMetaData['key'] = PgwModelConstants::PGW_NOTIFY_STATUS_KEY;
        $notifyMetaData['value'] = $notifyPaymentStatus;
        $this->paymentService->savePostPaymentMetaData($notifyMetaData);
        $notifyMetaData['key'] = 'response_ip';
        $notifyMetaData['value'] = $this->RequestObj->getIp();
        $this->paymentService->savePostPaymentMetaData($notifyMetaData);
    }

    /**
     * Save post payment meta data
     * @method _getPostPaymentMetaData
     * @param integer $cluesPrepaymentId
     * @param string $key key 
     * @param integer $value
     * @return array
     */
    private function _getPostPaymentMetaData($cluesPrepaymentId, $key, $value) {
        $response = array();
        if (!empty($cluesPrepaymentId) && isset($key) && isset($value)) {
            $response['prepayment_id'] = $cluesPrepaymentId;
            $response['key'] = $key;
            $response['value'] = $value;
            $response['status'] = 1;
            $response['created'] = $response['updated'] = time();
            $response['created_by'] = $response['updated_by'] = PgwModelConstants::PAYMENT_USER_ID;
        }
        return $response;
    }

    /**
     * this method used to check current order status
     * @method _getCurrentPaymentStatus
     * @access private
     * @return array
     */
    private function _getCurrentPaymentStatusData() {
        $orderPaymentStatus = array();
        $orderInfo = $this->getPgwOrderInfo($this->_formattedNotifyResponse['pgw_order_id']);
        $orderPaymentStatus = array('status' => -1, 'total' => $orderInfo->total);
        return $orderPaymentStatus;
    }

    /**
     * this method used to save notify response 
     * @method _saveNotifyData
     * @access private
     * @param integer $oldPrepaymentId old prepayment id
     * @param integer $currentPaymentStatus payment response status
     * @param integer $notifyPaymentStatus notify response status
     * @param integer $orderPgwId order_pgw_auto_increment_id
     * @return void
     */
    private function _saveNotifyData($oldPrepaymentId, $currentPaymentStatus, $notifyPaymentStatus, $orderPgwId = 0) {
        if (empty($oldPrepaymentId) || (!empty($oldPrepaymentId) && $currentPaymentStatus <= 0 && $notifyPaymentStatus == 1)) {
            $prepaymentId = $this->paymentService->saveAfterPaymentData($this->_formattedNotifyResponse['prepayment_data']);
            if (!empty($this->_requestData['pgw']) &&
                    in_array(
                            strtoupper($this->_requestData['pgw']), array_map(
                                    'strtoupper', array(
                        PgwModelConstants::NOTIFY_NAME_2C2P
                                    )
                            )
                    )) {
                $this->_savePgwMetaData($prepaymentId);
            }
            $notifyMetaData = $this->_getPostPaymentMetaData($prepaymentId, PgwModelConstants::PAYMENT_STATUS_KEY, $notifyPaymentStatus);
            $isPaymentMarkSuccess = $this->paymentService->savePostPaymentMetaData($notifyMetaData);
            if ($notifyPaymentStatus == 1 && !empty($isPaymentMarkSuccess)) {
                CluesRegistry::addObject(PgwModelConstants::PAYMENT_MARKED_SUCCESSFUL, PgwModelConstants::PAYMENT_SUCCESS_STATUS);
            }
            $singleUpdateRequired = $this->_pgwObj->cppmNotifySingleUpdateRequired();
            $this->paymentService->updatePrepaymentIdInMapping(array('pgw_order_id' => $this->_formattedNotifyResponse['pgw_order_id'], 'payment_id' => self::$_notifyPgwConfigArray['pgw'][$this->_requestData['pgw']]), $prepaymentId, PgwModelConstants::PAYMENT_USER_ID, $singleUpdateRequired);
            if (!empty($oldPrepaymentId)) {
                $notifyMetaData['key'] = PgwModelConstants::OLD_PREPAYMENT_ID_KEY;
                $notifyMetaData['value'] = $oldPrepaymentId;
                $this->paymentService->savePostPaymentMetaData($notifyMetaData);
            }
            $this->_prepaymentId = $prepaymentId;
            $ip = $this->RequestObj->getIp();
            $notifyMetaData = $this->_getPostPaymentMetaData($prepaymentId, 'response_ip', $ip);
            $this->paymentService->savePostPaymentMetaData($notifyMetaData);
        } else {
            $this->_saveNotifyResponseAndStatus($oldPrepaymentId, $notifyPaymentStatus);
            $this->_prepaymentId = $oldPrepaymentId;
            $notifyMetaData = $this->_getPostPaymentMetaData($oldPrepaymentId, PgwModelConstants::CARD_NUMBER, $this->_formattedNotifyResponse['prepayment_data']['cardObf']);
            $this->paymentService->savePostPaymentMetaData($notifyMetaData);
        }
    }

    /**
     * this method is used to add logs
     * @method _addLog
     * @access private
     * @param integer $line
     * @param string $errorname
     * @param integer $logLevel
     * @return void 
     */
    private function _addLog($line, $errorname, $logLevel) {
        $this->_loggerFieldArray['error_name'] = $errorname;
        $this->_loggerFieldArray['other_data'] = json_encode($this->_requestData);
        $this->_loggerFieldArray['file'] = __FILE__;
        $this->_loggerFieldArray['line'] = $line;
        $this->LoggerObj->log(PgwModelConstants::DOMAIN_NAME, PgwModelConstants::NOTIFY_MODULE_NAME, $this->_loggerFieldArray, $logLevel);
    }

    /**
     * this method is used to save the pgw meta data
     * @method _savePgwMetaData
     * @access private
     * @param integer $prepaymentId
     * @return void 
     */
    private function _savePgwMetaData($prepaymentId) {
        $metaDataArray = array();
        $this->_pgwObj->setPgwTxnData($this->_formattedNotifyResponse['prepayment_data']['other_details_meta_data'], $metaDataArray);
        if (!empty($metaDataArray)) {
            foreach ($metaDataArray as $metaKey => $metaValue) {
                $notifyMomoeMetaData = $this->_getPostPaymentMetaData($prepaymentId, $metaKey, $metaValue);
                $this->paymentService->savePostPaymentMetaData($notifyMomoeMetaData);
            }
        }
        unset($metaDataArray, $notifyMomoeMetaData);
    }
    
    /**
     * get ip details for order id
     * @method _getIpDetailsForOrderId
     * @access private
     * @param integer $orderId
     * @param integer $paymentOptionId
     * @return array $ip ['request_ip' => <ip_addr>, 'response_ip' => <ip_addr>]
     */
    private function _getIpDetailsForOrderId($orderId, $paymentOptionId) {
        $ip = ['request_ip' => '', 'response_ip' => ''];
        $prepaymentId = isset($this->_prepaymentId) && !empty($this->_prepaymentId) ? $this->_prepaymentId : 0;
        $orderPgwId = $this->paymentService->getMappingData($orderId, $paymentOptionId, $prepaymentId);
        if (!empty($orderPgwId)) {
            $ip['request_ip'] = $this->paymentService->getPrePgwMetaData($orderPgwId, 'request_ip');
        }
        if (!empty($prepaymentId)) {
            $ip['response_ip'] = $this->paymentService->getPostPgwMetaData($prepaymentId, 'response_ip');
        } else {
            $ip['response_ip'] = $this->RequestObj->getIp();
        }
        return $ip;
    }

    /**
     * this method is used to prepare and insert notify data
     * @method prepareAndInsertNotifyData
     * @access protected
     * @param array $orderDetails ['orderId' => <oid>, 'toStatus' => <status>, 'paymentOptionId' => <opt_id>, 'paymentId' => <payment_id>]
     * @return string
     */
    protected function prepareAndInsertNotifyData(array $orderDetails) {
        $formDataJson = '';
        $notifyId = 0;
        DI::Map('Payment', 'clues\\model\\payment\\Payment');
        $PaymentObj = DI::Singleton('Payment');
        $paymentDetails = $PaymentObj->getPaymentDetailsByOrderId($orderDetails['client_order_id']);
        if (!empty($paymentDetails) && is_array($paymentDetails)) {
            $keys = PgwModelConstants::NOTIFY_DATA_KEYS;
            $paymentDetails = array_pop($paymentDetails);
            $tempDetails = [];
            foreach ($keys as $key) {
                if (!isset($paymentDetails[$key])) {
                    $tempDetails[$key] = '';
                } else {
                    $tempDetails[$key] = $paymentDetails[$key];
                    unset($paymentDetails[$key]);
                }
            }
            if (strtoupper($orderDetails['toStatus']) == 'F') {
                $tempDetails['payment_status'] = 0;
                $tempDetails['flag'] = 'failure';
                $tempDetails['udf_payment_type'] = $tempDetails['pgw_txn_id'] = $tempDetails['card_number'] = $tempDetails['pgw_order_id'] = $tempDetails['prepayment_details_auto_increment_id'] = $tempDetails['direcpayreferenceid'] = '';
            }
            $paymentDetails = $tempDetails;
        } else {
            $paymentTypeDetails = $this->paymentService->getPaymentTypeDetails($orderDetails['payment_option_id']);
            $pgwName = $this->paymentService->getPgwNameFromPaymentId($orderDetails['payment_id']);
            $paymentDetails = array(
                'order_id' => $orderDetails['client_order_id'],
                'client_order_id' => $orderDetails['client_order_id'],
                'payment_id' => $orderDetails['payment_id'],
                'payment_option_id' => $orderDetails['payment_option_id'],
                'payment_status' => 0,
                'payment_status_info' => 'Payment Not Received',
                'flag' => 'failure',
                'emi_id' => 0,
                'emi_name' => NULL,
                'status' => 1,
                'pgw_type' => $paymentTypeDetails['pgw_type'],
                'pgw_option_name' => $paymentTypeDetails['pgw_option_name'],
                'payment_type_id' => $paymentTypeDetails['payment_type_id'],
                'pgw' => $pgwName['pgw'],
            );
            $paymentDetails['created_by'] = $paymentDetails['updated_by'] = PgwModelConstants::ADMIN_USER_ID;
            $paymentDetails['created_timestamp'] = $paymentDetails['updated'] = time();
            $paymentDetails['created'] = date('d M Y H:i:s', time());
            foreach ($keys as $key) {
                if (!isset($paymentDetails[$key])) {
                    $paymentDetails[$key] = '';
                }
            }
        }
        $ipDetails = $this->_getIpDetailsForOrderId($paymentDetails['order_id'], $paymentDetails['payment_option_id']);
        $paymentDetails['request_ip'] = $ipDetails['request_ip'];
        $paymentDetails['response_ip'] = $ipDetails['response_ip'];
        $formDataJson = json_encode($paymentDetails);
        $notifyId = $this->paymentService->insertPupNotifyMapping($orderDetails, $formDataJson, PgwModelConstants::ADMIN_USER_ID, PgwModelConstants::ATOM_NOTIFY_SOURCE);
        return ['notifyData' => $formDataJson, 'notifyId' => $notifyId];
    }
}