<?php

namespace clues\model\pgwVerify;

use clues\system\BasePlatform;
use clues\system\Logger;
use clues\system\DI;
use clues\model\pgwVerify\PgwNotifyConstants;

/**
 * This class is responsible for notify payment info to end client
 * @class ClientNotify
 * @access public
 * @category Model class
 * @package model/pgw
 */
class ClientNotify extends BasePlatform implements PgwNotifyConstants {

    /**
     * notify db object
     * @access protected
     * @var mixed 
     */
    protected $PgwNotifyDBO;

    /**
     * Enc dnc object
     * @access public
     * @var mixed 
     */
    public $EncryptDecrypt;

    /**
     * client id details
     * @var string
     */
    protected $clientDetails;

    /**
     * Class constructor
     * @method __construct
     * @access public
     * @return void
     */
    public function __construct(array $data = array()) {
        parent::__construct();
        DI::Map('PgwNotifyDBOperations', 'clues\\model\\pgwVerify\\PgwNotifyDBOperations');
        $this->PgwNotifyDBO = DI::Singleton('PgwNotifyDBOperations');
        DI::Map('EncryptDecrypt', 'clues\\model\\pgwVerify\\util\\EncryptDecrypt');
        $this->EncryptDecrypt = DI::Singleton('EncryptDecrypt');
    }

    /**
     * notify pending transactions
     * @method notifyPendingTransactions
     * @access public
     * @param integer $orderId
     * @param integer $debug
     * @return array
     */
    public function notifyPendingTransactions($orderId, $debug) {
        $returnResponse = array();
        $daysBefore = $this->CluesConfigObj->getKeyValue('days_before_for_notify_client', 'payment', 'api');
        $daysBefore = !empty($daysBefore) ? $daysBefore : PgwNotifyConstants::MAX_NOTIFY_DATA_PICK_DAYS;
        $maxReteycount = $this->CluesConfigObj->getKeyValue('max_retry_count_for_notify_client', 'payment', 'api');
        $maxReteycount = !empty($maxReteycount) && $maxReteycount > 0 ? $maxReteycount : PgwNotifyConstants::MAX_NOTIFY_RETRY_COUNT;
        $limit = $this->CluesConfigObj->getKeyValue('max_limit_for_notify_client', 'payment', 'api');
        $limit = !empty($limit) && $limit > 0 ? $limit : PgwNotifyConstants::MAX_NOTIFY_DATA_LIMIT;
        $orderList = $this->PgwNotifyDBO->getClientNotifyData($orderId, $daysBefore, $limit, $maxReteycount);
        if ($debug) {
            echo PHP_EOL . "<br>" . 'notify_data_order_list';
            print_r($orderList);
        }
        if (!empty($orderList)) {
            foreach ($orderList as $order) {
                $this->PgwNotifyDBO->updateClientNotifyStatus($order['id'], PgwNotifyConstants::NOTIFY_STATUSES['processing'], true);
                $status = PgwNotifyConstants::NOTIFY_STATUSES['no_processing_required'];
                $reqRes = array();
                $reqRes['notify_id'] = $order['id'];
                $reqRes['client_order_id'] = $order['client_order_id'];
                $reqRes['retry_count'] = $order['retry_count'];
                $reqRes['created'] = date('Y-m-d H:i:s');
                $reqRes['status'] = 1;
                $reqRes['created_by'] = self::ADMIN_USER_ID;
                $reqRes['log']['error'] = false;
                $clientId = !empty($order['client_id']) ? $order['client_id'] : PgwNotifyConstants::DEFAULT_CLIENT_ID;
                $this->setAppClientIdDetails($clientId);
                if ($debug) {
                    echo PHP_EOL . "<br>" . 'client_details_for_id' . $order['id'];
                    print_r($this->clientDetails);
                }

                //basic client data validation
                if (empty($this->clientDetails) || $this->clientDetails['client_id'] != $clientId ||
                        empty($this->clientDetails['client_configurations']['notify_url'])) {
                    $reqRes['log']['error_msg'] = 'end_point_or_client_configuration_missing';
                    $reqRes['log']['error'] = true;
                }

                //encryption and paylaod preparation
                if (!$reqRes['log']['error'] && !empty($this->clientDetails['client_configurations']['notify_url']) &&
                        !empty($this->clientDetails['client_configurations']['notify_key']) &&
                        !empty($this->clientDetails['client_configurations']['method'])) {
                    $reqRes['log']['request']['method'] = $this->clientDetails['client_configurations']['method'];
                    try {
                        $reqRes['log']['request']['encytption_required'] = true;
                        $this->EncryptDecrypt->setKey($this->clientDetails['client_configurations']['notify_key']);
                        $encString = $this->EncryptDecrypt->encrypt($order['notify_data'], $this->clientDetails['client_configurations']['method']);
                        // $tempString = $this->EncryptDecrypt->decrypt($encString, $this->clientDetails['client_configurations']['method']);
                        if ($encString) {
                            $reqRes['log']['request']['final_payload'] = json_encode([PgwNotifyConstants::NOTIFY_CLIENT_PAYLOAD_KEY => $encString]);
                        } else {
                            $reqRes['log']['request']['final_payload'] = $order['notify_data'];
                            $reqRes['log']['request']['encryption_error'] = 'empty_encrypted_data';
                        }
                    } catch (\Exception $ex) {
                        $reqRes['log']['request']['final_payload'] = $order['notify_data'];
                        $reqRes['log']['request']['encryption_error'] = $ex->getMessage();
                        if ($debug) {
                            echo PHP_EOL . "<br>" . 'exception_in_encryption_in_id' . $order['id'];
                            print_r($ex);
                        }
                    }
                } elseif (!$reqRes['log']['error'] && !empty($this->clientDetails['client_configurations']['notify_url'])) {
                    $reqRes['log']['request']['final_payload'] = $order['notify_data'];
                    $reqRes['log']['request']['encytption_required'] = false;
                } else {
                    $reqRes['log']['error'] = true;
                }
                $reqRes['log']['request']['notify_url'] = isset($this->clientDetails['client_configurations']['notify_url']) ? $this->clientDetails['client_configurations']['notify_url'] : null;

                //sending data to client
                if (!$reqRes['log']['error']) {
                    $reqRes['log']['response'] = $this->_curlRequest($order['client_order_id'], $this->clientDetails['client_configurations']['notify_url'], $reqRes['log']['request']['final_payload']);
                    if ($reqRes['log']['response']['http_status_code'] === 200 || $order['retry_count'] >= $maxReteycount) {
                        $status = PgwNotifyConstants::NOTIFY_STATUSES['processed'];
                    } else {
                        $status = PgwNotifyConstants::NOTIFY_STATUSES['partially_processed'];
                    }
                }

                //changing notify map status
                $returnResponse[$order['id']]['notify_log'] = $reqRes;
                $this->PgwNotifyDBO->updateClientNotifyStatus($order['id'], $status);
                $reqRes['log'] = json_encode($reqRes['log']);
                $returnResponse[$order['id']]['ack_id'] = $this->PgwNotifyDBO->insertNotifyReqRes($reqRes);
                unset($reqRes);
                $this->clientDetails = null;
            }
        }
        if ($debug) {
            echo PHP_EOL . "<br>";
            print_r($returnResponse);
        }
        return $returnResponse;
    }

    /**
     * This method is used to set app_client_id details
     * @method setAppClientIdDetails
     * @param string $clientId client id
     * @access protected
     * @return array
     */
    protected function setAppClientIdDetails($clientId = PgwNotifyConstants::DEFAULT_CLIENT_ID) {
        if (empty($clientId)) {
            $clientId = PgwNotifyConstants::DEFAULT_CLIENT_ID;
        }
        if (!empty($clientId)) {
            $details = $this->PgwNotifyDBO->getAppClientIdDetails($clientId);
            if (!empty($details)) {
                $details['client_configurations'] = json_decode($details['client_configurations'], true);
            }
            $this->clientDetails = $details;
        }
    }

    /**
     * This method is used to hit curl
     * @method _curlRequest
     * @access private
     * @param integer $orderId
     * @param string $url url
     * @param string $postData data
     * @param string $errorPrefix
     * @return array response
     */
    private function _curlRequest($orderId, $url, $postData = '', $errorPrefix = 'NOTIFY_CLIENT_') {
        $timeoutTimeSeconds = $this->CluesConfigObj->getKeyValue('notify_client_curl_timeout', 'payment', 'api');
        if (empty($timeoutTimeSeconds)) {
            $timeoutTimeSeconds = 30;
        }
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_HEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 30);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, FALSE);
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeoutTimeSeconds);
        if (!empty($postData)) {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        }
        $response = curl_exec($ch);
        $curlError = curl_error($ch);
        $res = json_decode($response, true);
        $res['http_status_code'] = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        if (!empty($curlError)) {
            $exception_data = array();
            $exception_data['field1'] = 'order_id:' . $orderId;
            $exception_data['field2'] = 'curl_response:' . $response;
            $exception_data['error_name'] = $errorPrefix . 'CURL_ERROR';
            $additionalInfo = array();
            $additionalInfo['input_params'] = $postData;
            $res['curl_error'] = $additionalInfo['curl_error'] = $curlError;
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['domain'] = PgwNotifyConstants::DOMAIN_NAME;
            $additionalInfo['module'] = 'notify_client_cron';
            $additionalInfo['level'] = Logger::ERROR;
            $exception_data['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwNotifyConstants::DOMAIN_NAME, 'notify_client_cron_curl_error', $exception_data, Logger::ERROR);
        }
        return $res;
    }

}
