<?php

namespace clues\model\pgwVerify;

use clues\model\pgwVerify\PgwVerifyBase;
use clues\model\pgwVerify\PgwVerifyConstants;
use clues\model\pgw\exception\PgwException;
use clues\model\pgw\exception\PgwStatus;
use clues\system\Logger;
use clues\system\util\JWT;
use clues\system\DI;

/**
 * This class is responsible for Pgw2C2P payment verify operations
 * @class Pgw2C2P
 * @access public
 * @category Model class
 * @package model/pgwVerify
 */
class Pgw2C2P extends PgwVerifyBase {

    /**
     * Used for prepayment details data
     * @var array 
     */
    protected static $_fieldsForPrepaymentDetails = [
        'referenceNo' => PgwVerifyConstants::PGW_REF_NO,
        'approvalCode' => PgwVerifyConstants::PGW_APPROVAL_CODE,
        'cardNo' => PgwVerifyConstants::CARD_NUMBER,
        'cardType' => PgwVerifyConstants::PG_CARD_TYPE,
        'cardToken' => PgwVerifyConstants::PG_CARD_TOKEN
    ];

    /**
     * Used to store gokwik order failed reason
     * @var string
     */
    private $_failedReason = null;

    /**
     * method used to verify transaction 
     * @method getPgwTransaction
     * @access public
     * @param array $inputParams array of order info and payment
     * @return array verify Response
     */
    public function getPgwTransaction(array $inputParams) {
        if (isset($this->verifyOrderCache['order_id']) && $this->verifyOrderCache['order_id'] == $inputParams['client_order_id']) {
            return $this->verifyOrderCache['response'];
        }
        $response = $this->_verifyResponse($inputParams);
        if (in_array($response['respCode'], array('0000')) && $response['amount'] >= $inputParams['amount'] && $response['invoiceNo'] == $inputParams['pgw_order_id']) {
                $this->_failedReason = '';
                $metaDataArray['payment_status'] = 1;
            } else if (!in_array($response['respCode'], array('0000'))) {
                $this->_failedReason = 'Wrong status';
            } else if ($response['amount'] < $inputParams['amount']) {
                $this->_failedReason = 'Less amt received';
            } else if ($response['invoiceNo'] != $inputParams['pgw_order_id']) {
                $this->_failedReason = 'Pgw order id mismatch';
            }
        $this->verifyOrderCache['order_id'] = $inputParams['client_order_id'];
        $this->verifyOrderCache['response'] = $response;
        return $response;
    }

    /**
     * method used to verify order status
     * @method _verifyResponse
     * @access private
     * @param array $inputParams
     * @return array
     * @throws exception
     */
    private function _verifyResponse($inputParams) {
        $responsePayloadData = array();
        try {
            $paymentInquiryDataArray = array (
                'merchantID' => $inputParams['payment_params']['merchantID'], //Unique merchant ID that is registered with 2C2P, Type-Alphanumeric
                'invoiceNo' => $inputParams['pgw_order_id'], //Unique merchant order number, Type-Alphanumeric
                'locale' => 'en'
            );
            $paymentInquiryUrl = $inputParams['payment_params']['baseUrl'] . $inputParams['payment_params']['paymentInquiryApiUrl'];
            $paymentInquiryPayload = JWT::createJWTPayload($paymentInquiryDataArray, $inputParams['payment_params']['secretKey']);
            $paymentInquiryPayloadArray = array(
                'payload' => $paymentInquiryPayload
            );
            $paymentInquiryPayload = json_encode($paymentInquiryPayloadArray, JSON_UNESCAPED_SLASHES);
            $paymentInquiryResData = $this->getCurlResponse($paymentInquiryUrl, $paymentInquiryPayload);
            if(isset($paymentInquiryResData['payload']) && !empty($paymentInquiryResData['payload'])) {
                $responsePayloadData = JWT::processJWTPayload($paymentInquiryResData['payload'], $inputParams['payment_params']['secretKey']);
                if(isset($responsePayloadData['cardNo']) && !empty($responsePayloadData['cardNo'])) {
                    $responsePayloadData['cardNo'] = 'XXXXXXXXXXXX' . substr($responsePayloadData['cardNo'], -4);
                }
                $orderData = json_decode($inputParams['order_data']);
                if(isset($responsePayloadData['cardToken']) && !empty($responsePayloadData['cardToken']) && isset($orderData->card_info) && !empty($orderData->card_info)) {
                    $clientToken = (isset($orderData->card_info->client_token) && !empty($orderData->card_info->client_token)) ? $orderData->card_info->client_token : '';
                    // $securePayToken = (isset($orderData->card_info->secure_pay_token) && !empty($orderData->card_info->secure_pay_token)) ? $orderData->card_info->secure_pay_token : '';
                    if($clientToken) {
                        $currentTokenIncId = $this->_pgwVerifyDBO->getDataForCurrentToken($clientToken);
                        $updatedCardToken = $this->_pgwVerifyDBO->updateCardToken($responsePayloadData['cardToken'], $clientToken, json_decode($inputParams['order_data'], true)['user']['user_id']);
                        if($updatedCardToken) {
                            DI::Map('CluesHistory', 'clues\\system\\CluesHistory');
                            $cluesHistoryObj = DI::Singleton('CluesHistory');
                            $cluesHistoryObj->insertIntoHistoryTable(PgwVerifyConstants::USER_CARD_INFO_TABLE, PgwVerifyConstants::USER_CARD_INFO_HISTORY_TABLE, 'id', $currentTokenIncId);
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            $logData = array();
            $logData['field3'] = 'thirdparty:2C2P';
            $logData['field6'] = PgwVerifyConstants::DOMAIN_NAME;
            $logData['error_name'] = $e->getMessage();
            $additionalInfo = array();
            $additionalInfo['file'] = __FILE__;
            $additionalInfo['line'] = __LINE__;
            $additionalInfo['module'] = PgwVerifyConstants::DOMAIN_NAME;
            $additionalInfo['domain'] = PgwVerifyConstants::MODULE_NAME;
            $additionalInfo['level'] = Logger::ERROR;
            $additionalInfo['method'] = __METHOD__;
            $logData['additional_data'] = $additionalInfo;
            $this->LoggerObj->log(PgwVerifyConstants::DOMAIN_NAME, PgwVerifyConstants::MODULE_NAME, $logData, Logger::ERROR);
        }
        return $responsePayloadData;
    }

    /**
     * method is used to check pg response
     * @method verifyPgTransaction
     * @access public
     * @param array $pgResponse pg response
     * @param array $inputParams
     * @return int 1 for verify success 0 for failure
     */
    public function verifyPgTransaction($pgResponse, array $inputParams) {
        if (empty($this->_failedReason) && in_array($pgResponse['respCode'], array('0000')) && $pgResponse['amount'] >= $inputParams['amount']) {
            return 1;
        }
        return 0;
    }

    /**
     * method is used to create dump to insert in prepayment details
     * @method createPrepaymentData
     * @access public
     * @param array $pgResponse response from payment gateway
     * @param array $order
     * @return array prepayment details array
     */
    public function createPrepaymentData($pgResponse, array $order) {
        $dump = array();
        $dump['prepayment_data']['direcpayreferenceid'] = isset($pgResponse['tranRef']) ? $pgResponse['tranRef'] : 0;
        $dump['prepayment_data']['order_id'] = $order['client_order_id'];
        $dump['prepayment_data']['flag'] = isset($pgResponse['respDesc']) ? $pgResponse['respDesc'] : '';
        $dump['prepayment_data']['amount'] = isset($pgResponse['amount']) ? $pgResponse['amount'] : 0;
        $dump['prepayment_data']['payment_gateway'] = !empty($this->paymentGateway) ? $this->paymentGateway : PgwVerifyConstants::Pgw2C2P;
        $dump['prepayment_dump'] = $pgResponse;
        if (!empty($this->_failedReason)) {
            $dump['prepayment_data']['flag'] = $this->_failedReason;
        }
        return $dump;
    }
}
