<?php

namespace clues\model\pgwVerify;

use clues\system\DBOperations;
use clues\model\pgwVerify\PgwVerifyConstants;
use clues\system\CluesConstants;
use Registry;

/**
 * This class is used for all payment verify db CRUD operations
 * @class PgwVerifyDBOperations
 * @category Model datasource
 * @package model/pgwVerify
 * @access public
 * @dependencies DBOperations
 */
class PgwVerifyDBOperations extends DBOperations {

    /**
     * This method is used to check insert data in pup_prepayment_details
     * @method insertCluesPrepaymentDetails
     * @access public
     * @param integer $direcpayreferenceid string direcpayreferenceid 
     * @param $orderId integer order_id 
     * @param $flag string pgw flag
     * @param $dump string pgw response
     * @param $amount integer amount
     * @param $gateway string payment gateway name
     * @param string $flag $flag $gateway gateway
     * @return integer
     */
    public function insertCluesPrepaymentDetails($direcpayreferenceid, $orderId, $flag, $dump, $amount, $gateway) {
        $sql = 'INSERT INTO pup_prepayment_details (direcpayreferenceid, client_order_id, flag, other_details, amount, payment_gateway) VALUES (?s, ?s, ?s, ?s, ?s, ?s)';
        $sql = $this->dbQuote($sql, $direcpayreferenceid, $orderId, $flag, $dump, $amount, $gateway);
        return $this->dbQueryInsert($sql);
    }

    /**
     * This method is used to save post payment response meta data
     * @method savePostPaymentMetaData
     * @access public
     * @param array $metaDataArray post payment meta data
     * @return integer
     */
    public function savePostPaymentMetaData(array $metaDataArray) {
        $query = 'INSERT INTO pup_prepayment_details_data (`prepayment_id`, `key`, `value`, `status`, '
                . '`created`, `updated`, `created_by`, `updated_by`) '
                . 'VALUES (?i, ?s, ?s, ?i, ?i, ?i, ?i, ?i)';
        $query = $this->dbQuote($query, $metaDataArray['prepayment_id'], $metaDataArray['key'], $metaDataArray['value'], $metaDataArray['status'], $metaDataArray['created'], $metaDataArray['updated'], $metaDataArray['created_by'], $metaDataArray['updated_by']);
        return $this->dbQueryInsert($query);
    }

    /**
     * Get Pending orders
     * @method getPendingOrders
     * @access public
     * @param integer $initialTime initial timestamp
     * @param integer $finalTime final timestamp
     * @param integer $paymentId payment id
     * @param integer $orderId order id 
     * @param integer $daysBefore daysBefore
     * @return array
     */
    public function getPendingOrders($initialTime, $finalTime, $paymentId, $orderId = 0, $daysBefore = 7) {
        $queryGetServerType = Registry::get('config.query_move_from_master_to_slave');
        $queryServerType = (!empty($queryGetServerType) ? self::SLAVE_NOCACHE : self::MASTER_NOCACHE);
        $sql = 'SELECT cppm.client_id,cppm.order_pgw_auto_increment_id,cop.merchant_params,cppm.id, '
                . 'cppm.client_order_id, cppm.pgw_order_id, cop.amount, cppm.created, cppm.payment_option_id, cop.payment_gateway, cop.order_data, '
                . 'cppm.payment_id, cppm.emi_id, cppd.value, cppm.retry_count, cppm.prepayment_details_auto_increment_id FROM pup_payment_pgw_mapping cppm '
                . 'INNER JOIN pup_order_pgw cop ON cop.id = cppm.order_pgw_auto_increment_id '
                . 'LEFT JOIN pup_prepayment_details_data cppd ON cppm.prepayment_details_auto_increment_id = cppd.prepayment_id AND cppd.key = ?s '
                . 'LEFT JOIN pup_prepayment_details_data cppd1 ON cppm.prepayment_details_auto_increment_id = cppd1.prepayment_id AND cppd1.key = ?s'
                . ' WHERE cppm.payment_id = ?i AND cppm.status in (?n) AND if(cppm.prepayment_details_auto_increment_id > 0,((cppd1.value = ?i or cppd1.value is null) and cppm.retry_count > ?i AND cppm.retry_count < ?i and if(cppd.value is null,?i,cppd.value) between ?i and ?i),(cppm.created BETWEEN ?i AND ?i)) and cppm.created > ?i and cppm.pgw_order_id is not null';
        $sql = $this->dbQuote($sql, PgwVerifyConstants::PGW_VERIFY_NEXT_PICKUP, PgwVerifyConstants::PAYMENT_STATUS_KEY, $paymentId, array(1, 5), 0, 0, PgwVerifyConstants::MAX_PGW_VERIFY_RETRY, time(), strtotime(" -$daysBefore day"), time(), $initialTime, $finalTime, strtotime("-$daysBefore day"));
        if (!empty($orderId)) {
            $sql .= ' AND cppm.client_order_id = ?s';
            $sql = $this->dbQuote($sql, $orderId);
        }
        $sql .= ' ORDER BY cppm.id DESC, cppm.client_order_id DESC';
        return $this->dbGetArray($sql, array(), $queryServerType);
    }

    /**
     * pic Failed orders
     * @method picFailedOrdersOfPgw
     * @access public
     * @param integer $initialTime initial timestamp
     * @param integer $finalTime final timestamp
     * @param integer $paymentId payment id
     * @param integer $orderId order id 
     * @param array $extra_data extra_data
     * @return array
     */
    public function picFailedOrdersOfPgw($initialTime, $finalTime, $paymentId, $orderId = 0, array $extra_data = array()) {
        $queryGetServerType = Registry::get('config.query_move_from_master_to_slave');
        $queryServerType = (!empty($queryGetServerType) ? self::SLAVE_NOCACHE : self::MASTER_NOCACHE);
        $sql = 'SELECT cppm.client_id,cppm.order_pgw_auto_increment_id,cop.merchant_params,cppm.id, '
                . 'cppm.client_order_id, cppm.pgw_order_id, cop.amount, cppm.created, cppm.payment_option_id, cop.payment_gateway, cop.order_data, '
                . 'cppm.payment_id, cppm.emi_id, cppd.value, cppm.retry_count, cppm.prepayment_details_auto_increment_id FROM pup_payment_pgw_mapping cppm '
                . 'INNER JOIN pup_order_pgw cop ON cop.id = cppm.order_pgw_auto_increment_id '
                . 'LEFT JOIN pup_prepayment_details_data cppd ON cppm.prepayment_details_auto_increment_id = cppd.prepayment_id AND cppd.key = ?s and cppd.value in (?n)'
                . ' WHERE cppm.payment_id = ?i AND cppm.status in (?n) AND cppm.prepayment_details_auto_increment_id > ?i  AND cppm.retry_count = ?i and cppd.value is null';
        $sql = $this->dbQuote($sql, PgwVerifyConstants::PAYMENT_STATUS_KEY, array(1, 2, 3), $paymentId, array(1, 5), 0, 0);
        if (isset($extra_data['payment_option_ids']) && !empty($extra_data['payment_option_ids'])) {
            $sql .= ' AND cppm.payment_option_id not in (?n)';
            $sql = $this->dbQuote($sql, $extra_data['payment_option_ids']);
        }
        if (!empty($orderId)) {
            $sql .= ' AND cppm.client_order_id = ?s';
            $sql = $this->dbQuote($sql, $orderId);
        } else {
            $sql .= ' AND cppm.created BETWEEN ?i AND ?i';
            $sql = $this->dbQuote($sql, $initialTime, $finalTime);
        }
        $sql .= ' ORDER BY cppm.id DESC, cppm.client_order_id DESC';
        return $this->dbGetArray($sql, array(), $queryServerType);
    }

    /**
     * Insert Payment deails by order id in pup_payment_pgw_mapping for Retry Option
     * @method insertPgwOrderMapping
     * @access public
     * @param array $pgwData  
     * @return integer
     */
    public function insertPgwOrderMappingForRetry(array $pgwData) {
        $sql = 'INSERT INTO pup_payment_pgw_mapping (client_order_id, pgw_order_id, order_pgw_auto_increment_id,prepayment_details_auto_increment_id, payment_id, payment_option_id, emi_id, status, created, updated, created_by, updated_by) VALUES (?s, ?s, ?i, ?i, ?i, ?i, ?i, ?i, ?i, ?i, ?i,?i)';
        $query = $this->dbQuote($sql, $pgwData['orderId'], $pgwData['pgwOrderId'], $pgwData['orderPgwPrimaryId'], $pgwData['prepaymentPrimaryId'], $pgwData['paymentId'], $pgwData['paymentOptionId'], $pgwData['emiId'], 1, time(), time(), $pgwData['userId'], $pgwData['userId']);
        return $this->dbQueryInsert($query);
    }

    /**
     * Get Payment Params
     * @method getPaymentParams
     * @access public
     * @param integer $paymentId
     * @return string
     */
    public function getPaymentParams($paymentId) {
        $sql = 'SELECT params FROM pup_payments WHERE payment_id = ?i AND status != ?s';
        $sql = $this->dbQuote($sql, $paymentId, 'R');
        return $this->dbGetField($sql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * method used to get transactions by paymentId
     * @method getTransactionsByOrderId
     * @access public
     * @param integer $orderId
     * @param integer $paymentId
     * @return array
     */
    public function getTransactionsByOrderId($orderId, $paymentId) {
        $sql = 'SELECT cppm.id, cppm.client_order_id, cppm.pgw_order_id, cppm.prepayment_details_auto_increment_id, cpdd.value, cppm.status FROM pup_payment_pgw_mapping cppm '
                . 'INNER JOIN pup_prepayment_details_data cpdd ON cpdd.prepayment_id = cppm.prepayment_details_auto_increment_id '
                . 'WHERE cpdd.key = ?s AND cppm.client_order_id = ?s AND cppm.prepayment_details_auto_increment_id != 0 AND cppm.payment_id = ?i ORDER BY cppm.id DESC';
        $sql = $this->dbQuote($sql, 'payment_status', $orderId, $paymentId);
        return $this->dbGetRow($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * Update Payment details by order id in pup_payment_pgw_mapping for Retry
     * @method updatePgwOrderMappingForRetry
     * @access public
     * @param integer $prepaymentPrimaryId
     * @param integer $scOrderId  
     * @param integer $userId
     * @param integer $paymentId
     * @param integer $id
     * @param integer $status
     * @return integer
     */
    public function updateVerifiedTransactionInMapping($prepaymentPrimaryId, $scOrderId, $userId, $paymentId, $id = 0, $status = 1) {
        $sql = 'UPDATE pup_payment_pgw_mapping SET prepayment_details_auto_increment_id = ?i, updated = ?i, updated_by = ?i, status = ?i WHERE client_order_id = ?s AND status = ?i AND payment_id = ?i';
        $sql = $this->dbQuote($sql, $prepaymentPrimaryId, time(), $userId, $status, $scOrderId, 1, $paymentId);
        if (!empty($id)) {
            $sql .= ' AND id = ?i';
            $sql = $this->dbQuote($sql, $id);
        }
        return $this->dbQueryUpdate($sql);
    }

    /**
     * this method used to update status in pup_payment_pgw_mapping
     * @method updateTransactionStatus
     * @access public
     * @param integer $id
     * @param integer $status
     * @return void 
     */
    public function updateTransactionStatus($id, $status) {
        $sql = 'UPDATE pup_payment_pgw_mapping SET status = ?i,updated=?i WHERE id = ?i';
        $sql = $this->dbQuote($sql, $status, time(), $id);
        $this->dbQueryUpdate($sql);
    }

    /**
     * this method used to get prepayment id from mapping table
     * @method getPrepaymentIdFromMapping
     * @access public
     * @param integer $id
     * @return integer
     */
    public function getPrepaymentIdFromMapping($id) {
        $sql = 'select prepayment_details_auto_increment_id from pup_payment_pgw_mapping where id = ?i';
        $sql = $this->dbQuote($sql, $id);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * used to get payment status from prepayment data
     * @method getPaymentStatusFromPrepaymentData
     * @access public
     * @param integer $prepaymentId
     * @return integer
     */
    public function getPaymentStatusFromPrepaymentData($prepaymentId) {
        $query = 'select value from pup_prepayment_details_data where `key`=?s and prepayment_id=?i';
        $sql = $this->dbQuote($query, 'payment_status', $prepaymentId);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * This method is used to get all active payment options name
     * @method getAllPaymentOptionsName
     * @access public
     * @return array
     */
    public function getAllPaymentOptionsName() {
        $query = "SELECT payment_option_id, payment_option_name AS name FROM pup_payment_options WHERE status != ?s";
        $sql = $this->dbQuote($query, 'R');
        return $this->dbGetHashArray($sql, 'payment_option_id', array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to get all active pgw name
     * @method getPgwName
     * @access public
     * @return
     */
    public function getPgwName() {
        $query = "SELECT payment_id, payment_gateway_name AS name FROM pup_payments WHERE status = 'A'";
        $sql = $this->dbQuote($query);
        return $this->dbGetHashArray($sql, 'payment_id', array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to get client details 
     * @method getPgwVerifyRulesForClientId
     * @access public
     * @param string $clientId
     * @param integer $moduleId
     * @return array hash array
     */
    public function getPgwVerifyRulesForClientId($clientId = CluesConstants::DEFAULT_CLIENT_ID, $moduleId = 0) {
        $clientId = empty($clientId) ? CluesConstants::DEFAULT_CLIENT_ID : $clientId;
        $sql = 'SELECT 
                    r.name, m.rule_status, m.rule_configurations
                FROM
                    pup_client_payment_rules r
                        INNER JOIN
                    pup_client_payment_rules_mapping m ON r.id = m.rule_id AND r.status = m.status
                WHERE
                    r.module_id = ?i AND r.status = ?i
                        AND m.client_id = ?s
                ORDER BY m.priority';
        $query = $this->dbQuote($sql, $moduleId, 1, $clientId);
        return $this->dbGetHashArray($query, 'name', array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to update retry count
     * @method updatePgwVerifyDetails
     * @access public
     * @param integer $id
     * @param integer $count
     * @param integer $userId
     * @param integer $prepaymentDetailsId
     * @return integer
     */
    public function updatePgwVerifyDetails($id, $count, $userId, $prepaymentDetailsId) {
        $sql = 'UPDATE pup_payment_pgw_mapping SET retry_count = ?i, updated = ?i, updated_by = ?i, prepayment_details_auto_increment_id = ?i WHERE id = ?i';
        $sql = $this->dbQuote($sql, $count, time(), $userId, $prepaymentDetailsId, $id);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to check insert data in pup_order_pgw
     * @method insertCluesOrderPgw
     * @access public
     * @param array $inputData
     * @return integer
     */
    public function insertCluesOrderPgw(array $inputData) {
        $sql = 'INSERT INTO pup_order_pgw (client_order_id, amount, payment_gateway, order_data, merchant_params) VALUES (?s, ?s, ?s, ?s, ?s)';
        $sql = $this->dbQuote($sql, $inputData['client_order_id'], $inputData['amount'], $inputData['payment_gateway'], $inputData['order_data'], $inputData['merchant_params']);
        return $this->dbQueryInsert($sql);
    }

    /**
     * Fetch status from table
     * @method fetchPaymentPgwMappingStatus
     * @access public
     * @param integer $id
     * @return integer
     */
    public function fetchPaymentPgwMappingStatus($id) {
        $query = 'SELECT status FROM pup_payment_pgw_mapping where id = ?i';
        $query = $this->dbQuote($query, $id);
        return $this->dbGetField($query, self::MASTER_NOCACHE);
    }

    /**
     * used to get payment status flag from prepayment details
     * @method getPaymentFlagByPrepaymentId
     * @access public
     * @param integer $prepaymentId
     * @return string flag
     */
    public function getPaymentFlagByPrepaymentId($prepaymentId) {
        $query = 'select flag from pup_prepayment_details where id = ?i';
        $sql = $this->dbQuote($query, $prepaymentId);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * get client configurations for given client id
     * @method getClientConfigurations
     * @access public
     * @param string $clientId
     * @return array
     */
    public function getClientConfigurations($clientId) {
        $sql = 'SELECT client_id, client_configurations FROM clues_oauth_clients_details WHERE client_id = ?s AND status = ?i';
        $sanitizedSql = $this->dbQuote($sql, $clientId, 1);
        return $this->dbGetHashArray($sanitizedSql, 'client_id', array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * get payment type id and payment type name from payment option id
     * @method getPaymentTypeDetails
     * @access public
     * @param integer $paymentOptionId
     * @return array
     */
    public function getPaymentTypeDetails($paymentOptionId) {
        $sql = "SELECT 
                    t.payment_type_id,
                    IF(t.payment_type_id > 0,
                        t.`name`,
                        'CB/CB+/GC/Credit') AS pgw_type,
                    IF(o.payment_option_id > 0,
                        o.`name`,
                        'CB/CB+/GC/Credit') AS pgw_option_name
                FROM
                    pup_payment_options o
                        INNER JOIN
                    pup_payment_types t ON t.payment_type_id = o.payment_type_id
                WHERE
                    o.payment_option_id = ?i AND o.status != ?s";
        $sanitizedSql = $this->dbQuote($sql, $paymentOptionId, 'R');
        return $this->dbGetRow($sanitizedSql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * get pgw name from payment id
     * @method getPgwNameFromPaymentId
     * @access public
     * @param integer $paymentId
     * @return array
     */
    public function getPgwNameFromPaymentId($paymentId) {
        $sql = "SELECT 
                    IF(payment_id > 0,
                        payment,
                        'CB/CB+/GC/Credit') AS pgw
                FROM
                    pup_payment_descriptions
                WHERE
                    payment_id = ?i AND cpd_status = ?i";
        $sanitizedSql = $this->dbQuote($sql, $paymentId, 1);
        return $this->dbGetRow($sanitizedSql, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * get object value for a given order id and object key from pup_order_pgw_data
     * @method getPrePgwMetaData
     * @access public
     * @param integer $orderPgwId
     * @param string $objectKey
     * @return string
     */
    public function getPrePgwMetaData($orderPgwId, $objectKey) {
        $sql = 'SELECT object_value FROM pup_order_pgw_data  WHERE order_pgw_id = ?i AND object_key = ?s AND status = ?i';
        $sanitizedSql = $this->dbQuote($sql, $orderPgwId, $objectKey, 1);
        return $this->dbGetField($sanitizedSql, array(), self::SLAVE_NOCACHE);
    }

    /**
     * get object value for a given order id and object key from pup_prepayment_details_data
     * @method getPostPgwMetaData
     * @access public
     * @param integer $prepaymentId
     * @param string $objectKey
     * @return string
     */
    public function getPostPgwMetaData($prepaymentId, $objectKey) {
        $sql = 'SELECT value FROM pup_prepayment_details_data  WHERE prepayment_id = ?i AND `key` = ?s AND status = ?i';
        $sanitizedSql = $this->dbQuote($sql, $prepaymentId, $objectKey, 1);
        return $this->dbGetField($sanitizedSql, array(), self::SLAVE_NOCACHE);
    }

    /**
     * get payment pgw mapping data
     * @method getMappingData
     * @access public
     * @param integer $orderId
     * @param integer $paymentOptionId
     * @param integer $prepaymentId {optional}
     * @return integer
     */
    public function getMappingData($orderId, $paymentOptionId, $prepaymentId = 0) {
        $sql = 'SELECT order_pgw_auto_increment_id FROM pup_payment_pgw_mapping WHERE client_order_id  = ?s AND payment_option_id = ?i ?p LIMIT 1';
        $extraSql = '';
        if (!empty($prepaymentId)) {
            $extraSql = $this->dbQuote(' AND prepayment_details_auto_increment_id = ?i', $prepaymentId);
        } else {
            $extraSql = 'ORDER BY id DESC';
        }
        $sql = $this->dbQuote($sql, $orderId, $paymentOptionId, $extraSql);
        return $this->dbGetField($sql, array(), self::SLAVE_NOCACHE);
    }

    /**
     * add pre pgw metadata
     * @method addPrePgwMetaData
     * @access public
     * @param array $metaDataArray ['order_pgw_id' = <id>, key = <k>, value = <v>, 'status' = <status>, 'created' = <created>, 'updated' = <updated>, 'created_by' = <created_by>, 'updated_by' = <updated_by>]
     * @return integer
     */
    public function addPrePgwMetaData(array $metaDataArray) {
        $sql = 'INSERT INTO pup_order_pgw_data (order_pgw_id, object_key, object_value, status, created, updated, created_by, updated_by) VALUES (?i, ?s, ?s, ?i, ?i, ?i, ?i, ?i)';
        $quote = $this->dbQuote($sql, $metaDataArray['order_pgw_id'], $metaDataArray['key'], $metaDataArray['value'], $metaDataArray['status'], $metaDataArray['created'], $metaDataArray['updated'], $metaDataArray['created_by'], $metaDataArray['updated_by']);
        return $this->dbQueryInsert($quote);
    }

    /**
     * This method is used for checking that the momoe order is already present in pup_payment_pgw_mapping table
     * @method checkMomoeOrderPgwMapping
     * @access public
     * @param string $momoeOrderId
     * @return array
     */
    public function checkMomoeOrderPgwMapping($momoeOrderId, $paymentId) {
        $query = 'SELECT retry_count, prepayment_details_auto_increment_id as prepayment_id from pup_payment_pgw_mapping where pgw_order_id = ?s and status in (?n) and  payment_id = ?i order by created desc limit 1';
        $query = $this->dbQuote($query, $momoeOrderId, array(1, 5), $paymentId);
        $data = $this->dbGetRow($query, array(), self::MASTER_NOCACHE);
        return $data;
    }

    /**
     * This method is used to get last clues order id for momoe_order
     * @method getCluesOrderIdForMomoeOrder
     * @access public
     * @param int $orderId
     * @return int $cluesPgwId
     */
    public function getCluesOrderIdForMomoeOrder($orderId) {
        $query = 'SELECT id FROM pup_order_pgw WHERE order_id = ?i ORDER BY id DESC LIMIT 1';
        $sql = $this->dbQuote($query, $orderId);
        return $this->dbGetField($sql, array(), self::MASTER_NOCACHE);
    }

    /**
     * Update Payment details by order id in pup_payment_pgw_mapping
     * @method updateOrderPgwPrimaryKeyByPgwOrderId
     * @access public
     * @param integer $orderPgwPrimaryId
     * @param integer $pgwOrderId  
     * @param integer $userId
     * @param integer $paymentId payment id
     * @return integer
     */
    public function updateOrderPgwPrimaryKeyByPgwOrderId($orderPgwPrimaryId, $pgwOrderId, $userId, $paymentId, $prepaymentId) {
        $sql = 'UPDATE pup_payment_pgw_mapping SET order_pgw_auto_increment_id = ?i, prepayment_details_auto_increment_id = ?i, updated = ?i, updated_by = ?i WHERE pgw_order_id = ?s AND payment_id = ?i';
        $sql = $this->dbQuote($sql, $orderPgwPrimaryId, $prepaymentId, time(), $userId, $pgwOrderId, $paymentId);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * update verify retry count
     * @method updateVerifyRetryCount
     * @access public
     * @param integer $momoeOrderId
     * @param integer $prepaymentId
     * @param integer $retry_count
     */
    public function updateVerifyRetryCount($momoeOrderId, $prepaymentId, $retry_count) {
        $query = 'Update pup_payment_pgw_mapping set retry_count = ?i, updated = ?i, updated_by = ?i where pgw_order_id = ?s and prepayment_details_auto_increment_id = ?i limit ?i';
        $updateQuery = $this->dbQuote($query, $retry_count, time(), PgwVerifyConstants::ADMIN_USER_ID, $momoeOrderId, $prepaymentId, 1);
        return $this->dbQueryUpdate($updateQuery);
    }

    /**
     * This method is used that the prepayment id is already present in table pup_prepayment_details_data
     * @method checkPrepaymentIdMetaData
     * @access public
     * @param integer $prepaymentId prepayment id
     * @param string $checkKey checking key
     * @return boolean trur or false
     */
    public function checkPrepaymentIdMetaData($prepaymentId, $checkKey, $checkValue = null) {
        $returnVal = false;
        $query = "SELECT count(status) from pup_prepayment_details_data where prepayment_id = ?i and `key`= ?s and status = ?i";
        $query = $this->dbQuote($query, $prepaymentId, $checkKey, 1);
        if (!is_null($checkValue)) {
            $query .= $this->dbQuote(' and `value` = ?s', $checkValue);
        }
        $data = $this->dbGetField($query, array(), self::MASTER_NOCACHE);
        if (!empty($data)) {
            $returnVal = true;
        }
        return $returnVal;
    }

    /**
     * This method is used to update Post payment metadata
     * @method updatePostPaymentMetaData
     * @access public
     * @param int $value
     * @return void
     */
    public function updatePostPaymentMetaData($value, $userId, $prepaymentPrimaryId, $key) {
        $sql = 'UPDATE pup_prepayment_details_data SET `value` = ?s, updated = ?i, updated_by = ?i, status = ?i WHERE prepayment_id = ?i and `key` = ?s';
        $sql = $this->dbQuote($sql, $value, time(), $userId, 1, $prepaymentPrimaryId, $key);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to get app_client_id details
     * @method getAppClientIdDetails
     * @param string $clientId client id
     * @access public
     * @return array
     */
    public function getAppClientIdDetails($clientId) {
        $sql = 'select c.*,cd.* from oauth_clients as c left join clues_oauth_clients_details as cd'
                . ' on c.client_id=cd.client_id and cd.status=1 where c.status=1 and c.client_id=?s limit 1';
        $query = $this->dbQuote($sql, $clientId);
        return $result = $this->dbGetRow($query, array(), DBOperations::SLAVE_CACHE);
    }

    /**
     * This method is used to check insert data in pup_pgw_notify_mapping
     * @method insertPupNotifyMapping
     * @access public
     * @param array $orderDetails
     * @param string $formDataJson
     * @param integer $userId
     * @return integer
     */
    public function insertPupNotifyMapping($orderDetails, $formDataJson, $userId, $source) {
        $sql = 'INSERT INTO pup_pgw_notify_mapping (client_order_id, ppm_id, source, notify_data, status, created, created_by, updated, updated_by) VALUES (?s, ?i, ?i, ?s, ?i, ?s, ?i, ?s, ?i)';
        $sql = $this->dbQuote($sql, $orderDetails['client_order_id'], $orderDetails['id'], $source, $formDataJson, 1, date('Y:m:d H:i:s'), $userId, date('Y:m:d H:i:s'), $userId);
        return $this->dbQueryInsert($sql);
    }

    /**
     * This method is used to update card token
     * @method updateCardToken
     * @param string $cardToken
     * @param string $clientToken
     * @param string $userId
     * @return array
     */
    public function updateCardToken($cardToken, $clientToken, $userId) {
        $sql = 'UPDATE pup_user_card_info SET card_token = ?s, updated_by = ?i WHERE status = ?i AND pup_tk = ?s';
        $sql = $this->dbQuote($sql, $cardToken, $userId, 1, $clientToken);
        return $this->dbQueryUpdate($sql);
    }

    /**
     * This method is used to get card info from token
     * @method getDataForCurrentToken
     * @param string $pupToken
     * @return array
     */
    public function getDataForCurrentToken($pupToken) {
        $sql = 'select id from pup_user_card_info where pup_tk = ?s and status = ?i';
        $query = $this->dbQuote($sql, $pupToken, 1);
        return $this->dbGetRow($query, self::MASTER_NOCACHE);
    }

}
